% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/means.R
\name{nested_mean}
\alias{nested_mean}
\alias{fisher_mean}
\title{Nested generalized mean}
\usage{
nested_mean(r1, r2, t = c(1, 1))

fisher_mean(x, w1 = NULL, w2 = NULL, na.rm = FALSE)
}
\arguments{
\item{r1}{A finite number giving the order of the outer generalized mean.}

\item{r2}{A pair of finite numbers giving the order of the inner generalized
means.}

\item{t}{A pair of strictly positive weights for the inner generalized
means. The default is equal weights.}

\item{x}{A strictly positive numeric vector.}

\item{w1, w2}{A strictly positive numeric vector of weights, the same length
as \code{x}. The default is to equally weight each element of \code{x}.}

\item{na.rm}{Should missing values in \code{x}, \code{w1}, and \code{w2} be
removed? By default missing values in \code{x}, \code{w1}, or \code{w2}
return a missing value.}
}
\value{
\code{nested_mean()} returns a function:

\preformatted{function(x, w1 = NULL, w2 = NULL, na.rm = FALSE){...}}

This computes the generalized mean of order \code{r1} of the generalized
mean of order \code{r2[1]} of \code{x} with weights \code{w1} and the
generalized mean of order \code{r2[2]} of \code{x} with weights \code{w2}.

\code{fisher_mean()} returns a numeric value for the geometric mean of the
arithmetic and harmonic means (i.e., \code{r1 = 0} and \code{r2 = c(1, -1)}).
}
\description{
Calculate the (outer) generalized mean of two (inner) generalized means
(i.e., crossing generalized means).
}
\note{
There is some ambiguity about how to remove missing values in
\code{w1} or \code{w2} when \code{na.rm = TRUE}. The approach here is to
remove missing values when calculating each of the inner means individually,
rather than removing all missing values prior to any calculations. This
means that a different number of data points could be used to calculate the
inner means. Use the \code{\link[=balanced]{balanced()}} operator to balance
missing values across \code{w1} and w2 prior to any calculations.
}
\examples{
x <- 1:3
w1 <- 4:6
w2 <- 7:9

#---- Making superlative indexes ----

# A function to make the superlative quadratic mean price index by
# Diewert (1976) as a product of generalized means

quadratic_mean_index <- function(x, w0, w1, r) {
  x <- sqrt(x)
  generalized_mean(r)(x, w0) * generalized_mean(-r)(x, w1)
}

quadratic_mean_index(x, w1, w2, 2)

# Same as the nested generalized mean (with the order halved)

quadratic_mean_index2 <- function(r) nested_mean(0, c(r / 2, -r / 2))

quadratic_mean_index2(2)(x, w1, w2)

# The arithmetic AG mean index by Lent and Dorfman (2009)

agmean_index <- function(tau) nested_mean(1, c(0, 1), c(tau, 1 - tau))

agmean_index(0.25)(x, w1, w1)

#---- Walsh index ----

# The (arithmetic) Walsh index is the implicit price index when using a
# superlative quadratic mean quantity index of order 1

p1 <- price6[[2]]
p0 <- price6[[1]]
q1 <- quantity6[[2]]
q0 <- quantity6[[1]]

walsh <- quadratic_mean_index2(1)

sum(p1 * q1) / sum(p0 * q0) / walsh(q1 / q0, p0 * q0, p1 * q1)

sum(p1 * sqrt(q1 * q0)) / sum(p0 * sqrt(q1 * q0))

# Counter to the PPI manual (par. 1.105), it is not a superlative
# quadratic mean price index of order 1

walsh(p1 / p0, p0 * q0, p1 * q1)

#---- Missing values ----

x[1] <- NA
w1[2] <- NA

fisher_mean(x, w1, w2, na.rm = TRUE)

# Same as using obs 2 and 3 in an arithmetic mean, and obs 3 in a
# harmonic mean

geometric_mean(c(
  arithmetic_mean(x, w1, na.rm = TRUE),
  harmonic_mean(x, w2, na.rm = TRUE)
))

# Use balanced() to use only obs 3 in both inner means

balanced(fisher_mean)(x, w1, w2, na.rm = TRUE)

}
\references{
Diewert, W. E. (1976). Exact and superlative index numbers.
\emph{Journal of Econometrics}, 4(2): 114--145.

ILO, IMF, OECD, Eurostat, UN, and World Bank. (2004).
\emph{Producer Price Index Manual: Theory and Practice}. International Monetary
Fund.

Lent, J. and Dorfman, A. H. (2009). Using a weighted average of base period
price indexes to approximate a superlative index.
\emph{Journal of Official Statistics}, 25(1):139--149.
}
\seealso{
\code{\link[=nested_contributions]{nested_contributions()}} for percent-change contributions for
indexes based on nested generalized means, like the Fisher index.

Other means: 
\code{\link{extended_mean}()},
\code{\link{generalized_mean}()},
\code{\link{lehmer_mean}()}
}
\concept{means}
