% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rmc.R
\name{rmc}
\alias{rmc}
\title{Regression for Multiple Comparison}
\usage{
rmc(data, distribution = c("dnorm", "dfnorm", "dlnorm"), level = 0.95,
  outplot = c("mcb", "lines"), select = NULL, plot = TRUE, ...)
}
\arguments{
\item{data}{Matrix or data frame with observations in rows and variables in
columns.}

\item{distribution}{Type of the distribution to use. If this is a clear forecast error,
then \code{"dnorm"} is appropriate, leading to a simple Gausian linear
regression. \code{"dfnorm"} would lead to a alm model with folded normal
distribution. Finally, \code{"dlnorm"} would lead to the alm with log normal
distribution. This value is passed to \code{alm()} function. You can try some
other distributions, but don't expect anything meaningful.}

\item{level}{The width of the confidence interval. Default is 0.95.}

\item{outplot}{What outplot of plot to use after the calculations. This can be
either "MCB" (\code{"mcb"}) outplot or "Vertical lines" (\code{"lines"}).}

\item{select}{What column of data to highlight on the plot. If NULL, then
the method with the lowest value is selected.}

\item{plot}{If \code{TRUE} then the graph is produced after the calculations.
You can also use plot method on the produced object in order to get the same
effect.}

\item{...}{Other parameters passed to plot function}
}
\value{
If \code{plot=TRUE}, then the function plots the results after all
the calculations. In case of \code{distribution="dnorm"}, the closer to zero the
intervals are, the better model performs. When \code{distribution="dfnorm"} or
\code{distribution="dlnorm"}, the lower, the better.

Function returns a list of a class "rmc", which contains the following
variables:
\itemize{
\item{mean}{Mean values for each method.}
\item{interval}{Confidence intervals for each method.}
\item{vlines}{Coordinates used for outplot="l", marking the groups of methods.}
\item{groups}{The table containing the groups. \code{TRUE} - methods are in the
same group, \code{FALSE} - they are not.}
\item{p.value}{p-value for the test of the significance of the model. This is a
log-likelihood ratios chi-squared test, comparing the model with the one with
intercept only.}
\item{importance}{The weights of the estimated model in comparison with the
model with the constant only. 0 means that the constant is better, 1 means that
the estimated model is the best.}
\item{level}{Significance level.}
\item{model}{lm model produced for the calculation of the intervals.}
\item{outplot}{Style of the plot to produce.}
\item{select}{The selected variable to highlight.}
}
}
\description{
RMC stands for "Regression for Methods Comparison". This is a parametric
test for the comparison of means of several distributions
This test is a parametric counterpart of Nemenyi / MCB test (Demsar, 2006) and
uses asymptotic properties of regression models. It relies on distributional
assumptions about the provided data. For instance, if the mean forecast errors
are used, then it is safe to assume that the regression model constructed on
them will have symmetrically distributed residuals, thus normal regression can
be used for the parameters estimation.
}
\details{
The test constructs the regression model of the kind:

y = b' X + e,

where y is the vector of the provided data (as.vector(data)), X is the matrix
of dummy variables for each column of the data (forecasting method), b is the
vector of coefficients for the dummies and e is the error term of the model.

Depending on the provided data, it might make sense to use different types of
regressions. The function supports Gausian linear regression
(\code{distribution="dnorm"}, when the data is normal), advanced linear regression with
folded normal distribution (\code{distribution="dfnorm"}, for example, absolute errors,
assuming that the original errors are normally distributed) and advanced linear
regression with log normal distribution (\code{distribution="dlnorm"}, which might be
useful in case of relative error measures.

The advisable error measures to use in the test are RelMAE and RelMSE together with
\code{distribution="dlnorm"}. They are unbiased and their logarithms are
symmetrically distributed (Davydenko & Fildes, 2013). Although their distributions
are not normal, given the typically large samples of datasets, the Central Limit
Theorem helps in the adequate construction of the confidence intervals for the
parameters.

The test is equivalent to Nemenyi test, when applied to the ranks of the error
measures on large samples.

There is also a \code{plot()} method that allows producing either "mcb" or "lines"
style of plot. This can be regulated via \code{plot(x, outplot="lines")}.
}
\examples{
N <- 50
M <- 4
ourData <- matrix(rnorm(N*M,mean=0,sd=1), N, M)
ourData[,2] <- ourData[,2]+4
ourData[,3] <- ourData[,3]+3
ourData[,4] <- ourData[,4]+2
colnames(ourData) <- c("Method A","Method B","Method C - long name","Method D")
rmc(ourData, distribution="dnorm", level=0.95)
# In case of AE-based measures, distribution="dfnorm" should be selected
rmc(abs(ourData), distribution="dfnorm", level=0.95)

# APE-based measures should not be used in general...

# If RelMAE or RelMSE is used for measuring data, then it makes sense to use
# distribution="dlnorm" for the RelMAE / RelMSE, as it can be approximated by
# log normal distribution, because according to Davydenko & Fildes (2013) the
# logarithms of these measures have symmetric distribution.
ourTest <- rmc((abs(ourData) / rfnorm(N, 0.3, 1)), distribution="dlnorm", level=0.95)
# The exponents of mean values from this function will correspond to the
# geometric means of RelMAE / RelMSE.
exp(ourTest$mean)
# The same is for the intervals:
exp(ourTest$interval)

# You can also reproduce plots in different styles:
plot(ourTest, outplot="lines")

# Or you can use the default "mcb" style and set additional parameters for the plot():
par(mar=c(2,2,4,0)+0.1)
plot(ourTest, main="Four methods")

# The following example should give similar results to Nemenyi test on
# large samples, which compares medians of the distributions:
rmc(t(apply(ourData,1,rank)), distribution="dnorm", level=0.95)

# You can also give a try to SE-based measures with distribution="dchisq":
rmc(ourData^2, distribution="dchisq", level=0.95)

}
\references{
\itemize{
\item  Demsar, J. (2006). Statistical Comparisons of Classifiers over
Multiple Data Sets. Journal of Machine Learning Research, 7, 1-30.
\url{http://www.jmlr.org/papers/volume7/demsar06a/demsar06a.pdf}
\item Davydenko, A., Fildes, R. (2013). Measuring Forecasting Accuracy:
The Case Of Judgmental Adjustments To Sku-Level Demand Forecasts.
International Journal of Forecasting, 29(3), 510-522.
\url{https://doi.org/10.1016/j.ijforecast.2012.09.002}
\item Hea-Jung Kim (2006) On the Ratio of Two Folded Normal
Distributions, Communications in Statistics Theory and Methods, 35:6,
965-977, \url{https://doi.org/10.1080/03610920600672229}
}
}
\seealso{
\code{\link[greybox]{alm}}
}
\author{
Ivan Svetunkov, \email{ivan@svetunkov.ru}
}
\keyword{htest}
