% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/substitution.R
\name{sub_small_vals}
\alias{sub_small_vals}
\title{Substitute small values in the table body}
\usage{
sub_small_vals(
  data,
  columns = everything(),
  rows = everything(),
  threshold = 0.01,
  small_pattern = if (sign == "+") "<{x}" else md("<*abs*(-{x})"),
  sign = "+"
)
}
\arguments{
\item{data}{\emph{The gt table data object}

\verb{obj:<gt_tbl>} // \strong{required}

This is the \strong{gt} table object that is commonly created through use of the
\code{\link[=gt]{gt()}} function.}

\item{columns}{\emph{Columns to target}

\verb{<column-targeting expression>} // \emph{default:} \code{everything()}

The columns to which substitution operations are constrained. Can either
be a series of column names provided in \code{\link[=c]{c()}}, a vector of column indices,
or a select helper function. Examples of select helper functions include
\code{\link[=starts_with]{starts_with()}}, \code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}}, \code{\link[=matches]{matches()}}, \code{\link[=one_of]{one_of()}},
\code{\link[=num_range]{num_range()}}, and \code{\link[=everything]{everything()}}.}

\item{rows}{\emph{Rows to target}

\verb{<row-targeting expression>} // \emph{default:} \code{everything()}

In conjunction with \code{columns}, we can specify which of their rows should
form a constraint for targeting operations. The default \code{\link[=everything]{everything()}}
results in all rows in \code{columns} being formatted. Alternatively, we can
supply a vector of row IDs within \code{\link[=c]{c()}}, a vector of row indices, or a
select helper function. Examples of select helper functions include
\code{\link[=starts_with]{starts_with()}}, \code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}}, \code{\link[=matches]{matches()}}, \code{\link[=one_of]{one_of()}},
\code{\link[=num_range]{num_range()}}, and \code{\link[=everything]{everything()}}. We can also use expressions to filter
down to the rows we need (e.g., \verb{[colname_1] > 100 & [colname_2] < 50}).}

\item{threshold}{\emph{Threshold value}

\verb{scalar<numeric|integer>} // \emph{default:} \code{0.01}

The threshold value with which values should be considered small enough for
replacement.}

\item{small_pattern}{\emph{Pattern specification for small values}

\verb{scalar<character>} // \emph{default:} \code{if (sign == "+") "<{x}" else md("<*abs*(-{x})")}

The pattern text to be used in place of the suitably small values in the
rendered table.}

\item{sign}{\emph{Consider positive or negative values?}

\verb{scalar<character>} // \emph{default:} \code{"+"}

The sign of the numbers to be considered in the replacement. By default, we
only consider positive values (\code{"+"}). The other option (\code{"-"}) can be used
to consider only negative values.}
}
\value{
An object of class \code{gt_tbl}.
}
\description{
Wherever there is numerical data that are very small in value, replacement
text may be better for explanatory purposes. The \code{sub_small_vals()} function
allows for this replacement through specification of a \code{threshold}, a
\code{small_pattern}, and the sign of the values to be considered. The
substitution will occur for those values found to be between \code{0} and the
threshold value. This is possible for small positive and small negative
values (this can be explicitly set by the \code{sign} option). Note that the
interval does not include the \code{0} or the \code{threshold} value. Should you need
to include zero values, use the \code{\link[=sub_zero]{sub_zero()}} function.
}
\section{Targeting cells with \code{columns} and \code{rows}}{


Targeting of values is done through \code{columns} and additionally by \code{rows} (if
nothing is provided for \code{rows} then entire columns are selected). The
\code{columns} argument allows us to target a subset of cells contained in the
resolved columns. We say resolved because aside from declaring column names
in \code{c()} (with bare column names or names in quotes) we can use
\strong{tidyselect}-style expressions. This can be as basic as supplying a select
helper like \code{starts_with()}, or, providing a more complex incantation like

\code{where(~ is.numeric(.x) && max(.x, na.rm = TRUE) > 1E6)}

which targets numeric columns that have a maximum value greater than
1,000,000 (excluding any \code{NA}s from consideration).

By default all columns and rows are selected (with the \code{everything()}
defaults). Cell values that are incompatible with a given substitution
function will be skipped over. So it's safe to select all columns with a
particular substitution function (only those values that can be substituted
will be), but, you may not want that. One strategy is to work on the bulk of
cell values with one substitution function and then constrain the columns for
later passes with other types of substitution (the last operation done to a
cell is what you get in the final output).

Once the columns are targeted, we may also target the \code{rows} within those
columns. This can be done in a variety of ways. If a stub is present, then we
potentially have row identifiers. Those can be used much like column names in
the \code{columns}-targeting scenario. We can use simpler \strong{tidyselect}-style
expressions (the select helpers should work well here) and we can use quoted
row identifiers in \code{c()}. It's also possible to use row indices (e.g., \code{c(3, 5, 6)}) though these index values must correspond to the row numbers of the
input data (the indices won't necessarily match those of rearranged rows if
row groups are present). One more type of expression is possible, an
expression that takes column values (can involve any of the available columns
in the table) and returns a logical vector. This is nice if you want to base
the substitution on values in the column or another column, or, you'd like to
use a more complex predicate expression.
}

\section{Examples}{


Let's generate a simple, single-column tibble that contains an assortment of
values that could potentially undergo some substitution.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{tbl <- dplyr::tibble(num = c(10^(-4:2), 0, NA))

tbl
#> # A tibble: 9 x 1
#>        num
#>      <dbl>
#> 1   0.0001
#> 2   0.001 
#> 3   0.01  
#> 4   0.1   
#> 5   1     
#> 6  10     
#> 7 100     
#> 8   0     
#> 9  NA
}\if{html}{\out{</div>}}

The \code{tbl} contains a variety of smaller numbers and some might be small
enough to reformat with a threshold value. With \code{sub_small_vals()} we can
do just that:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{tbl |>
  gt() |>
  fmt_number(columns = num) |>
  sub_small_vals()
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_sub_small_vals_1.png" alt="This image of a table was generated from the first code example in the `sub_small_vals()` help file." style="width:100\%;">
}}

Small and negative values can also be handled but they are handled specially
by the \code{sign} parameter. Setting that to \code{"-"} will format only the small,
negative values.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{tbl |>
  dplyr::mutate(num = -num) |>
  gt() |>
  fmt_number(columns = num) |>
  sub_small_vals(sign = "-")
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_sub_small_vals_2.png" alt="This image of a table was generated from the second code example in the `sub_small_vals()` help file." style="width:100\%;">
}}

You don't have to settle with the default \code{threshold} value or the default
replacement pattern (in \code{small_pattern}). This can be changed and the
\code{"{x}"} in \code{small_pattern} (which uses the \code{threshold} value) can even be
omitted.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{tbl |>
  gt() |>
  fmt_number(columns = num) |>
  sub_small_vals(
    threshold = 0.0005,
    small_pattern = "smol"
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_sub_small_vals_3.png" alt="This image of a table was generated from the third code example in the `sub_small_vals()` help file." style="width:100\%;">
}}
}

\section{Function ID}{

3-29
}

\section{Function Introduced}{

\code{v0.6.0} (May 24, 2022)
}

\seealso{
Other data formatting functions: 
\code{\link{data_color}()},
\code{\link{fmt_auto}()},
\code{\link{fmt_bins}()},
\code{\link{fmt_bytes}()},
\code{\link{fmt_currency}()},
\code{\link{fmt_datetime}()},
\code{\link{fmt_date}()},
\code{\link{fmt_duration}()},
\code{\link{fmt_engineering}()},
\code{\link{fmt_flag}()},
\code{\link{fmt_fraction}()},
\code{\link{fmt_icon}()},
\code{\link{fmt_image}()},
\code{\link{fmt_index}()},
\code{\link{fmt_integer}()},
\code{\link{fmt_markdown}()},
\code{\link{fmt_number}()},
\code{\link{fmt_partsper}()},
\code{\link{fmt_passthrough}()},
\code{\link{fmt_percent}()},
\code{\link{fmt_roman}()},
\code{\link{fmt_scientific}()},
\code{\link{fmt_spelled_num}()},
\code{\link{fmt_time}()},
\code{\link{fmt_units}()},
\code{\link{fmt_url}()},
\code{\link{fmt}()},
\code{\link{sub_large_vals}()},
\code{\link{sub_missing}()},
\code{\link{sub_values}()},
\code{\link{sub_zero}()}
}
\concept{data formatting functions}
