% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/HS_normal_means.R
\name{HS.normal.means}
\alias{HS.normal.means}
\title{The horseshoe prior for the sparse normal means problem}
\usage{
HS.normal.means(y, method.tau = c("fixed", "truncatedCauchy", "halfCauchy"),
  tau = 1, method.sigma = c("fixed", "Jeffreys"), Sigma2 = 1,
  burn = 1000, nmc = 5000, alpha = 0.05)
}
\arguments{
\item{y}{The data. A \eqn{n*1} vector.}

\item{method.tau}{Method for handling \eqn{\tau}. Select "fixed" to plug in an
estimate of tau (empirical Bayes), "truncatedCauchy" for the half-
Cauchy prior truncated to [1/n, 1], or "halfCauchy" for a
non-truncated half-Cauchy prior. The truncated Cauchy prior is recommended over
the non-truncated version.}

\item{tau}{Use this argument to pass the (estimated) value of \eqn{\tau} in case "fixed"
is selected for method.tau. Not necessary when method.tau is equal to"halfCauchy" or
"truncatedCauchy". The function \code{\link{HS.MMLE}} can be used to compute an
estimate of tau. The default (tau = 1) is not suitable for most purposes and should
be replaced.}

\item{method.sigma}{Select "fixed" for a fixed error variance, or "Jeffreys"
to use Jeffrey's prior.}

\item{Sigma2}{The variance of the data - only necessary when "fixed" is
selected for method.sigma. The default (Sigma2 = 1) is not suitable for
most purposes and should be replaced.}

\item{burn}{Number of samples used for burn-in. Default is 1000.}

\item{nmc}{Number of MCMC samples taken after burn-in. Default is 5000.}

\item{alpha}{The level for the credible intervals. E.g. alpha = 0.05 yields
95\% credible intervals}
}
\value{
\item{BetaHat}{The posterior mean (horseshoe estimator) for each of the datapoints.}
\item{LeftCI}{The left bounds of the credible intervals.}
\item{RightCI}{The right bounds of the credible intervals.}
\item{BetaMedian}{Posterior median of Beta, a \eqn{n} by 1 vector.}
\item{Sigma2Hat}{Posterior mean of error variance \eqn{\sigma^2}. If method.sigma =
"fixed" is used, this value will be equal to the user-selected value of Sigma2
passed to the function.}
\item{TauHat}{Posterior mean of global scale parameter tau, a positive scalar.
If method.tau = "fixed" is used, this value will be equal to the user-selected value
of tau passed to the function.}
\item{BetaSamples}{Posterior samples of Beta.}
\item{TauSamples}{Posterior samples of tau.}
\item{Sigma2Samples}{Posterior samples of Sigma2.}
}
\description{
Apply the horseshoe prior to the normal means problem
 (i.e. linear regression with the design matrix equal to the identity matrix).
 Computes the posterior mean, median and credible intervals. There are options for
 empirical Bayes (estimate of tau and or Sigma2 plugged in) and full Bayes (truncated
 or non-truncated half-Cauchy on tau, Jeffrey's prior on Sigma2). For the full Bayes
 version, the truncated half-Cauchy prior is recommended by Van der Pas et al. (2016).
}
\details{
The normal means model is:
\deqn{y_i=\beta_i+\epsilon_i, \epsilon_i \sim N(0,\sigma^2)}

And the horseshoe prior:
\deqn{\beta_j \sim N(0,\sigma^2 \lambda_j^2 \tau^2)}
\deqn{\lambda_j \sim Half-Cauchy(0,1).}

Estimates of \eqn{\tau} and \eqn{\sigma^2} may be plugged in (empirical Bayes), or those
parameters are equipped with hyperpriors (full Bayes).
}
\examples{
#Empirical Bayes example with 20 signals, rest is noise
#Posterior mean for the signals is plotted
#And variable selection is performed using the credible intervals
#And the credible intervals are plotted
truth <- c(rep(0, 80), rep(8, 20))
data <-  truth + rnorm(100, 1)
tau.hat <- HS.MMLE(data, Sigma2 = 1)
res.HS1 <- HS.normal.means(data, method.tau = "fixed", tau = tau.hat,
method.sigma = "fixed", Sigma2 = 1)
#Plot the posterior mean against the data (signals in blue)
plot(data, res.HS1$BetaHat, col = c(rep("black", 80), rep("blue", 20)))
#Find the selected betas (ideally, the last 20 are equal to 1)
HS.var.select(res.HS1, data, method = "intervals")
#Plot the credible intervals
library(Hmisc)
xYplot(Cbind(res.HS1$BetaHat, res.HS1$LeftCI, res.HS1$RightCI) ~ 1:100)


#Full Bayes example with 20 signals, rest is noise
#Posterior mean for the signals is plotted
#And variable selection is performed using the credible intervals
#And the credible intervals are plotted
truth <- c(rep(0, 80), rep(8, 20))
data <-  truth + rnorm(100, 3)
res.HS2 <- HS.normal.means(data, method.tau = "truncatedCauchy", method.sigma = "Jeffreys")
#Plot the posterior mean against the data (signals in blue)
plot(data, res.HS2$BetaHat, col = c(rep("black", 80), rep("blue", 20)))
#Find the selected betas (ideally, the last 20 are equal to 1)
HS.var.select(res.HS2, data, method = "intervals")
#Plot the credible intervals
library(Hmisc)
xYplot(Cbind(res.HS2$BetaHat, res.HS2$LeftCI, res.HS2$RightCI) ~ 1:100)
}
\references{
van der Pas, S. L., Szabo, B., and an der Vaart, A. W. (2016), How many needles
  in the haystack? Adaptive inference and uncertainty quantification for
  the horseshoe. arXiv:1607.01892
}
\seealso{
\code{\link{HS.post.mean}} for a fast way to compute the posterior mean
if an estimate of tau is available. \code{\link{horseshoe}} for linear regression.
\code{\link{HS.var.select}} to perform variable selection.
}

