% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/perm_importance.R
\name{perm_importance}
\alias{perm_importance}
\alias{perm_importance.default}
\alias{perm_importance.ranger}
\alias{perm_importance.Learner}
\alias{perm_importance.explainer}
\title{Permutation Importance}
\usage{
perm_importance(object, ...)

\method{perm_importance}{default}(
  object,
  X,
  y,
  v = NULL,
  pred_fun = stats::predict,
  loss = "squared_error",
  m_rep = 4L,
  agg_cols = FALSE,
  normalize = FALSE,
  n_max = 10000L,
  w = NULL,
  verbose = TRUE,
  ...
)

\method{perm_importance}{ranger}(
  object,
  X,
  y,
  v = NULL,
  pred_fun = function(m, X, ...) stats::predict(m, X, ...)$predictions,
  loss = "squared_error",
  m_rep = 4L,
  agg_cols = FALSE,
  normalize = FALSE,
  n_max = 10000L,
  w = NULL,
  verbose = TRUE,
  ...
)

\method{perm_importance}{Learner}(
  object,
  X,
  y,
  v = NULL,
  pred_fun = NULL,
  loss = "squared_error",
  m_rep = 4L,
  agg_cols = FALSE,
  normalize = FALSE,
  n_max = 10000L,
  w = NULL,
  verbose = TRUE,
  ...
)

\method{perm_importance}{explainer}(
  object,
  X = object[["data"]],
  y = object[["y"]],
  v = NULL,
  pred_fun = object[["predict_function"]],
  loss = "squared_error",
  m_rep = 4L,
  agg_cols = FALSE,
  normalize = FALSE,
  n_max = 10000L,
  w = object[["weights"]],
  verbose = TRUE,
  ...
)
}
\arguments{
\item{object}{Fitted model object.}

\item{...}{Additional arguments passed to \code{pred_fun(object, X, ...)},
for instance \code{type = "response"} in a \code{\link[=glm]{glm()}} model, or \code{reshape = TRUE} in a
multiclass XGBoost model.}

\item{X}{A data.frame or matrix serving as background dataset.}

\item{y}{Vector/matrix of the response, or the corresponding column names in \code{X}.}

\item{v}{Vector of feature names, or named list of feature groups.
The default (\code{NULL}) will use all column names of \code{X} with the following exception:
If \code{y} or \code{w} are passed  as column names, they are dropped.}

\item{pred_fun}{Prediction function of the form \verb{function(object, X, ...)},
providing \eqn{K \ge 1} predictions per row. Its first argument represents the
model \code{object}, its second argument a data structure like \code{X}. Additional arguments
(such as \code{type = "response"} in a GLM, or \code{reshape = TRUE} in a multiclass XGBoost
model) can be passed via \code{...}. The default, \code{\link[stats:predict]{stats::predict()}}, will work in
most cases.}

\item{loss}{One of "squared_error", "logloss", "mlogloss", "poisson",
"gamma", "absolute_error", "classification_error". Alternatively, a loss function
can be provided that turns observed and predicted values into a numeric vector or
matrix of unit losses of the same length as \code{X}.
For "mlogloss", the response \code{y} can either be a dummy matrix or a discrete vector.
The latter case is handled via a fast version of \code{model.matrix(~ as.factor(y) + 0)}.
For "classification_error", both predictions and responses can be non-numeric.
For "squared_error", both predictions and responses can be factors with identical
levels. In this case, squared error is evaulated for each one-hot-encoded column.}

\item{m_rep}{Number of permutations (default 4).}

\item{agg_cols}{Should multivariate losses be summed up? Default is \code{FALSE}.
In combination with the squared error loss, \code{agg_cols = TRUE} gives
the Brier score for (probabilistic) classification.}

\item{normalize}{Should importance statistics be divided by average loss?
Default is \code{FALSE}. If \code{TRUE}, an importance of 1 means that the average loss
has been doubled by shuffling that feature's column.}

\item{n_max}{If \code{X} has more than \code{n_max} rows, a random sample of \code{n_max} rows is
selected from \code{X}. In this case, set a random seed for reproducibility.}

\item{w}{Optional vector of case weights. Can also be a column name of \code{X}.}

\item{verbose}{Should a progress bar be shown? The default is \code{TRUE}.}
}
\value{
An object of class "hstats_matrix" containing these elements:
\itemize{
\item \code{M}: Matrix of statistics (one column per prediction dimension), or \code{NULL}.
\item \code{SE}: Matrix with standard errors of \code{M}, or \code{NULL}.
Multiply with \code{sqrt(m_rep)} to get \emph{standard deviations} instead.
Currently, supported only for \code{\link[=perm_importance]{perm_importance()}}.
\item \code{m_rep}: The number of repetitions behind standard errors \code{SE}, or \code{NULL}.
Currently, supported only for \code{\link[=perm_importance]{perm_importance()}}.
\item \code{statistic}: Name of the function that generated the statistic.
\item \code{description}: Description of the statistic.
}
}
\description{
Calculates permutation importance for a set of features or a set of feature groups.
By default, importance is calculated for all columns in \code{X} (except column names
used as response \code{y} or case weight \code{w}).
}
\details{
The permutation importance of a feature is defined as the increase in the average
loss when shuffling the corresponding feature values before calculating predictions.
By default, the process is repeated \code{m_rep = 4} times, and the results are averaged.
In most of the cases, importance values should be derived from an independent test
data set. Set \code{normalize = TRUE} to get \emph{relative} increases in average loss.
}
\section{Methods (by class)}{
\itemize{
\item \code{perm_importance(default)}: Default method.

\item \code{perm_importance(ranger)}: Method for "ranger" models.

\item \code{perm_importance(Learner)}: Method for "mlr3" models.

\item \code{perm_importance(explainer)}: Method for DALEX "explainer".

}}
\section{Losses}{


The default \code{loss} is the "squared_error". Other choices:
\itemize{
\item "absolute_error": The absolute error is the loss corresponding to median regression.
\item "poisson": Unit Poisson deviance, i.e., the loss function used in
Poisson regression. Actual values \code{y} and predictions must be non-negative.
\item "gamma": Unit gamma deviance, i.e., the loss function of Gamma regression.
Actual values \code{y} and predictions must be positive.
\item "logloss": The Log Loss is the loss function used in logistic regression,
and the top choice in probabilistic binary classification. Responses \code{y} and
predictions must be between 0 and 1. Predictions represent probabilities of
having a "1".
\item "mlogloss": Multi-Log-Loss is the natural loss function in probabilistic multi-class
situations. If there are K classes and n observations, the predictions form
a (n x K) matrix of probabilities (with row-sums 1).
The observed values \code{y} are either passed as (n x K) dummy matrix,
or as discrete vector with corresponding levels.
The latter case is turned into a dummy matrix by a fast version of
\code{model.matrix(~ as.factor(y) + 0)}.
\item "classification_error": Misclassification error. Both the
observed values \code{y} and the predictions can be character/factor. This
loss function can be used in non-probabilistic classification settings.
BUT: Probabilistic classification (with "mlogloss") is clearly preferred in most
situations.
\item A function with signature \code{f(actual, predicted)}, returning a numeric
vector or matrix of the same length as the input.
}
}

\examples{
# MODEL 1: Linear regression
fit <- lm(Sepal.Length ~ ., data = iris)
s <- perm_importance(fit, X = iris, y = "Sepal.Length")
s
s$M
s$SE  # Standard errors are available thanks to repeated shuffling
plot(s)
plot(s, err_type = "SD")  # Standard deviations instead of standard errors

# Groups of features can be passed as named list
v <- list(petal = c("Petal.Length", "Petal.Width"), species = "Species")
s <- perm_importance(fit, X = iris, y = "Sepal.Length", v = v, verbose = FALSE)
s
plot(s)

# MODEL 2: Multi-response linear regression
fit <- lm(as.matrix(iris[, 1:2]) ~ Petal.Length + Petal.Width + Species, data = iris)
s <- perm_importance(fit, X = iris[, 3:5], y = iris[, 1:2], normalize = TRUE)
s
plot(s)
plot(s, swap_dim = TRUE, top_m = 2)
}
\references{
Fisher A., Rudin C., Dominici F. (2018). All Models are Wrong but many are Useful:
Variable Importance for Black-Box, Proprietary, or Misspecified Prediction
Models, using Model Class Reliance. Arxiv.
}
