#' Mapping sequential ui
#'
#' @param id Shiny module identifier
mappingSequentialUserInterface <- function(id) {
    nsMapSeq <- shiny::NS(id)
    bslib::layout_sidebar(
      htmltools::tags$head(
        htmltools::tags$style(htmltools::HTML("
        .tooltip-inner {
          max-width: 400px;  /* Adjust the width */
          white-space: normal;  /* Allow text to wrap */
          text-align: left;  /* Optional: aligns text to the left */
          border-radius: 4px !important;  /* Less curvature */
          background-color: #202020;      /* Optional: lighter background */
          color: #FFFFFF;                 /* Optional: darker text for readability */
          border: 1px solid #ccc;         /* Optional: subtle border */
        }
      "))
      ),
      htmltools::tags$style(".selectize-dropdown {position: static}"),
      shinyjs::useShinyjs(), # Enables JavaScript for more interactive features
      bslib::input_dark_mode(id = "dark_mode"), # Toggle for dark mode

        sidebar = bslib::sidebar(
            width = "23%",
            bslib::accordion(
                    bslib::accordion_panel(
                        "BASIC OPTIONS", icon = bsicons::bs_icon("sliders"),
                        shiny::fluidRow(shiny::column(12,align="center",
                                                      htmltools::tags$hr(htmltools::h5("Step 1 - Set input/output")),
                                        htmltools::br())), #close fluid row
                        ## genome index 1
                        shiny::fluidRow(shiny::column(width = 12,shiny::splitLayout(cellWidths = c("90%", "10%"),
                                                               shinyFiles::shinyDirButton(nsMapSeq("SeqIndex1"),
                                                                              "Select first reference genome index folder",
                                                                              title = "Select the folder containing the first reference genome index",
                                                                              icon = shiny::icon("upload")),
                                                               bslib::tooltip(
                                                                   bsicons::bs_icon("question-circle"),
                                                                   "This folder contains the index for genome 1, generated by the previous genome indexing module in sequential mode.",
                                                                   placement = "right"
                                                               )
                        ),
                        # Displays the selected input directory path
                        shiny::verbatimTextOutput(nsMapSeq("MappingPathIndexOneSequential")), ##cartella INDICE UNO selezionata
                        )), #close fluidrow
                        ## genome index 2
                        shiny::fluidRow(shiny::column(width = 12,shiny::splitLayout(cellWidths = c("90%", "10%"),
                                                               shinyFiles::shinyDirButton(nsMapSeq("SeqIndex2"),
                                                                              "Select second reference genome index folder",
                                                                              title = "Select the folder containing the second reference genome index",
                                                                              icon = shiny::icon("upload")),
                                                               bslib::tooltip(
                                                                   bsicons::bs_icon("question-circle"),
                                                                   "This folder contains the index for genome 2, generated by the previous genome indexing module in sequential mode.",
                                                                   placement = "right"
                                                               )
                        ),
                        # Displays the selected input directory path
                        shiny::verbatimTextOutput(nsMapSeq("MappingPathIndexTwoSequential")), ## index 1 selected
                        )), #close fluidrow
                        shiny::fluidRow(shiny::column(width = 12,shiny::splitLayout(cellWidths = c("90%", "10%"),
                                                               shinyFiles::shinyDirButton(nsMapSeq("SequentialReads"),
                                                                              "Choose the folder that stores the read files",
                                                                              title = "Please select the folder where the sample files (FASTQ format) are stored.",
                                                                              icon = shiny::icon("upload")),
                                                               bslib::tooltip(
                                                                   bsicons::bs_icon("question-circle"),
                                                                   "This is the input folder containing the raw read files in FASTQ format, which can be either compressed or uncompressed. Supported file extensions include:
                                                                                                             filename1_1.fastq
                                                                                                             filename1_2.fastq
                                                                                                             filename1_1.fastq.gz
                                                                                                             filename1_2.fastq.gz
                                                                                                             filename1_1.fq
                                                                                                             filename1_2.fq
                                                                                                             filename1_1.fq.gz
                                                                                                             filename1_2.fq.gz",
                                                                   placement = "right"
                                                               )
                        ),
                        # Displays the selected input directory path
                        shiny::verbatimTextOutput(nsMapSeq("MappingPathProcessedReadsSequential")), ## read folder selected
                        )), #close fluidrow
                        shiny::fluidRow(shiny::column(width = 12,shiny::splitLayout(cellWidths = c("90%", "10%"),
                                                               shinyFiles::shinyDirButton(nsMapSeq("SeqBam1"),
                                                                              "Select first alignment output folder",
                                                                              title = "Select the output folder where the alignments onto the genome 1 will be stored.",
                                                                              icon = shiny::icon("upload")),
                                                               bslib::tooltip(
                                                                   bsicons::bs_icon("question-circle"),
                                                                   "This folder will store the read alignment files for genome 1.",
                                                                   placement = "right"
                                                               )
                        ),
                        # Displays the selected input directory path
                        shiny::verbatimTextOutput(nsMapSeq("MappingPathOutputOneSequential")), ## result 1 folder selected
                        )), #close fluidrow
                        shiny::fluidRow(shiny::column(width = 12,shiny::splitLayout(cellWidths = c("90%", "10%"),
                                                               shinyFiles::shinyDirButton(nsMapSeq("SeqBam2"),
                                                                              "Select second alignment output folder",
                                                                              title = "Select the output folder where the alignments onto the genome 2 will be stored.",
                                                                              icon = shiny::icon("download")),
                                                               bslib::tooltip(
                                                                   bsicons::bs_icon("question-circle"),
                                                                   "This folder will store the read alignment files for genome 2.",
                                                                   placement = "right"
                                                               )
                        ),
                        # Displays the selected input directory path
                        shiny::verbatimTextOutput(nsMapSeq("MappingPathOutputTwoSequential")), ## result folder 2 selected
                        )), #close fluidrow
                    ), #close accordion panel
                  bslib::accordion_panel(
                        "ADVANCED OPTIONS", icon = bsicons::bs_icon("sliders"),
                        shiny::fluidRow(shiny::column(12,align="center",
                                                      htmltools::tags$hr(htmltools::h5("Miscellaneous")),
                                        htmltools::br())),
                        ## numero di processi in parallelo
                        shiny::fluidRow(shiny::column(width = 12,shiny::splitLayout(cellWidths = c("90%", "10%"),
                                                                                    shiny::sliderInput(inputId = nsMapSeq("MappingProcessSequential"),
                                                                           label = "Parallelization",
                                                                           min = 1,
                                                                           max = parallel::detectCores(logical = TRUE),
                                                                           value = 2,
                                                                           step = 1),
                                                               bslib::tooltip(
                                                                   bsicons::bs_icon("question-circle"),
                                                                   "Parallel computation.
                                                                           Choose the number of CPU cores to analize samples in parallel.
                                                                           '2' by default",
                                                                   placement = "right"
                                                               )
                        ))), #close fluidrow
                        shiny::fluidRow(shiny::column(width = 12,shiny::splitLayout(cellWidths = c("90%", "10%"),
                                                                                    shiny::sliderInput(inputId = nsMapSeq("MappingThreadsSequential"), label = "Number of subprocesses",
                                                                           value = 2,
                                                                           min = 1,
                                                                           max = as.integer(parallel::detectCores(logical = TRUE)),
                                                                           step = 1),
                                                               bslib::tooltip(
                                                                   bsicons::bs_icon("question-circle"),
                                                                   "Set the number of subprocesses to divide the analysis into.
                                                                           '2' by default.",
                                                                   placement = "right"
                                                               )
                        ))), #close fluidrow
                        shiny::fluidRow(shiny::column(width = 12,shiny::splitLayout(cellWidths = c("90%", "10%"),
                                                               shiny::radioButtons(inputId = nsMapSeq("MappingFormatSequential"), label = "Output format",
                                                                            choices = c("BAM", "SAM"), selected = "BAM"),
                                                               bslib::tooltip(
                                                                   bsicons::bs_icon("question-circle"),
                                                                   "Choose whether to generate a compressed binary alignment file (BAM) or an uncompressed output (SAM).
                                                                           'BAM' by default.",
                                                                   placement = "right"
                                                               )
                        ))), #close fluidrow
                        shiny::fluidRow(shiny::column(width = 12,shiny::splitLayout(cellWidths = c("90%", "10%"),
                                                               shiny::radioButtons(inputId = nsMapSeq("MappingSeqPhredScore"), label = "Phred score",
                                                                            choices = c(33, 64), selected = 33),
                                                               bslib::tooltip(
                                                                   bsicons::bs_icon("question-circle"),
                                                                   "Specify the base-calling Phred score format (33 or 64) used to define the quality of your FASTQ input data.
                                                                           '33' by default",
                                                                   placement = "right"
                                                               )
                        ))), #close fluidrow
                        shiny::fluidRow(shiny::column(width = 12,shiny::splitLayout(cellWidths = c("90%", "10%"),
                                                               shiny::numericInput(inputId = nsMapSeq("MappingNumberSubreadsSequential"),
                                                                            label = "Number of subreads", value = 14, step = 1),
                                                               bslib::tooltip(
                                                                   bsicons::bs_icon("question-circle"),
                                                                   "Specify the number of subreads (short reads for optimal map location) to extract for each input read.
                                                                           A higher number of subreads may increase both processing time and mapping accuracy.
                                                                           '14' by default.",
                                                                   placement = "right"
                                                               )
                        ))), #close fluidrow
                        shiny::fluidRow(shiny::column(width = 12,shiny::splitLayout(cellWidths = c("90%", "10%"),
                                                               shiny::numericInput(inputId = nsMapSeq("MappingConsensusThresholdSequential"),
                                                                            label = "Consensus threshold", value = 1, step = 1),
                                                               bslib::tooltip(
                                                                   bsicons::bs_icon("question-circle"),
                                                                   "Specify the consensus threshold, which defines the minimum number of consensus subreads required to report a hit. Consensus subreads are those that agree on the same location in the reference genome for the read.
                                                                           Increasing the required number of consensus subreads may reduce the number of mapped reads.
                                                                           '1' by default.",
                                                                   placement = "right"
                                                               )
                        ))), #close fluidrow
                        shiny::fluidRow(shiny::column(width = 12,shiny::splitLayout(cellWidths = c("90%", "10%"),
                                                               shiny::numericInput(inputId = nsMapSeq("MappingMaxMismatchSequential"),
                                                                            label = "Mismatch number", value = 3, step = 1),
                                                               bslib::tooltip(
                                                                   bsicons::bs_icon("question-circle"),
                                                                   "Specify the maximum number of mismatches allowed in the alignment. Mismatches in soft-clipped regions are not counted.
                                                                           '3' by default.",
                                                                   placement = "right"
                                                               )
                        ))), #close fluidrow
                        shiny::fluidRow(shiny::column(width = 12,shiny::splitLayout(cellWidths = c("90%", "10%"),
                                                               shiny::radioButtons(inputId = nsMapSeq("MappingUniqueOnlySequential"), label = "Exclude reads that map to multiple locations",
                                                                            choices = c("TRUE", "FALSE"), selected = "FALSE"),
                                                               bslib::tooltip(
                                                                   bsicons::bs_icon("question-circle"),
                                                                   "Specify if only uniquely mapped reads should be included in the alignment file (setting TRUE).
                                                                           'FALSE' by default.",
                                                                   placement = "right"
                                                               )
                        ))), #close fluidrow
                        shiny::fluidRow(shiny::column(width = 12,shiny::splitLayout(cellWidths = c("90%", "10%"),
                                                               shiny::numericInput(inputId = nsMapSeq("MappingMultiMapMaxSequential"),
                                                                            label = "Max multi-mapped reads", value = 1, step = 1),
                                                               bslib::tooltip(
                                                                   bsicons::bs_icon("question-circle"),
                                                                   "When both uniquely mapped and multimapped reads are included, set the maximum number of equally best mapping locations per read.
                                                                           '1' by default.",
                                                                   placement = "right"
                                                               )
                        ))), #close fluidrow
                        shiny::fluidRow(shiny::column(width = 12,shiny::splitLayout(cellWidths = c("90%", "10%"),
                                                               shiny::numericInput(inputId = nsMapSeq("MappingIndelLengthSequential"),
                                                                            label = "Indel length", value = 5, max = 200, step = 1),
                                                               bslib::tooltip(
                                                                   bsicons::bs_icon("question-circle"),
                                                                   "Specify the maximum number of insertions/deletions (indels) permitted in the alignment.
                                                                           '5' by default.",
                                                                   placement = "right"
                                                               )
                        ))), #close fluidrow
                        shiny::fluidRow(shiny::column(width = 12,shiny::splitLayout(cellWidths = c("90%", "10%"),
                                                               shiny::numericInput(inputId = nsMapSeq("MappingMinFragLengthSequential"),
                                                                            label = "Min fragment length", value = 50, step = 1),
                                                               bslib::tooltip(
                                                                   bsicons::bs_icon("question-circle"),
                                                                   "Set the minimum fragment length as a numeric value.
                                                                           '50' by default.",
                                                                   placement = "right"
                                                               )
                        ))), #close fluidrow
                        shiny::fluidRow(shiny::column(width = 12,shiny::splitLayout(cellWidths = c("90%", "10%"),
                                                               shiny::numericInput(inputId = nsMapSeq("MappingMaxFragLengthSequential"),
                                                                            label = "Max fragment length", value = 600, step = 1),
                                                               bslib::tooltip(
                                                                   bsicons::bs_icon("question-circle"),
                                                                   "Set the minimum fragment length as a numeric value.
                                                                           '600' by default.",
                                                                   placement = "right"
                                                               )
                        ))), #close fluidrow
                        shiny::fluidRow(shiny::column(width = 12,shiny::splitLayout(cellWidths = c("90%", "10%"),
                                                               shiny::radioButtons(inputId = nsMapSeq("MappingPeOrientationSequential"),
                                                                            label = "Orientation of paired-end reads",
                                                                            choices = c("fr", "rf", "ff"), selected = "fr"),
                                                               bslib::tooltip(
                                                                   bsicons::bs_icon("question-circle"),
                                                                   "Specify the orientation of the paired reads. 'f' denotes the forward strand and 'r' denotes the reverse strand. The default is 'fr', meaning the first read is on the forward strand and the second on the reverse strand.",
                                                                   placement = "right"
                                                               )
                        ))), #close fluidrow
                        shiny::fluidRow(shiny::column(width = 12,shiny::splitLayout(cellWidths = c("90%", "10%"),
                                                               shiny::radioButtons(inputId = nsMapSeq("MappingKeepOrderSequential"),
                                                                            label = "Keep the order of FASTQ files (applies only for BAM output)",
                                                                            choices = c("TRUE", "FALSE"), selected = "FALSE"),
                                                               bslib::tooltip(
                                                                   bsicons::bs_icon("question-circle"),
                                                                   "Choose whether to preserve the order of reads in the BAM output as they appear in the input file.
                                                                           Reads from the same pair will always be placed next to each other, regardless of this setting.
                                                                           'FALSE' by default.",
                                                                   placement = "right"
                                                               )
                        ))), #close fluidrow
                        shiny::fluidRow(shiny::column(width = 12,shiny::splitLayout(cellWidths = c("90%", "10%"),
                                                               shiny::radioButtons(inputId = nsMapSeq("MappingSortByCoordinatesSequential"), label = "Sort read by their mapping locations (applies only for BAM output)",
                                                                            choices = c("TRUE", "FALSE"), selected = "FALSE"),
                                                               bslib::tooltip(
                                                                   bsicons::bs_icon("question-circle"),
                                                                   "Decide whether reads should be sorted by their mapping locations in the output.
                                                                           This option is applicable for BAM output only.
                                                                           A BAI index file will also be generated for each BAM file, allowing direct loading into genome browsers like IGB or IGV.
                                                                           'FALSE' by default.",
                                                                   placement = "right"
                                                               )
                        ))), #close fluidrow
                        shiny::fluidRow(shiny::column(width = 12,shiny::splitLayout(cellWidths = c("90%", "10%"),
                                                               shiny::radioButtons(inputId = nsMapSeq("MappingAllJunctionsSequential"), label = "Reporting all splice junctions",
                                                                            choices = c("TRUE", "FALSE"), selected = "FALSE"),
                                                               bslib::tooltip(
                                                                   bsicons::bs_icon("question-circle"),
                                                                   "Choose if all discovered junctions, including exon-exon junctions and gene fusions, should be reported. Setting this option to TRUE does not require the presence of donor/acceptor sites.
                                                                           'FALSE' by default.",
                                                                   placement = "right"
                                                               )
                        ))), #close fluidrow
                        shiny::fluidRow(shiny::column(width = 12,shiny::splitLayout(cellWidths = c("90%", "10%"),
                                                               shiny::numericInput(inputId = nsMapSeq("SubsetBam"),
                                                                            label = "Subset the aligned reads",
                                                                            value = 1000000, min = 1, step = 1),
                                                               bslib::tooltip(
                                                                   bsicons::bs_icon("question-circle"),
                                                                   "Set the number of alignments to process in each batch before writing the alignment file.
                                                                           This option is useful when RAM memory is limited.
                                                                           '1000000' by default.",
                                                                   placement = "right"
                                                               )
                        ))), #close fluidrow
                        shiny::fluidRow(shiny::column(width = 12,shiny::splitLayout(cellWidths = c("90%", "10%"),
                                                               shinyFiles::shinyDirButton(id = nsMapSeq("MappingTemporaryFolderSequential"),
                                                                              label = "Choose an empty folder to store temporary files",
                                                                              title = "Folder for temporary files",
                                                                              icon = shiny::icon("hourglass")),                                       bslib::tooltip(
                                                                                  bsicons::bs_icon("question-circle"),
                                                                                  "Select an empty folder to store temporary files.
                                                                                          This folder and its contents will be deleted after the analysis is complete.",
                                                                                  placement = "right"
                                                                              )
                        )))# close fluidrow
                    ), #close accordion panel
            ), #close accordion
            shiny::fluidRow(shiny::column(12,align="center",
                                          htmltools::tags$hr(htmltools::h5("Step 2 - Run mapping analysis for dual RNA-seq (sequential mode)")),
                            htmltools::br())),
            ## running button
            shiny::fluidRow(
                shiny::actionButton(inputId = nsMapSeq("runMappingSequential"), label = "Run",
                             icon = shiny::icon(name = "bookmark"),
                             class = "btn-primary"),
                ## killing button
                shinyjs::disabled(
                    shiny::actionButton(nsMapSeq("StopMappingSequential"), "Kill", icon = shiny::icon("fire"),
                                 class = "btn-warning"))
            ) #close fluidrow
        ), #close sidebar
        ## MAIN PANEL
        ## DROPDOWN BUTTON
      shinyWidgets::dropdownButton(
            inputId = nsMapSeq("MappingSequentialMydropdown"),
            label = "Info",
            size = "sm",
            status = "info",
            circle = FALSE,
            htmltools::h3(shiny::strong("Mapping process")),
            htmltools::br(),
            htmltools::h5(shiny::strong("WHEN TO PERFORM")),
            htmltools::h5("The mapping step should be performed after genome indexing and before the summarization step to align the read dataset to the reference genome(s)."),
            htmltools::br(),
            htmltools::h6(shiny::strong("WHAT IT DOES")),
            htmltools::h6("During this step, RNA-seq reads are aligned to the indexed reference genome(s) to identify transcripts. Mapping allows for the detection of novel genes or transcripts and requires a spliced or gapped mapper, as reads may span exon-exon junctions. The module also detects novel splice junctions, including those resulting from sequencing errors, differences with the reference, or the search for non-canonical junctions and fusion transcripts."),
            htmltools::br(),
            htmltools::h6(shiny::strong("OPERATIONAL INSTRUCTIONS")),
            htmltools::h6(shiny::strong("Step 1")),
            htmltools::h6("- Select the first and then the second folder where indexed reference genomes are stored (indexed genomes must be generated with the previous indexing module)."),
            htmltools::h6("- Select the folder where samples (FASTQ format) are stored"),
            htmltools::h6("- Choose two empty folders where the alignment files (BAM or SAM format), unmapped reads (FASTA format), and other documents (text format) will be saved."),
            htmltools::h6(shiny::strong("Step 2")),
            htmltools::h6("- Start the analysis by clicking the 'Run' button."),
            htmltools::br(),
            htmltools::h6(shiny::strong("RESULTS")),
            htmltools::h6("Inside each selected folder, the following output files will be generated for each sample:"),
            htmltools::h6("- an alignment file (BAM/SAM format);"),
            htmltools::h6("- a text document reporting detected polymorphisms between the reads and the reference genome (VCF format);"),
            htmltools::h6("- a text document reporting detected exon-exon junctions (BED format);"),
            htmltools::h6("- a text document summarizing the mapping analysis statistics;;"),
            htmltools::h6("- unmapped reads (gzip compressed FASTQ format)."),
            htmltools::br(),
            htmltools::h6(shiny::strong("ADDITIONAL NOTES")),
            htmltools::h6("You can permanently kill the analysis by clicking the 'Kill' button."),
            htmltools::h6("Optionally, you can configure advanced parameters for further customization."),
            htmltools::br()
        ),
        htmltools::br(),
        # text output
        shiny::verbatimTextOutput(nsMapSeq("MappingSequentialProcessstatus")),
        ## reassuring table
        DT::DTOutput(nsMapSeq("MappingSequentialReassure")),
        DT::DTOutput(nsMapSeq("MappingSequentialReassure2"))
    ) #close layoutsidebar
} # close mappingSequentialUI function
