% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/FindOptimalSubset.R
\name{FindOptimalSubset}
\alias{FindOptimalSubset}
\title{Genetic Algorithm for Subset Selection}
\usage{
FindOptimalSubset(n, k, Fitness, ..., popSize = 100L, migrationRate = 0.1,
  migrationInterval = 10L, pcrossover = 0.8, pmutation = 0.1,
  elitism = 0L, maxiter = 1000L, run = maxiter, suggestions = NULL,
  parallel = TRUE, seed = NULL)
}
\arguments{
\item{n}{'integer'.
Maximum permissible index, that is, the length of the finite sequence (\code{1:n}).
The GA chooses a subset from this sequence.}

\item{k}{'integer'.
Number of indices to choose, that is, the fixed size of the subset.}

\item{Fitness}{'function'.
Fitness function, also known as the objective function, is any allowable \R function which
takes as its first argument the binary \code{string} representing a potential solution.
And as its second argument the maximum permissible index, \code{n}.
Use the \code{\link{DecodeChromosome}(string, n)} command to decode the binary \code{string}.
The fitness function returns a single numerical value describing its fitness.
Recall that the GA searches for a maximum fitness value.}

\item{...}{Additional arguments to be passed to the fitness function.}

\item{popSize}{'integer'.
Population size}

\item{migrationRate}{'numeric'.
Proportion of individuals that should migrate between islands.}

\item{migrationInterval}{'integer'.
Number of iterations at which exchange of individuals takes place.
This interval between migrations is called an \emph{epoch}.}

\item{pcrossover}{'numeric'.
Probability of crossover between pairs of chromosomes.}

\item{pmutation}{'numeric'.
Probability of mutation in a parent chromosome.}

\item{elitism}{'integer'.
Number of chromosomes to survive into the next generation.}

\item{maxiter}{'integer'.
Maximum number of iterations to run before the GA search is halted.}

\item{run}{'integer'.
Number of consecutive generations without any improvement in the
\dQuote{best} fitness value before the GA is stopped.}

\item{suggestions}{'matrix'.
Binary representation of integer chromosomes to be included in the initial population.
Use the \code{\link{EncodeChromosome}(string, n)} command
to encode integer chromosomes as bit strings.}

\item{parallel}{'logical' or 'integer'.
Whether to use parallel computing.
This argument can also be used to specify the number of cores
(and number of islands) to employ; by default,
this is taken from \code{\link[parallel]{detectCores}}.
The \pkg{parallel} and \pkg{doParallel} packages must be
installed for parallel computing to work.}

\item{seed}{'integer'.
Random number generator state for random number generation, used to replicate the results.
The \pkg{doRNG} package must be installed if using parallel computing.}
}
\value{
Returns a 'list' with components:
  \describe{
    \item{\code{call}}{original call which can be used for later re-use.}
    \item{\code{solution}}{a 'matrix' representation of the best solution found.
      Each row represents a unique solution giving the best fitness at the final iteration.
      More than one row indicates a non-unique solution.
      The number of columns is equal to the subset size \code{k}.}
    \item{\code{ga_output}}{output from the ISLGAs,
      see \code{\link[=gaisl-class]{gaisl-class}} for format description.}
    \item{\code{ga_time}}{time required to run the ISLGAs,
      see \code{\link{system.time}} for details.}
  }
}
\description{
This function identifies an optimal subset of a fixed size \code{k}
from a finite sequence of length \code{n}.
A distributed multiple-population genetic algorithm (GA) is used to do
subset selection based on the maximization of a user-supplied fitness function.
}
\details{
The fitness function (see \code{Fitness} argument) is solved using
  the \code{\link[GA]{gaisl}} function in the \pkg{GA} package (Scrucca, 2013, 2016).
  The function implements an islands evolution model (first proposed by Cohoon and others, 1987).
  That is, it maximizes a fitness function using islands genetic algorithms (ISLGAs)
  (Luke, 2013, p. 103-104; Scrucca, 2016, p. 197-200).
  Independent GAs are configured to use integer chromosomes
  represented with a binary codification, linear-rank selection,
  uniform crossover, and uniform mutation.
}
\examples{
# Problem: Choose the 4 smallest numbers from a list of 100 values
# genearated from a standard uniform distribution.
k <- 4
n <- 100
seed <- 321
set.seed(seed); numbers <- sort(runif(n))
Fitness <- function(string, n, numbers) {
  idxs <- DecodeChromosome(string, n)
  value <- -sum(numbers[idxs])
  return(value)
}
\dontrun{
out <- FindOptimalSubset(n, k, Fitness, numbers, elitism = 1, run = 10, seed = seed)
plot(out[["ga_output"]])
summary(out[["ga_output"]])
print(out[["solution"]])
print(out[["ga_output"]]@fitnessValue)
}

}
\references{
Cohoon, J.P., Hegde, S.U., Martin, W.N., and Richards, D., 1987,
  Punctuated Equilibria: A Parallel Genetic Algorithm,
  in Genetic Algorithms and their Applications:
  Proceedings of the Second International Conference on Genetic Algorithms,
  Grefenstette, J.J., Lawrence Earlbaum Associates, p. 155-161.

  Luke, Sean, 2015, Essentials of metaheuristics (2nd ed.): Lulu, 263 p.,
  available for free at \url{https://cs.gmu.edu/~sean/book/metaheuristics/}.

  Scrucca, Luca, 2013, GA: A Package for Genetic Algorithms in R:
  Journal of Statistical Software, v. 53, no. 4, p. 1-37.

  Scrucca, Luca, 2017, On some extensions to GA package: hybrid optimisation,
  parallelisation and islands evolution: The R Journal, v. 9, no. 1, p. 187-206.
}
\author{
J.C. Fisher, U.S. Geological Survey, Idaho Water Science Center
}
\keyword{optimize}
