% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/afths.R
\name{afths}
\alias{afths}
\title{Horseshoe shrinkage prior in Bayesian survival regression}
\usage{
afths(ct, X, method.tau = c("fixed", "truncatedCauchy", "halfCauchy"),
  tau = 1, method.sigma = c("fixed", "Jeffreys"), Sigma2 = 1,
  burn = 1000, nmc = 5000, thin = 1, alpha = 0.05, Xtest = NULL,
  cttest = NULL)
}
\arguments{
\item{ct}{survival response, a \eqn{n*2} matrix with first column as response and second column as right censored indicator,
1 is event time and 0 is right censored.}

\item{X}{Matrix of covariates, dimension \eqn{n*p}.}

\item{method.tau}{Method for handling \eqn{\tau}. Select "truncatedCauchy" for full
Bayes with the Cauchy prior truncated to [1/p, 1], "halfCauchy" for full Bayes with
the half-Cauchy prior, or "fixed" to use a fixed value (an empirical Bayes estimate,
for example).}

\item{tau}{Use this argument to pass the (estimated) value of \eqn{\tau} in case "fixed"
is selected for method.tau. Not necessary when method.tau is equal to"halfCauchy" or
"truncatedCauchy". The default (tau = 1) is not suitable for most purposes and should be replaced.}

\item{method.sigma}{Select "Jeffreys" for full Bayes with Jeffrey's prior on the error
variance \eqn{\sigma^2}, or "fixed" to use a fixed value (an empirical Bayes
estimate, for example).}

\item{Sigma2}{A fixed value for the error variance \eqn{\sigma^2}. Not necessary
when method.sigma is equal to "Jeffreys". Use this argument to pass the (estimated)
value of Sigma2 in case "fixed" is selected for method.sigma. The default (Sigma2 = 1)
is not suitable for most purposes and should be replaced.}

\item{burn}{Number of burn-in MCMC samples. Default is 1000.}

\item{nmc}{Number of posterior draws to be saved. Default is 5000.}

\item{thin}{Thinning parameter of the chain. Default is 1 (no thinning).}

\item{alpha}{Level for the credible intervals. For example, alpha = 0.05 results in
95\% credible intervals.}

\item{Xtest}{test design matrix.}

\item{cttest}{test survival response.}
}
\value{
\item{SurvivalHat}{Predictive survival probability}
\item{LogTimeHat}{Predictive log time}
\item{Beta.sHat}{Posterior mean of Beta, a \eqn{p} by 1 vector}
\item{LeftCI.s}{The left bounds of the credible intervals}
\item{RightCI.s}{The right bounds of the credible intervals}
\item{Beta.sMedian}{Posterior median of Beta, a \eqn{p} by 1 vector}
\item{LambdaHat}{Posterior samples of \eqn{\lambda}, a \eqn{p*1} vector}
\item{Sigma2Hat}{Posterior mean of error variance \eqn{\sigma^2}. If method.sigma =
"fixed" is used, this value will be equal to the user-selected value of Sigma2
passed to the function}
\item{TauHat}{Posterior mean of global scale parameter tau, a positive scalar}
\item{Beta.sSamples}{Posterior samples of \eqn{\beta}}
\item{TauSamples}{Posterior samples of \eqn{\tau}}
\item{Sigma2Samples}{Posterior samples of Sigma2}
}
\description{
This function employs the algorithm provided by van der Pas et. al. (2016) for
log normal Accelerated Failure Rate (AFT) model to fit survival regression. The censored observations are updated
according to the data augmentation of approach of Tanner and Wong (1984).
}
\details{
The model is:
 \eqn{t_i} is response,
 \eqn{c_i} is censored time,
 \eqn{t_i^* = \min_(t_i, c_i)} is observed time,
 \eqn{w_i} is censored data, so \eqn{w_i = \log t_i^*} if \eqn{t_i} is event time and
 \eqn{w_i = \log t_i^*} if \eqn{t_i} is right censored
 \eqn{\log t_i=X\beta+\epsilon, \epsilon \sim N(0,\sigma^2)}.
}
\examples{
burnin <- 500
nmc    <- 500
thin <- 1
y.sd   <- 1  # standard deviation of the response

p <- 100  # number of predictors
ntrain <- 100  # training size
ntest  <- 50   # test size
n <- ntest + ntrain  # sample size
q <- 10   # number of true predictos

beta.t <- c(sample(x = c(1, -1), size = q, replace = TRUE), rep(0, p - q))  # randomly assign sign

Sigma <- matrix(0.9, nrow = p, ncol = p)
for(j in 1:p)
{
Sigma[j, j] <- 1
}

x <- mvtnorm::rmvnorm(n, mean = rep(0, p), sigma = Sigma)    # correlated design matrix

tmean <- x \%*\% beta.t
yCorr <- 0.5
yCov <- matrix(c(1, yCorr, yCorr, 1), nrow = 2)


y <- mvtnorm::rmvnorm(n, sigma = yCov)
t <- y[, 1] + tmean
X <- scale(as.matrix(x))  # standarization

t <- as.numeric(as.matrix(c(t)))
T <- exp(t)   # AFT model
C <- rgamma(n, shape = 1.75, scale = 3)   # 42\% censoring time
time <- pmin(T, C)  # observed time is min of censored and true
status = time == T   # set to 1 if event is observed
ct <- as.matrix(cbind(time = time, status = status))  # censored time


# Training set
cttrain <- ct[1:ntrain, ]
Xtrain  <- X[1:ntrain, ]

# Test set
cttest <- ct[(ntrain + 1):n, ]
Xtest  <- X[(ntrain + 1):n, ]

posterior.fit.aft <- afths(ct = cttrain, X = Xtrain, method.tau = "halfCauchy",
                             method.sigma = "Jeffreys", burn = burnin, nmc = nmc, thin = 1,
                             Xtest = Xtest, cttest = cttest)
                             
posterior.fit.aft$Beta.sHat
}
\references{
Maity, A. K., Carroll, R. J., and Mallick, B. K. (2019) 
            "Integration of Survival and Binary Data for Variable Selection and Prediction: 
            A Bayesian Approach", 
            Journal of the Royal Statistical Society: Series C (Applied Statistics).
}
