% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/iv.R
\name{iv}
\alias{iv}
\alias{iv_pairs}
\title{Create an interval vector}
\usage{
iv(start, end, ..., ptype = NULL, size = NULL)

iv_pairs(..., ptype = NULL)
}
\arguments{
\item{start, end}{\verb{[vector]}

A pair of vectors to represent the bounds of the intervals.

To be a valid interval vector, \code{start} must be strictly less than \code{end}.

If either \code{start} or \code{end} are incomplete / missing, then both bounds will
be coerced to missing values.

\code{start} and \code{end} are recycled against each other and are cast to the same
type.}

\item{...}{For \code{iv_pairs()}:

\verb{[vector pairs]}

Vectors of size 2 representing intervals to include in the result.

All inputs will be cast to the same type.

For \code{iv()}:

These dots are for future extensions and must be empty.}

\item{ptype}{\verb{[vector(0) / NULL]}

A prototype to force for the inner type of the resulting iv. If \code{NULL},
this defaults to the common type of the inputs.}

\item{size}{\verb{[integer(1) / NULL]}

A size to force for the resulting iv. If \code{NULL}, this defaults to the
common size of the inputs.}
}
\value{
An iv.
}
\description{
\itemize{
\item \code{iv()} creates an interval vector from \code{start} and \code{end} vectors. This
is how you will typically create interval vectors, and is often used with
columns in a data frame.
\item \code{iv_pairs()} creates an interval vector from \emph{pairs}. This is often useful
for interactive testing, as it provides a more intuitive interface for
creating small interval vectors. It should generally not be used on a large
scale because it can be slow.
}
\subsection{Intervals}{

Interval vectors are \emph{right-open}, i.e. \verb{[start, end)}. This means that
\code{start < end} is a requirement to generate an interval vector. In particular,
empty intervals with \code{start == end} are not allowed.

Right-open intervals tend to be the most practically useful. For example,
\verb{[2019-01-01 00:00:00, 2019-01-02 00:00:00)} nicely encapsulates all times on
\code{2019-01-01}. With closed intervals, you'd have to attempt to specify this as
\verb{2019-01-01 23:59:59}, which is inconvenient and inaccurate, as it doesn't
capture fractional seconds.

Right-open intervals also have the extremely nice technical property that
they create a closed algebra. Concretely, the complement of a vector of
right-open intervals and the union, intersection, or difference of two
vectors of right-open intervals will always result in another vector of
right-open intervals.
}

\subsection{Missing intervals}{

When creating interval vectors with \code{iv()}, if either bound is
\link[vctrs:vec_detect_complete]{incomplete}, then both bounds are set to
their missing value.
}
}
\examples{
library(dplyr, warn.conflicts = FALSE)

set.seed(123)

x <- tibble(
  start = as.Date("2019-01-01") + 1:5,
  end = start + sample(1:10, length(start), replace = TRUE)
)

# Typically you'll use `iv()` with columns of a data frame
mutate(x, iv = iv(start, end), .keep = "unused")

# `iv_pairs()` is useful for generating interval vectors interactively
iv_pairs(c(1, 5), c(2, 3), c(6, 10))
}
