#' Bivariate kernel copula density estimation
#' 
#' Based on samples from a bivariate copula, the copula density is estimated.
#' The user can choose between different methods. If no bandwidth is provided
#' by the user, it will be set by a method-specific automatic selection
#' procedure. The related (d/p/r)kdecop functions evaluate the density and cdf 
#' or simulate synthetic data, respectively.
#' 
#' Details on the estimation methods and bandwidth selection can be found in
#' Geenens et al. (2014) for methods \code{TLL1/2} and Nagler (2014) for other
#' methods. We use a Gaussian product kernel function for all methods except
#' the beta kernel estimator.\cr 
#' 
#' Kernel estimates are usually no proper copula densities. In particular, the
#' estimated marginal densities are not uniform. We mitigate this issue bei
#' implementing a renormalization procedure. The number of iterations of the
#' renormalization algorithm can be specified with the \code{renorm.iter}
#' argument. Typically, a very small number of iterations is sufficient.
#'  
#' @param udata \code{nx2} matrix of copula data.
#' @param bw bandwidth specification; if missing, \code{bw} is selected
#' automatically; otherwise, a positive real number for methods \code{"MR"} and
#' \code{"beta"}, a \eqn{2x2} matrix for method \code{"T"}, or a a list \code{B
#' = <your.B>, alpha = <your.alpha>} containing the \eqn{2x2} rotation matrix
#' and the nearest-neighbor fraction alpha.
#' @param mult bandwidth multiplier, has to be positive; the bandwidth used for
#' estimation is \code{bw*mult}.
#' @param method \code{"MR"}: mirror-reflection estimator, \cr \code{"beta"}:
#' beta kernel estimator, \cr \code{"T"}: transformation estimator with
#' standard bivariate kernel estimation, \cr \code{"TLL1"}: transformation
#' estimator with log-linear local likelihood estimation and nearest-neighbor
#' bandwidths, \cr \code{"TLL2"}: transformation estimator with log-quadradtic
#' local likelihood estimation and nearest-neighbor bandwidths.
#' @param knots integer; number of knots in each dimension for the spline
#' approximation; defaults to \code{50}.
#' @param renorm.iter integer; number of iterations for the renormalization
#' procedure (see \emph{Details}).
#' @param info logical; if \code{TRUE}, additional information about the
#' estimate will be gathered (see \emph{Value}).
#' 
#' 
#' @return The function \code{\link[kdecopula:kdecop]{kdecop}} returns an
#' object of class \code{kdecopula} that contains all information necessary for
#' evaluation of the estimator. If no bandwidth was provided in the function
#' call, the automatically selected value can be found in the variable
#' \code{object$bw}. If \code{info=TRUE}, also the following will be available
#' under \code{object$info}: \item{likvalues}{Estimator evaluated in sample
#' points} \item{loglik}{Log likelihood} \item{effp}{Effective number of
#' parameters} \item{AIC}{Akaike information criterion}
#' \item{cAIC}{Bias-corrected version of Akaike information criterion}
#' \item{BIC}{Bayesian information criterion.} \cr The density estimate can be
#' evaluated on arbitrary points with \code{\link[kdecopula:dkdecop]{dkdecop}};
#' the cdf with \code{\link[kdecopula:pkdecop]{pkdecop}}. Furthermore,
#' synthetic data can be simulated with
#' \code{\link[kdecopula:rkdecop]{rkdecop}}, and several plot options are
#' provided by \code{\link[kdecopula:plot.kdecopula]{plot.kdecopula}}.
#' 
#' 
#' @author Thomas Nagler
#' 
#' @seealso 
#' \code{\link[kdecopula:kdecopula]{kdecopula}},
#' \code{\link[kdecopula:plot.kdecopula]{plot.kdecopula}},
#' \code{\link[kdecopula:dkdecop]{dkdecop}},
#' \code{\link[kdecopula:pkdecop]{pkdecop}},
#' \code{\link[kdecopula:rkdecop]{rkdecop}}
#' 
#' @references 
#' Geenens, G., Charpentier, A., and Paindaveine, D. (2014).
#' Probit transformation for nonparametric kernel estimation of the copula
#' density.
#' arXiv:1404.4414 [stat.ME]. 
#' \cr \cr 
#' Nagler, T. (2014). 
#' Kernel Methods for Vine Copula Estimation.
#' Master's Thesis, Technische Universitaet Muenchen,
#' \url{https://mediatum.ub.tum.de/node?id=1231221} 
#' 
#' 
#' @examples
#' 
#' ## load data and transform with empirical cdf
#' data(wdbc)
#' udat <- apply(wdbc[, -1], 2, function(x) rank(x)/(length(x)+1))
#' 
#' ## estimation of copula density of variables 5 and 6
#' dens.est <- kdecop(udat[, 5:6])
#' plot(dens.est) 
#' 
#' ## evaluate density estimate at (u1,u2)=(0.123,0.321)
#' dkdecop(c(0.123, 0.321), dens.est) 
#' 
#' ## evaluate cdf estimate at (u1,u2)=(0.123,0.321)
#' pkdecop(c(0.123, 0.321), dens.est) 
#' 
#' ## simulate 500 samples from density estimate
#' rkdecop(500, dens.est)
#' 
kdecop <- function(udata, bw, mult = 1, method = "TLL2", knots = NA, renorm.iter = 3L, info = TRUE) {
    udata <- as.matrix(udata)
    n <- nrow(udata)
    d <- ncol(udata)
    
    ## sanity checks
    if (n < 2)
        stop("Number of observations has to be at least 2.")
    if (d != 2)
        stop("Dimension has to be 2.")
    if (any(udata > 1) | any(udata < 0))
        stop("'udata' have to be in the interval [0,1].")
    if (!(method %in% c("MR", "beta", "T", "TLL1", "TLL2")))
        stop("method not implemented")
    if (mult <= 0)
        stop("'mult' has to be a positive number.")
    if (is.na(knots))
        knots <- 100/d
    knots <- round(knots)
    stopifnot(is.numeric(renorm.iter))
    renorm.iter <- round(renorm.iter)
    stopifnot(renorm.iter >= 0)
    stopifnot(is.logical(info))
    
    ## bandwidth selection and adjustment (with bandwidth checks)
    if (missing(bw))
        bw <- bw_select(udata, method)
    if (method %in% c("TLL1", "TLL2")) {
        if (is.null(bw$B) | is.null(bw$alpha))
            stop("For methods 'TLL1/2', you have to provide a list 'bw = list(B = <your.B>,  alpha = <your.alpha>)'
where both parts of the bandwidth specification are provided via  'your.B', and 'your.alpha'.")
        if (any(c(diag(bw$B), bw$alpha) <= 0))
            stop("Bandwidths have to be positive.")
        bw$alpha <- mult * bw$alpha
    } else if (method %in% c("T")) {
        B <- as.matrix(bw)
        if (nrow(B) == 1)
            bw <- diag(as.numeric(bw), d)
        bw <- mult * bw
        bw <- bw * sqrt(min((n^(-1/6))^2 / det(bw), 1))
    } else if (method == "MR") {
        bw <- min(bw * mult, 1)
    } else if (method == "beta") {
        bw <- bw * mult
    }
    
    ## fit model for method TLL
    if (method %in% c("TLL1", "TLL2")) {
        zdata <-  qnorm(udata)
        lfit <- my_locfit(zdata,
                          bw$B,
                          bw$alpha,
                          deg = as.numeric(substr(method, 4, 4)))
    } else {
        lfit <- NA
    }
    
    ## construct grid with k knots in dimension d
    pnts <- pnorm(seq(-3.25, 3.25, l = knots))
    # pnts <- 1:knots/(knots + 1)
    grid <- as.matrix(do.call(expand.grid,
                              split(rep(pnts, d), ceiling(1:(knots*d)/knots))))
    
    ## evaluate estimator on grid
    evalf <- eval_func(method)  # get evaluation function
    object <- list(udata = udata,
                   bw = bw,
                   lfit = lfit,
                   method = method)
    vals <- array(evalf(grid, obj=object), dim = rep(knots, d))
    
    ## rescale copula density to have uniform margins
    if (renorm.iter > 0) {
        tmplst <- split(rep(seq.int(knots)-1, d-1),
                        ceiling(seq.int(knots*(d-1))/knots))
        helpind <- as.matrix(do.call(expand.grid, tmplst))
        vals <- renorm(vals, pnts, renorm.iter, helpind)
    }
    
    ## store results
    res <- list(udata    = udata,
                grid     = pnts,
                estimate = vals,
                bw       = bw,
                method   = method)
    class(res) <- "kdecopula"
    
    if (info) {
        # likelihood
        likvalues <- dkdecop(udata, res)
        loglik <- sum(log(likvalues))
        # effective number of parameters
        effp <- eff_num_par(udata, likvalues, bw, method, lfit)
        # information criteria
        AIC  <- - 2 * loglik + 2 * effp
        cAIC <- AIC + (2 * effp * (effp + 1)) / (n - effp - 1)
        BIC  <- - 2 * loglik + log(n) * effp
        
        ## store results
        res$info <- list(likvalues = likvalues,
                         loglik    = loglik,
                         effp      = effp ,
                         AIC       = AIC,
                         cAIC      = cAIC,
                         BIC       = BIC)
    }
    
    ## return results as kdecopula object
    res
}
