% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/learning_rate_schedules.R
\name{learning_rate_schedule_inverse_time_decay}
\alias{learning_rate_schedule_inverse_time_decay}
\title{A LearningRateSchedule that uses an inverse time decay schedule}
\usage{
learning_rate_schedule_inverse_time_decay(
  initial_learning_rate,
  decay_steps,
  decay_rate,
  staircase = FALSE,
  ...,
  name = NULL
)
}
\arguments{
\item{initial_learning_rate}{A scalar \code{float32} or \code{float64} \code{Tensor} or an
R number. The initial learning rate.}

\item{decay_steps}{A scalar \code{int32} or \code{int64} \code{Tensor} or an R number. How
often to apply decay.}

\item{decay_rate}{An R number. The decay rate.}

\item{staircase}{Boolean. Whether to apply decay in a discrete staircase, as
opposed to continuous, fashion.}

\item{...}{For backwards and forwards compatibility}

\item{name}{String.  Optional name of the operation.  Defaults to
'InverseTimeDecay'.}
}
\description{
A LearningRateSchedule that uses an inverse time decay schedule
}
\details{
When training a model, it is often useful to lower the learning rate as
the training progresses. This schedule applies the inverse decay function
to an optimizer step, given a provided initial learning rate.
It requires a \code{step} value to compute the decayed learning rate. You can
just pass a TensorFlow variable that you increment at each training step.

The schedule is a 1-arg callable that produces a decayed learning
rate when passed the current optimizer step. This can be useful for changing
the learning rate value across different invocations of optimizer functions.
It is computed as:

\if{html}{\out{<div class="sourceCode R">}}\preformatted{decayed_learning_rate <- function(step) \{
  initial_learning_rate / (1 + decay_rate * step / decay_step)
\}
}\if{html}{\out{</div>}}

or, if \code{staircase} is \code{TRUE}, as:

\if{html}{\out{<div class="sourceCode R">}}\preformatted{decayed_learning_rate function(step) \{
 initial_learning_rate / (1 + decay_rate * floor(step / decay_step))
\}
}\if{html}{\out{</div>}}

You can pass this schedule directly into a keras Optimizer
as the \code{learning_rate}.

Example: Fit a Keras model when decaying \code{1/t} with a rate of \code{0.5}:

\if{html}{\out{<div class="sourceCode R">}}\preformatted{...
initial_learning_rate <- 0.1
decay_steps <- 1.0
decay_rate <- 0.5
learning_rate_fn <- learning_rate_schedule_inverse_time_decay(
  initial_learning_rate, decay_steps, decay_rate)

model \%>\%
  compile(optimizer = optimizer_sgd(learning_rate = learning_rate_fn),
          loss = 'sparse_categorical_crossentropy',
          metrics = 'accuracy')

model \%>\% fit(data, labels, epochs = 5)
}\if{html}{\out{</div>}}
}
\seealso{
\itemize{
\item \url{https://www.tensorflow.org/api_docs/python/tf/keras/optimizers/schedules/InverseTimeDecay}
}
}
