% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kexpmv.R
\name{expokit_dmexpv}
\alias{expokit_dmexpv}
\title{EXPOKIT dmexpv matrix exponentiation on a square matrix}
\usage{
expokit_dmexpv(mat = NULL, t = 15, vector = NULL,
  transpose_needed = TRUE, transform_to_crs = TRUE, crs_n = NULL,
  anorm = NULL, mxstep = 10000, tol = as.numeric(1e-10))
}
\arguments{
\item{mat}{an input square matrix.}

\item{t}{a time value to exponentiate by.}

\item{vector}{If NULL (default), the full matrix exponential is returned. However, in 
order to fully exploit the efficient speed of EXPOKIT on sparse matrices, this vector argument should
be equal to a vector, v. This vector is an n dimensional vector, which in the Markovian case, can
represent the starting probabilities.}

\item{transpose_needed}{If TRUE (default), matrix will be transposed before the exponential operator
is performed.}

\item{transform_to_crs}{If the input matrix is in square format then the matrix will
need transformed into CRS format. This is required for EXPOKIT's sparse-matrix functions DMEXPV and 
DGEXPV. Default TRUE; if FALSE, then the \code{mat} argument must be a CRS-formatted matrix.}

\item{crs_n}{If a CRS matrix is input, \code{crs_n} specifies the order (# of rows  or # of columns) of
the matrix. Default is NULL.}

\item{anorm}{The \code{expokit_dmexpv} requires an initial guess at the norm of the matrix. Using the
R function \code{\link{norm}} might get slow with large matrices. Default is NULL.}

\item{mxstep}{The EXPOKIT code performs integration steps and \code{mxstep} is the maximum number of 
integration steps performed. Default is 10000 steps; May need to increase this value if function 
outputs a warning message.}

\item{tol}{the tolerance defined for the calculations.}
}
\description{
This function converts a matrix to CRS format and exponentiates
it via the EXPOKIT dmexpv function with the wrapper functions \code{wrapalldmexpv_} 
and \code{dmexpv_} around dmexpv. This function can be used to calculate both the 
matrix exponential in isolation or the product of the matrix exponential with a 
vector. This can be achieved by modifying the vector variable as shown below.\cr
}
\details{
From EXPOKIT:\cr
\code{*     The method used is based on Krylov subspace projection}\cr
\code{*     techniques and the matrix under consideration interacts only}\cr
\code{*     via the external routine 'matvec' performing the matrix-vector}\cr
\code{*     product (matrix-free method).}\cr
\code{*}\cr
\code{*     This is a customised version for Markov Chains. This means that a}\cr
\code{*     check is done within this code to ensure that the resulting vector}\cr
\code{*     w is a probability vector, i.e., w must have all its components }\cr
\code{*     in [0,1], with sum equal to 1. This check is done at some expense}\cr
\code{*     and the user may try DGEXPV which is cheaper since it ignores }\cr
\code{*     probability constraints.}\cr

This check assumes that the transition matrix Q, satisfies Qe = 0 where e is a column 
vector of 1's. If this condition does not hold then use the DGEPXV function instead. It 
should be noted that both the DMEXPV and DGEXPV functions within EXPOKIT require the 
matrix-vector product y = A*x = Q'*x i.e, where A is the transpose of Q. Failure to
remember this leads to wrong results.\cr

CRS (Compressed Row Storage) format is a compressed format that is
required for EXPOKIT's sparse-matrix functions such as DGEXPV and
DMEXPV. However this format is not necessary in EXPOKIT's padm-related functions.\cr

This function is recommended for large sparse matrices, however the infinity norm of the matrix 
proves to be crucial when selecting the most efficient routine. If the infinity norm of the large
sparse matrix is approximately >100 may be of benefit to use the \code{expokit_dgpadm} function 
for faster computations.
}
\examples{
# Make a square matrix A
# Use expokit_dmexpv to calculate both exp(At) and exp(At)v, where t is a 
# time value and v is an n dimensional column vector.
mat=matrix(c(-0.071207, 0.065573, 0.005634, 0, -0.041206, 0.041206, 0, 0, 0), 
nrow=3, byrow=TRUE)

# Set the time variable t
t=15

# Exponentiate with EXPOKIT's dmexpv to obtain the full matrix exponential
OutputMat = expokit_dmexpv(mat=mat, t=t, transpose_needed=TRUE, vector=NULL)

print(OutputMat$output_mat)
print(OutputMat$message)

# Can also calculate the matrix exponential with the product of a vector.
# Create the n dimensional vector
v = matrix(0,3,1)
v[1] = 1

# Exponentiate with EXPOKIT's dmexpv
OutputMat = expokit_dmexpv(mat=mat, t=t, transpose_needed=TRUE, vector=v)

print(OutputMat$output_probs)
print(OutputMat$message)

# If message is 'NULL' then no error has occured and the number of 
# mxsteps defined in the function is acceptable.

}
\author{
Meabh G. McCurdy \email{mmccurdy01@qub.ac.uk}
}
