% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/return_level.R
\name{return_level}
\alias{return_level}
\title{Return Level Inferences for Stationary Extreme Value Models}
\usage{
return_level(
  x,
  m = 100,
  level = 0.95,
  npy = 1,
  prof = TRUE,
  inc = NULL,
  type = c("vertical", "cholesky", "spectral", "none")
)
}
\arguments{
\item{x}{An object inheriting from class \code{"lax"} returned from
\code{\link{alogLik}}.}

\item{m}{A numeric scalar.  The return period, in units of the number
of observations.  See \strong{Details} for information.}

\item{level}{A numeric scalar in (0, 1).  The confidence level required for
confidence interval for the \code{m}-observation return level.}

\item{npy}{A numeric scalar.  The}

\item{prof}{A logical scalar.  Should we calculate intervals based on
profile loglikelihood?}

\item{inc}{A numeric scalar. Only relevant if \code{prof = TRUE}. The
increment in return level by which we move upwards and downwards from the
MLE for the return level in the search for the lower and upper confidence
limits.  If this is not supplied then \code{inc} is set to one hundredth
of the length of the symmetric confidence interval for return level.}

\item{type}{A character scalar.  The argument \code{type} to the function
returned by \code{\link[chandwich]{adjust_loglik}}, that is, the type of
adjustment made to the independence loglikelihood function in creating
an adjusted loglikelihood function.  See \strong{Details} and
\strong{Value} in \code{\link[chandwich]{adjust_loglik}}.}
}
\value{
A object (a list) of class \code{"retlev", "lax"} with the
  components
  \item{rl_sym,rl_prof }{Named numeric vectors containing the respective
    lower 100\code{level}\% limit, the MLE and the upper
    100\code{level}\% limit for the return level.
    If \code{prof = FALSE} then \code{rl_prof} will be missing.}
  \item{rl_se }{Estimated standard error of the return level.}
  \item{max_loglik,crit,for_plot }{If \code{prof = TRUE} then
    these components will be present, containing respectively: the maximised
    loglikelihood; the critical value and a matrix with return levels in
    the first column (\code{ret_levs}) and the corresponding values of the
    (adjusted) profile loglikelihood (\code{prof_loglik}).}
  \item{m,level }{The input values of \code{m} and \code{level}.}
  \item{call }{The call to \code{return_level}.}
}
\description{
Calculates point estimates and confidence intervals for \code{m}-observation
return levels for \strong{stationary} extreme value fitted model objects
returned from \code{\link{alogLik}}.  Two types of interval may be returned:
(a) intervals based on approximate large-sample normality of the maximum
likelihood estimator for return level, which are symmetric about the point
estimate, and (b) profile likelihood-based intervals based on an (adjusted)
loglikelihood.
}
\details{
At present \code{return_level} only supports GEV models.

  Care must be taken in specifying the input value of \code{m},
  taking into account the parameterisation of the original fit.

  For GEV models it is common for each observation to relate to a year.
  In this event the \code{m}-observation return level is an \code{m}-year
  return level.

  For details about the definition and estimation of return levels see
  Chapter 3 and 4 of Coles (2001).

  The profile likelihood-based intervals are calculated by
  reparameterising in terms of the \code{m}-year return level and estimating
  the values at which the (adjusted) profile loglikelihood reaches
  the critical value \code{logLik(x) - 0.5 * stats::qchisq(level, 1)}.
  This is achieved by calculating the profile loglikelihood for a sequence
  of values of this return level as governed by \code{inc}. Once the profile
  loglikelhood drops below the critical value the lower and upper limits are
  estimated by interpolating linearly between the cases lying either side of
  the critical value. The smaller \code{inc} the more accurate (but slower)
  the calculation will be.
}
\examples{
got_evd <- requireNamespace("evd", quietly = TRUE)

if (got_evd) {
  library(evd)
  # An example from the evd::fgev documentation
  set.seed(4082019)
  uvdata <- evd::rgev(100, loc = 0.13, scale = 1.1, shape = 0.2)
  M1 <- fgev(uvdata)
  adj_fgev <- alogLik(M1)
  # Large inc set here for speed, sacrificing accuracy
  rl <- return_level(adj_fgev, inc = 0.5)
  summary(rl)
  plot(rl)
}

got_ismev <- requireNamespace("ismev", quietly = TRUE)

if (got_ismev) {
  library(ismev)
  # An example from the ismev::gev.fit documentation
  gev_fit <- gev.fit(revdbayes::portpirie, show = FALSE)
  adj_gev_fit <- alogLik(gev_fit)
  # Large inc set here for speed, sacrificing accuracy
  rl <- return_level(adj_gev_fit, inc = 0.05)
  summary(rl)
  plot(rl)
}
}
\references{
Coles, S. G. (2001) \emph{An Introduction to Statistical
  Modeling of Extreme Values}, Springer-Verlag, London.
  \doi{10.1007/978-1-4471-3675-0_3}
}
\seealso{
\code{\link{plot.retlev}} for plotting the profile loglikelihood
  for a return level.
}
