#' Create species' values based on the species co-occurrence within focal ranges
#' 
#' @author Bruno Vilela & Fabricio Villalobos
#' 
#' @description Create single species' values based on the attributes of species co-occurring within individual ranges.
#'
#' @usage lets.field(x, y, z, weigth=TRUE, count=FALSE)
#' 
#' @param x A PresenceAbsence object.
#' @param y Species attribute to be considered. It must be a numeric attribute.
#' @param z Species names in the same order as the attributes.
#' @param weigth If \code{TRUE} the value is weighted by species' range size, 
#' if \code{FALSE} the value is the mean of all species that co-occur within the
#'  focal species.
#' @param count Logical, if \code{TRUE} a counting window will open.
#'  
#' @details If the species do not co-occur with any other species NaN will be returned. 
#'
#' @references Villalobos, F. and Arita, H.T. 2010. The diversity field of New World leaf-nosed bats (Phyllostomidae). Global Ecology and Biogeography. 19, 200-211. 
#' @references Villalobos, F., Rangel, T.F., and Diniz-Filho, J.A.F. 2013. Phylogenetic fields of species: cross-species patterns of phylogenetic structure and geographical coexistence. Proceedings of the Royal Society B. 280, 20122570
#' 
#' @seealso \code{\link{lets.presab.birds}}
#' @seealso \code{\link{lets.presab}}
#' 
#' @examples \dontrun{
#' data(PAM)
#' range <- lets.rangesize(x=PAM, units="cell")
#' field <- lets.field(PAM, range, PAM$S, weigth=TRUE)
#' }
#' 
#' @export


lets.field <- function(x, y, z, weigth=TRUE, count=FALSE){
  
 if(is.factor(y)){
  y <- as.numeric(levels(y))[y]
 }

 p <- x$P[, -(1:2)]
 p2 <- p


 for(i in 1:ncol(p2)){
  pos <- which(z==x$S[i])
  if(length(pos)>0){
    p2[, i] <- p2[, i]*y[pos]
    pos2 <- which(p2[, i]==0)
    p2[pos2, i] <- NA
  }else{
    p2[, i] <- NA
  }  
 }

 media <- numeric(ncol(p))
 n <- length(media)
 
 
 if(count == TRUE){
 dev.new(width=2, height=2, pointsize = 12)
 par(mar=c(0, 0, 0, 0)) 
 
 for(i in 1:n){
   
   plot.new()
   text(0.5, 0.5, paste(paste("Total:", n, "\n","Runs to go: ", (n-i))))
   
  pos3 <- p[, i]==1
  p3 <- p[pos3, -i]
  p4 <- p2[pos3, -i]
  mult <- p3*p4
  if(weigth==T){
   media[i] <- mean(mult, na.rm=T)
  }
  if(weigth==F){
  mult <- matrix(mult, ncol=(ncol(p)-1)) 
  me <- colMeans(mult, na.rm=T)
  media[i] <- mean(me, na.rm=T)
  }
 }
 dev.off()
 }
 
 if(count == FALSE){
   for(i in 1:n){
          
     pos3 <- p[, i]==1
     p3 <- p[pos3, -i]
     p4 <- p2[pos3, -i]
     mult <- p3*p4
     if(weigth==T){
       media[i] <- mean(mult, na.rm=T)
     }
     if(weigth==F){
       mult <- matrix(mult, ncol=(ncol(p)-1)) 
       me <- colMeans(mult, na.rm=T)
       media[i] <- mean(me, na.rm=T)
     }
   }
 }
 
 resultado <- cbind(x$S, media)
 colnames(resultado) <- c("Species", "Value")
 resultado <- as.data.frame(resultado)
 resultado[, 2] <- as.numeric(levels(resultado[, 2]))[resultado[, 2]]
 return(resultado)
}
