\name{cgsolve}
\alias{cgsolve}
\title{Solve a symmetric linear system with the conjugate gradient method}
\description{

\code{cgsolve} uses a conjugate gradient algorithm to solve the linear
system \eqn{A x = b} where \eqn{A} is a symmetric matrix.  \code{cgsolve} is
used internally in \pkg{lfe} in the routines \code{\link{fevcov}} and
\code{\link{bccorr}}, but has been made public because it might be
useful for other purposes as well.

}

\usage{ cgsolve(A, b, eps=0.001, init=NULL, symmtest=FALSE, name="")}

\arguments{
  \item{A}{matrix, Matrix or function.}
  \item{b}{vector or matrix of columns vectors.}
  \item{eps}{numeric. Tolerance.}
  \item{init}{numeric. Initial guess.}
  \item{symmtest}{logical. Should the matrix be tested for symmetry?}
  \item{name}{character. Arbitrary name used in progress reports.}
}

\value{

 A solution \eqn{x} of the linear system \eqn{A x = b} is returned. 

}

\details{

  The argument \code{A} can be a symmetric matrix or a symmetric
  sparse matrix inheriting from \code{"Matrix"} of the package
  \pkg{Matrix}.  It can also be a function which performs the
  matrix-vector product. If so, the function must be able to take a
  matrix input if \code{b} is a matrix.

  If the matrix \code{A} is rank deficient, some solution is returned.
  If there is no solution, a vector is returned which may or may not be
  close to a solution.  If \code{symmtest} is \code{FALSE}, no check is
  performed that \code{A} is symmetric. If not symmetric, \code{cgsolve}
  is likely to raise an error about divergence.

  The tolerance \code{eps} is a relative tolerance, i.e.
  \eqn{||x - x_0|| < \epsilon ||x_0||} where \eqn{x_0} is the true
  solution and \eqn{x} is the solution returned by \code{cgsolve}. Use a
  negative \code{eps} for absolute tolerance.  The termination criterion
  for \code{cgsolve} is the one from \cite{Kaasschieter (1988)},
  Algorithm 3.

  Preconditioning is currently not supported.
  
  If \code{A} is a function, the test for symmetry is performed by
  drawing two random vectors \code{x,y}, and testing whether \eqn{|(Ax,
  y) - (x, Ay)| < 10^{-6} sqrt((||Ax||^2 + ||Ay||^2)/N)}, where \eqn{N} is
  the vector length. Thus, the test is neither deterministic nor
  perfect.

}
\seealso{
\code{\link{kaczmarz}}
}

\references{
  Kaasschieter, E. (1988)
  \cite{A practical termination criterion for the conjugate gradient
    method}, BIT Numerical Mathematics, 28(2):308-322.
  \url{http://link.springer.com/article/10.1007\%2FBF01934094}
}
\examples{
  N <- 100000
# create some factors
  f1 <- factor(sample(34000,N,replace=TRUE))
  f2 <- factor(sample(25000,N,replace=TRUE))
# a matrix of dummies, which probably is rank deficient
  B <- makeDmatrix(list(f1,f2))
  dim(B)
# create a right hand side
  b <- as.matrix(B \%*\% rnorm(ncol(B)))
# solve B' B x = B' b
  sol <- cgsolve(crossprod(B), crossprod(B, b), eps=-1e-2)
  #verify solution
  sqrt(sum((B \%*\% sol - b)^2))
}