% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gap_detection.R
\name{gap_detection}
\alias{gap_detection}
\title{Gap detection in a Canopy Height Model}
\usage{
gap_detection(
  las,
  res = 1,
  ratio = 2,
  gap_max_height = 1,
  min_gap_surface = 25,
  max_gap_surface = +Inf,
  closing_height_bin = 1,
  nl_filter = "Median",
  nl_size = 3,
  gap_reconstruct = FALSE
)
}
\arguments{
\item{las}{An object of class \code{\link[terra]{SpatRaster-class}},
\code{\link[lidR]{LAS-class}} or \code{\link[lidR]{LAScatalog-class}}}

\item{res}{numeric. The size of a grid cell in point cloud coordinates units,
used to rasterize the point cloud. In case the \code{las} argument is a 
\code{SpatRaster} \code{res} is not used.}

\item{ratio}{numeric. maximum ratio between surrounding canopy height and gap
distance (a pixel belongs to the gap only if for any vegetation pixel around
it, the distance to the vegetation pixel is larger than pixel height/ratio).
If ratio is set to NULL, this criterion is not taken into account}

\item{gap_max_height}{numeric. maximum canopy height to be considered as gap}

\item{min_gap_surface}{numeric. minimum gap surface}

\item{max_gap_surface}{numeric. maximum gap surface}

\item{closing_height_bin}{numeric. height bin width for morphological closing
of gaps to test ratio between canopy height and gap distance}

\item{nl_filter}{string. type of non-linear filter to apply to canopy height
model to remove artefacts, should be an option of \code{\link{dem_filtering}}}

\item{nl_size}{numeric. kernel width in pixel for non-linear filtering}

\item{gap_reconstruct}{boolean. default behaviour is that areas that do not
fulfill the ratio criterion are removed from gaps. If set to TRUE, in case
some pixels of a gap fulfill the distance criterion, the connected pixels that
fulfill the height criterion are also integrated to it.}
}
\value{
A \code{SpatRaster} object with three layers: gap labels, gap surface 
and canopy height model after filter.
}
\description{
Performs gaps detection on a canopy height model provided as object  of class 
\code{\link[terra]{SpatRaster-class}}, or computed from the point cloud of 
objects of class \code{\link[lidR]{LAS-class}} or 
\code{\link[lidR]{LAScatalog-class}}. Function \code{\link{dem_filtering}} 
is first applied to the canopy height model to remove artefacts. 
Gaps are then extracted based on several criteria:
\enumerate{
\item Vegetation height must be smaller than a threshold
\item Gap width must be large enough, depending on surrounding canopy height;
distance to surrounding vegetation is tested with morphological closings
\item Gap must have a minimum surface
}
}
\examples{
data(chm_chablais3)
chm_chablais3 <- terra::rast(chm_chablais3)

# fill NA values in canopy height model
chm_chablais3[is.na(chm_chablais3)] <- 0

# gap detection with distance larger than canopy height / 2
gaps <- gap_detection(chm_chablais3, ratio = 2, gap_max_height = 1, 
min_gap_surface = 0)

# gap detection with distance larger than canopy height / 2
# and reconstruction of border areas
gaps1 <- gap_detection(chm_chablais3,
  ratio = 2, gap_max_height = 1, min_gap_surface = 0,
  gap_reconstruct = TRUE
)

# gap detection without distance criterion
gaps2 <- gap_detection(chm_chablais3, ratio = NULL, gap_max_height = 1, 
min_gap_surface = 0)

# gap id and corresponding surface for third detection parameters
table(terra::values(gaps2$gap_id)) * terra::res(gaps2$gap_id)[1]^2

# plot original image
terra::plot(chm_chablais3, main = "Initial image")

# plot binary image of gaps
terra::plot(gaps$gap_id > 0, main = "Gaps", legend = FALSE)
terra::plot(gaps1$gap_id > 0, main = "Gaps, with reconstruction", legend = FALSE)
terra::plot(gaps2$gap_id > 0, main = "Gaps, no width criterion", legend = FALSE)

# plot filtered CHM
terra::plot(gaps2$filled_chm, main = "Filtered CHM")
}
\seealso{
\code{\link{dem_filtering}}, \code{\link{edge_detection}}
}
