% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lmvar.R
\name{lmvar}
\alias{lmvar}
\title{Linear regression with non-constant variances}
\usage{
lmvar(y, X_mu = NULL, X_sigma = NULL, check_hessian = TRUE, ...)
}
\arguments{
\item{y}{Vector of observations}

\item{X_mu}{Model matrix for the expected values \eqn{\mu}}

\item{X_sigma}{Model matrix for the logarithms of the standard deviations \eqn{\sigma}}

\item{check_hessian}{Boolean, if TRUE it is checked whether the Hessian is negative-definite}

\item{...}{Additional arguments, not used in the current implementation}
}
\value{
An object of class 'lmvar'.
}
\description{
Performs a Gaussian regression with non-constant variances and outputs an 'lmvar' object.
}
\details{
If the matrix \code{X_mu} is not specified, the model for the expected values \eqn{\mu} will consist of an intercept term only.
Likewise, the model for \eqn{\log \sigma} will consist of an intercept term only if \code{X_sigma} is not specified.
In the latter case, the model reduces to a standard linear model.

The input matrices can be of class \code{\link{matrix}} or of class \link[Matrix:Matrix-class]{Matrix}

When \code{check_hessian = TRUE}, it is checked whether the fitted log-likelihood is at a maximum. A warning will be issued if
that is not the case.

The vector \code{y} must be a numeric vector. It can not contain special values like \code{NULL}, \code{NaN}, etc.

Both model matrices must be numeric matrices. They can not contain special values like \code{NULL}, \code{NaN}, etc.

All columns in the matrix \code{X_mu} must either have a name, or no column has a name at all. It is not allowed that some
colums have a
name while others don't. The same is true for \code{X_sigma}.

If supplied, the column names
must be unique for \code{X_mu}. The same is true for \code{X_sigma}. A column name can be used in both \code{X_mu} and
\code{X_sigma} though.

In case the matrix \code{X_mu} has no columns with a column name, \code{lmvar}  gives the names \code{v1},
\code{v2} etc. to the columns. Likewise, if the matrix \code{X_sigma} has no columns with a column name, \code{lmvar}
gives the names \code{v1_s}, \code{v2_s} etc. to the columns.

Matrix \code{X_mu} can not have a column with the name '(Intercept)'.
Matrix \code{X_sigma} can not have a column with the name '(Intercept_s)'. Both names are reserved names.

The function \code{nleqslv} from the \code{nleqslv} package, is used to solve the non-linear equations that comprise the fit.
\code{nleqslv} returns a
termination code which reports whether a solution was found, see \code{\link[nleqslv]{nleqslv}}.
A potential problem is reported by a termination code different from \code{1}. Termination codes different from \code{1}
as passed on by \code{lmvar}. Termination codes \code{2} and \code{3} are passed on as a warning.
Other termination codes are passed on as an error.

An \code{lmvar} object is a list. Users are discouraged to access list-members directly.
Instead, list-members are to be accessed with the various accessor and utility functions in the package.
Exceptions are the following list members for which no accessor functions exist:
\itemize{
\item \code{object$y} the vector of observations
\item \code{object$X_mu} the  model matrix for \eqn{\mu}. In general, it differs from the user-supplied \code{X_mu} because
\code{lmvar} adds an intercept-column and makes the matrix full-rank.
\item \code{object$X_sigma} the  model matrix for \eqn{\log \sigma}. In general, it differs from the user-supplied
\code{X_sigma} because
\code{lmvar} adds an intercept-column and makes the matrix full-rank.
}

See the vignettes
that come with the \code{lmvar} package for more info. Run \code{vignette(package="lmvar")} to list the available vignettes.
}
\examples{
# As example we use the dataset 'attenu' from the library 'datasets'. The dataset contains
# the response variable 'accel' and two explanatory variables 'mag'  and 'dist'.
library(datasets)

# For more info on the data, study the dataset
help("attenu")

# Create the model matrix for the expected values
X = cbind(attenu$mag, attenu$dist)
colnames(X) = c("mag", "dist")

# Create the model matrix for the standard deviations. The standard deviation
# is large for small distances and small for large distances. The use of 'dist'
# as explanatory variable makes the beta for the intercept term blow up.
# Therefore we use '1 / dist' as explanatory variable
X_s = cbind(attenu$mag, 1 / attenu$dist)
colnames(X_s) = c("mag", "dist_inv")

# Carry out the fit
fit_lmvar  = lmvar(attenu$accel, X, X_s)

# Inspect the results. Note from the p-value for the difference in
# deviance that this fit appears to be significantly better than
# a classical linear fit
summary(fit_lmvar)

# Carry out a classical linear fit for comparison
fit_lm = lm(attenu$accel ~ mag + dist, attenu)

# A comparison of the AIC values also favours the fit with 'lmvar'
AIC(fit_lm)
AIC(fit_lmvar)
}
