% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lmvar_no_fit.R
\name{lmvar_no_fit}
\alias{lmvar_no_fit}
\title{Create an 'lmvar'-like object without a model fit}
\usage{
lmvar_no_fit(y, X_mu = NULL, X_sigma = NULL, intercept_mu = TRUE,
  intercept_sigma = TRUE, sigma_min = 0, slvr_options = list(method =
  "NR"), control = list(), ...)
}
\arguments{
\item{y}{Vector of observations}

\item{X_mu}{Model matrix for the expected values \eqn{\mu}}

\item{X_sigma}{Model matrix for the logarithms of the standard deviations \eqn{\sigma}}

\item{intercept_mu}{see the function \code{\link{lmvar}}}

\item{intercept_sigma}{see the function \code{lmvar}.}

\item{sigma_min}{see the function \code{lmvar}.}

\item{slvr_options}{see the function \code{lmvar}}

\item{control}{see the function \code{lmvar}.}

\item{...}{Additional arguments, not used in the current implementation}
}
\value{
An object of class 'lmvar_no_fit', which is a list. The list-members are the same as for an object of call 'lmvar', but with
members that are the result of the model fit left out.

Users are discouraged to access list-members directly.
Instead, list-members are to be accessed with the various accessor and utility functions in the package.
Exceptions are the following list members for which no accessor functions exist:
\itemize{
\item \code{call} the function call
\item \code{y} the vector of observations
\item \code{X_mu} the  model matrix for \eqn{\mu}. In general, it differs from the user-supplied \code{X_mu} because
\code{lmvar_no_fit} adds an intercept-column (unless \code{intercept_mu} is \code{FALSE}) and makes the matrix full-rank.
\item \code{X_sigma} the  model matrix for \eqn{\log \sigma}. In general, it differs from the user-supplied
\code{X_sigma} because \code{lmvar_no_fit} adds an intercept-column (unless \code{intercept_sigma} is {FALSE}) and makes
the matrix full-rank.
\item \code{intercept_mu} boolean which tells whether or not an intercept column \code{(Intercept)} has been added to the
model matrix \eqn{X_\mu}
\item \code{intercept_sigma} boolean which tells whether or not an intercept column \code{(Intercept_s)} has been added to the
model matrix \eqn{X_\sigma}
\item \code{sigma_min} the value of the argument \code{sigma_min} in the call of \code{lmvar}
\item \code{slvr_options} the value of the argument \code{slvr_options} in the call of \code{lmvar}
\item \code{control} the value of the argument \code{control} in the call of \code{lmvar}
}
}
\description{
Creates an 'lmvar'-like object without carrying out a model fit. This object is a 'lmvar' object from which all members
have been left out
that are the result of the fit. Such an object can be used in functions which typically use an 'lmvar' object as input but do not
need the fit results. Since no fit is performed, \code{lmvar_no_fit} is convenient when the fit is time-consuming or, e.g.,
does not converge.
}
\details{
See \code{\link{lmvar}} for the requirements and a further explanation of all the arguments.

The class 'lmvar' is an extension of the class 'lmvar_no_fit'. This means that each object which is of class 'lmvar', is  of class
'lmvar_no_fit' as well. Wherever an object of class 'lmvar_no_fit' is required, an object of class 'lmvar' can be used as well.

Accessor and utility functions for a 'lmvar_no_fit' object are: \code{\link{nobs.lmvar_no_fit}}, \code{\link{alias.lmvar_no_fit}}
and \code{\link{dfree}}

The function \code{lmvar_no_fit} is especially useful in combination with \code{\link{fwbw.lmvar_no_fit}}. In case a
model with many degrees of freedom does not converge with \code{\link{lmvar}}, one can create an 'lmvar_no_fit' object. This
is used as
input for \code{fwbw} with the argument \code{fw = TRUE}. The \code{fwbw} algorithm will try to select an optimal subset of all
degrees of freedom, starting with the smallest subset possible.

Although \code{lmvar_no_fit} does not carry out a model fit, it will do the following:
\itemize{
\item add an intercept term to the model matrices (unless \code{intercept_mu} is \code{FALSE} and/or
\code{intercept_sigma} is \code{FALSE}), and
\item make the model matrices full rank.
}
}
\examples{
# As example we sue the dataset 'iris' from the library 'datasets'
library(datasets)

# Create the model matrix for both the expected values and the standard deviations
X = model.matrix( ~ Species - 1, data = iris)

# Take as response variabe the variable Sepal.length
y = iris$Sepal.Length

# Construct a 'lmvar_no_fit' object
no_fit = lmvar_no_fit( y, X, X)

# The following functions all work on such an object
nobs(no_fit)
dfree(no_fit)
alias(no_fit)

# You can also supply 'lmvar' arguments
no_fit = lmvar_no_fit( y, X[,-1], X[,-1], intercept_mu = FALSE, intercept_sigma = FALSE)
dfree(no_fit)

# Some (most) arguments have no effect except that they are stored in the 'lmvar_no_fit'
# object
no_fit = lmvar_no_fit( y, X, X, control = list( slvr_log = TRUE, remove_df_sigma = TRUE))
no_fit$control
}
