#' @title Determine the size of an LPJmL input/output file header
#'
#' @description Returns the size in bytes of an LPJmL file header based on a
#'   header list object read by [`read_header()`] or generated by
#'   [`create_header()`].
#'
#' @param header Header list object as returned by [`read_header()`] or
#'   [`create_header()`].
#'
#' @return Integer value giving the size of the header in bytes. This can be
#'   used when seeking in the file or to calculate the expected total file size
#'   in combination with the number of included data values and the data type.
#'
#' @examples
#' \dontrun{
#' header <- read_header("filename.clm")
#' size <- get_headersize(header)
#' # Open file for reading
#' fp <- file("filename.clm", "rb")
#' # Skip over file header
#' seek(fp, size)
#' # Add code to read data from file
#' }
#' @seealso
#' * [`read_header()`] for reading a header from an LPJmL input/output file.
#' * [`create_header()`] for creating a header from scratch.
#' @export
get_headersize <- function(header) {
  is_valid_header(header)
  if (header$header["version"] < 1 || header$header["version"] > 4) {
    stop("Invalid header version. Expecting value between 1 and 4.")
  }
  nchar(header$name) +
    switch(as.integer(header$header["version"]), 7, 9, 11, 13) * 4
}
