% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/esp_get_nuts.R
\name{esp_get_nuts}
\alias{esp_get_nuts}
\title{Get NUTS boundaries of Spain}
\source{
\href{https://gisco-services.ec.europa.eu/distribution/v2/}{GISCO API}
}
\usage{
esp_get_nuts(
  year = "2016",
  epsg = "4258",
  cache = TRUE,
  update_cache = FALSE,
  cache_dir = NULL,
  verbose = FALSE,
  resolution = "01",
  spatialtype = "RG",
  region = NULL,
  nuts_level = "all",
  moveCAN = TRUE
)
}
\arguments{
\item{year}{Release year. One of "2003", "2006",`"2010", "2013", "2016" or
"2021".}

\item{epsg}{projection of the map: 4-digit \href{https://epsg.io/}{EPSG code}.
One of:
\itemize{
\item "4258": ETRS89
\item "4326": WGS84
\item "3035": ETRS89 / ETRS-LAEA
\item "3857": Pseudo-Mercator
}}

\item{cache}{A logical whether to do caching. Default is \code{TRUE}.}

\item{update_cache}{A logical whether to update cache. Default is \code{FALSE}.
When set to \code{TRUE} it would force a fresh download of the source
\code{.geojson} file.}

\item{cache_dir}{A path to a cache directory. The directory can also be
set globally with:
\itemize{
\item \code{options(mapSpain_cache_dir = "path/to/dir")}.
}

See Details on \code{\link[=esp_get_nuts]{esp_get_nuts()}}.}

\item{verbose}{Display information. Useful for debugging,
default is \code{FALSE}.}

\item{resolution}{Resolution of the geospatial data. One of
\itemize{
\item "60": 1:60million
\item "20": 1:20million
\item "10": 1:10million
\item "03": 1:3million
\item "01": 1:1million
}}

\item{spatialtype}{Type of geometry to be returned:
\itemize{
\item "RG": Regions - \code{MULTIPOLYGON/POLYGON} object.
\item "LB": Labels - \code{POINT} object.
}}

\item{region}{Optional. A vector of region names, NUTS or ISO codes
(see \code{\link[=esp_dict_region_code]{esp_dict_region_code()}}.}

\item{nuts_level}{NUTS level. One of "0" (Country-level), "1", "2" or "3".
See \url{https://ec.europa.eu/eurostat/web/nuts/background}.}

\item{moveCAN}{A logical \code{TRUE/FALSE} or a vector of coordinates
\code{c(lat, lon)}. It places the Canary Islands close to Spain's mainland.
Initial position can be adjusted using the vector of coordinates.}
}
\value{
A \code{POLYGON/POINT} object.
}
\description{
Loads a simple feature (\code{sf}) object containing the NUTS boundaries of Spain.
}
\details{
\code{cache_dir = NULL} (default) uses and creates \verb{/mapSpain} directory in the
temporary directory \code{\link[=tempdir]{tempdir()}}. The directory can also be set via options
with \code{options(mapSpain_cache_dir = "path/to/dir")} or
\code{options(gisco_cache_dir = "path/to/dir")} (See \code{\link[giscoR:gisco_get]{giscoR::gisco_get()}})

Sometimes cached files may be corrupt. On that case, try redownloading
the data using \code{update_cache = TRUE}.
}
\note{
Please check the download and usage provisions on
\code{\link[giscoR:gisco_attributions]{giscoR::gisco_attributions()}}

While \code{moveCAN} is useful for visualization, it would alter the actual
geographical position of the Canary Islands. When using the output for
spatial analysis or using tiles (\code{\link[=esp_getTiles]{esp_getTiles()}} or
\code{\link[=addProviderEspTiles]{addProviderEspTiles()}})  this option should be set to \code{FALSE} in order to
get the actual coordinates.
}
\examples{

library(sf)

pal <- hcl.colors(5, palette = "Lisbon")

NUTS1 <- esp_get_nuts(nuts_level = 1, moveCAN = TRUE)
plot(st_geometry(NUTS1), col = pal)

NUTS1_alt <- esp_get_nuts(nuts_level = 1, moveCAN = c(15, 0))
plot(st_geometry(NUTS1_alt), col = pal)

NUTS1_orig <- esp_get_nuts(nuts_level = 1, moveCAN = FALSE)
plot(st_geometry(NUTS1_orig), col = pal)

AndOriental <-
  esp_get_nuts(region = c("Almeria", "Granada", "Jaen", "Malaga"))
plot(st_geometry(AndOriental), col = pal)

RandomRegions <- esp_get_nuts(region = c("ES1", "ES300", "ES51"))
plot(st_geometry(RandomRegions), col = pal)

MixingCodes <- esp_get_nuts(region = c("ES4", "ES-PV", "Valencia"))
plot(st_geometry(MixingCodes), col = pal)
}
\seealso{
\code{\link{esp_nuts.sf}}, \code{\link[=esp_dict_region_code]{esp_dict_region_code()}}, \code{\link{esp_codelist}},
\code{\link[giscoR:gisco_get]{giscoR::gisco_get()}}.
}
\author{
dieghernan, \url{https://github.com/dieghernan/}
}
\concept{political}
