% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/TMVN_sampler.R
\name{create_TMVN_sampler}
\alias{create_TMVN_sampler}
\title{Set up a sampler object for sampling from a possibly truncated and degenerate multivariate normal distribution}
\usage{
create_TMVN_sampler(
  Q,
  mu = NULL,
  Xy = NULL,
  update.Q = FALSE,
  update.mu = update.Q,
  name = "x",
  coef.names = NULL,
  R = NULL,
  r = NULL,
  S = NULL,
  s = NULL,
  lower = NULL,
  upper = NULL,
  check.constraints = FALSE,
  method = NULL,
  reduce = NULL,
  chol.control = chol_control(),
  debug = FALSE
)
}
\arguments{
\item{Q}{precision matrix of the (unconstrained) multivariate normal distribution.}

\item{mu}{mean of the (unconstrained) multivariate normal distribution.}

\item{Xy}{alternative to specifying mu; in this case \code{mu} is computed as \eqn{Q^{-1}\code{Xy}}.}

\item{update.Q}{whether \code{Q} is updated for each draw. Currently only supported
by methods 'direct' and 'HMC'.}

\item{update.mu}{whether \code{mu} is updated for each draw. By default equal to \code{update.Q}.
Currently only supported by methods 'direct' and 'HMC'.}

\item{name}{name of the TMVN vector parameter.}

\item{coef.names}{optional labels for the components of the vector parameter.}

\item{R}{equality restriction matrix.}

\item{r}{rhs vector for equality constraints \eqn{R'x = r}, where \eqn{R'} denotes the transpose of R.}

\item{S}{inequality restriction matrix.}

\item{s}{rhs vector for inequality constraints \eqn{S'x >= s}, where \eqn{S'} denotes the transpose of S.}

\item{lower}{alternative to \code{s} for two-sided inequality restrictions \eqn{\code{lower} <= S'x <= \code{upper}}.}

\item{upper}{alternative to \code{s} for two-sided inequality restrictions \eqn{\code{lower} <= S'x <= \code{upper}}.}

\item{check.constraints}{if \code{TRUE} check whether the starting values satisfy all constraints.}

\item{method}{sampling method. The options are "direct" for direct sampling from the
unconstrained or equality constrained multivariate normal (MVN). For inequality constrained
MVN sampling three methods are supported: "HMC" for (exact) Hamiltonian Monte Carlo,
"HMCZigZag" for (exact) Hamiltonian Monte Carlo with Laplace momentum, "Gibbs" for a
component-wise Gibbs sampling approach, and "softTMVN" for a data augmentation method that samples
from a smooth approximation to the truncated MVN. Alternatively, the method setting
functions \code{m_direct}, \code{m_HMC}, \code{m_HMC_ZigZag}, \code{m_Gibbs} or
\code{m_softTMVN} can be used to select the method and possibly set some of its
options to non-default values, see \code{\link{TMVN-methods}}.}

\item{reduce}{whether to a priori restrict the simulation to the subspace defined by the
equality constraints.}

\item{chol.control}{options for Cholesky decomposition, see \code{\link{chol_control}}.}

\item{debug}{if \code{TRUE} a breakpoint is set at the beginning of the TMVN sampling
function \code{draw}.}
}
\value{
An environment for sampling from a possibly degenerate and truncated multivariate normal
 distribution.
}
\description{
This function sets up an object for multivariate normal sampling based on a specified precision matrix.
Linear equality and inequality restrictions are supported.
For sampling under inequality restrictions four algorithms are available. The default in that case is
an exact Hamiltonian Monte Carlo algorithm (Pakman and Paninski, 2014). A related algorithm is the zig-zag
Hamiltonian Monte Carlo method (Nishimura et al., 2021) in which momentum is sampled from a Laplace instead
of normal distribution. Alternatively, a Gibbs sampling algorithm can be used (Rodriguez-Yam et al., 2004).
The fourth option is a data augmentation method that samples from a smooth approximation to the truncated
multivariate normal distribution (Souris et al., 2018).
}
\details{
The componentwise Gibbs sampler uses univariate truncated normal samplers as described
in Botev and L'Ecuyer (2016). These samplers are implemented in R package \pkg{TruncatedNormal},
but here translated to C++ for an additional speed-up.
}
\examples{
\donttest{
S <- cbind(diag(2), c(-1, 1), c(1.1, -1))  # inequality matrix
# S'x >= 0 represents the wedge x1 <= x2 <= 1.1 x1
# example taken from Pakman and Paninski (2014)
# 1. exact Hamiltonian Monte Carlo (Pakman and Paninski, 2014)
sampler <- create_TMVN_sampler(Q=diag(2), mu=c(4, 4), S=S, method="HMC")
sim <- MCMCsim(sampler, n.iter=600, verbose=FALSE)
summary(sim)
plot(as.matrix(sim$x), pch=".")
# 2. exact Hamiltonian Monte Carlo with Laplace momentum (Nishimura et al., 2021)
sampler <- create_TMVN_sampler(Q=diag(2), mu=c(4, 4), S=S, method="HMCZigZag")
sim <- MCMCsim(sampler, n.iter=600, verbose=FALSE)
summary(sim)
plot(as.matrix(sim$x), pch=".")
# 3. Gibbs sampling approach (Rodriguez-Yam et al., 2004)
sampler <- create_TMVN_sampler(Q=diag(2), mu=c(4, 4), S=S, method="Gibbs")
sim <- MCMCsim(sampler, burnin=500, n.iter=2000, verbose=FALSE)
summary(sim)
plot(as.matrix(sim$x), pch=".")
# 4. soft TMVN approximation (Souris et al., 2018)
sampler <- create_TMVN_sampler(Q=diag(2), mu=c(4, 4), S=S, method="softTMVN")
sim <- MCMCsim(sampler, n.iter=600, verbose=FALSE)
summary(sim)
plot(as.matrix(sim$x), pch=".")
}

}
\references{
Z.I. Botev and P. L'Ecuyer (2016).
   Simulation from the Normal Distribution Truncated to an Interval in the Tail.
   in VALUETOOLS.

 Y. Cong, B. Chen and M. Zhou (2017).
   Fast simulation of hyperplane-truncated multivariate normal distributions.
   Bayesian Analysis 12(4), 1017-1037.

 Y. Li and S.K. Ghosh (2015). Efficient sampling methods for truncated multivariate normal
   and student-t distributions subject to linear inequality constraints.
   Journal of Statistical Theory and Practice 9(4), 712-732.

 A. Nishimura, Z. Zhang and M.A. Suchard (2021). Hamiltonian zigzag sampler got more momentum
   than its Markovian counterpart: Equivalence of two zigzags under a momentum refreshment limit.
   arXiv:2104.07694.

 A. Pakman and L. Paninski (2014).
   Exact Hamiltonian Monte Carlo for truncated multivariate gaussians.
   Journal of Computational and Graphical Statistics 23(2), 518-542.

 G. Rodriguez-Yam, R.A. Davis and L.L. Scharf (2004).
   Efficient Gibbs sampling of truncated multivariate normal with application to constrained linear regression.
   Unpublished manuscript.

 H. Rue and L. Held (2005).
   Gaussian Markov Random Fields.
   Chapman & Hall/CRC.

 A. Souris, A. Bhattacharya and P. Debdeep (2018).
   The Soft Multivariate Truncated Normal Distribution.
   arXiv:1807.09155.

 K.A. Valeriano, C.E. Galarza and L.A. Matos (2023).
   Moments and random number generation for the truncated elliptical family of distributions.
   Statistics and Computing 33(1), 1-20.
}
\author{
Harm Jan Boonstra, with help from Grzegorz Baltissen
}
