% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mcmcse.R
\name{mcse}
\alias{mcse}
\title{Compute Monte Carlo standard errors for expectations.}
\usage{
mcse(x, size = NULL, g = NULL, r = 3,  method = "bm", warn = FALSE)
}
\arguments{
\item{x}{a vector of values from a Markov chain of length n.}

\item{size}{represents the batch size in \dQuote{\code{bm}} and the truncation point in \dQuote{\code{bartlett}} and 
\dQuote{\code{tukey}}. Default is \code{NULL} which implies that an optimal batch size is calculated using the 
  \code{batchSize} function. Can take character values of \dQuote{\code{sqroot}} and \dQuote{\code{cuberoot}} or any numeric
  value between 1 and n/2. \dQuote{\code{sqroot}} means size is \eqn{\lfloor n^{1/2} \rfloor} and \dQuote{\code{cuberoot}} means size is
  \eqn{\lfloor n^{1/3} \rfloor}.}

\item{g}{a function such that \eqn{E(g(x))} is the quantity of interest. The default is
\code{NULL}, which causes the identity function to be used.}

\item{r}{The lugsail parameters (\code{r}) that converts a lag window into its lugsail
equivalent. Larger values of \code{r} will typically imply less underestimation of \dQuote{\code{cov}},
but higher variability of the estimator. Default is \code{r = 3} and \code{r = 1,2} are
also good choices, but will likely underestimation of variance. \code{r > 5} is not recommended.}

\item{method}{any of \dQuote{\code{bm}},\dQuote{\code{obm}},\dQuote{\code{bartlett}}, \dQuote{\code{tukey}}. \dQuote{\code{bm}}
represents batch means estimator, \dQuote{\code{obm}} represents overlapping batch means estimator with, \dQuote{\code{bartlett}}
and \dQuote{\code{tukey}} represents the modified-Bartlett window and the Tukey-Hanning windows for spectral variance estimators.}

\item{warn}{a logical value indicating whether the function should issue a warning if the sample
size is too small (less than 1,000).}
}
\value{
\code{mcse} returns a list with three elements:
        \item{est}{an estimate of \eqn{E(g(x))}.}
        \item{se}{the Monte Carlo standard error.}
        \item{nsim}{The number of samples in the input Markov chain.}
}
\description{
Compute Monte Carlo standard errors for expectations.
}
\examples{

## Bivariate Normal with mean (mu1, mu2) and covariance sigma
n <- 1e3
mu = c(2, 50)
sigma = matrix(c(1, 0.5, 0.5, 1), nrow = 2)
out = BVN_Gibbs(n, mu, sigma)
x = out[,1]
mcse(x)
mcse.q(x, 0.1)
mcse.q(x, 0.9)

# Estimate the mean, 0.1 quantile, and 0.9 quantile with MCSEs using overlapping batch means.

mcse(x, method = "obm")
mcse.q(x, 0.1, method = "obm")
mcse.q(x, 0.9, method = "obm")

# Estimate E(x^2) with MCSE using spectral methods.

g = function(x) { x^2 }
mcse(x, g = g, method = "tukey")

}
\references{
Flegal, J. M. (2012) Applicability of subsampling bootstrap methods in Markov chain Monte Carlo.
In Wozniakowski, H. and Plaskota, L., editors, \emph{Monte Carlo and Quasi-Monte Carlo Methods
2010}, pp. 363-372. Springer-Verlag.

Flegal, J. M. and Jones, G. L. (2010) Batch means and spectral variance estimators in Markov
chain Monte Carlo. \emph{The Annals of Statistics}, \bold{38}, 1034--1070.

Flegal, J. M. and Jones, G. L. (2011) Implementing Markov chain Monte Carlo: Estimating with
confidence. In Brooks, S., Gelman, A., Jones, G. L., and Meng, X., editors, \emph{Handbook of
Markov Chain Monte Carlo}, pages 175--197. Chapman & Hall/CRC Press.

Doss, C. R., Flegal, J. M., Jones, G. L., and Neath, R. C. (2014). Markov chain Monte Carlo estimation of quantiles. \emph{Electronic Journal of Statistics}, \bold{8}, 2448-2478.
Jones, G. L., Haran, M., Caffo, B. S. and Neath, R. (2006) Fixed-width output analysis for Markov
chain Monte Carlo. \emph{Journal of the American Statistical Association}, \bold{101}, 1537--154.
}
\seealso{
\code{\link{mcse.mat}}, which applies \code{mcse} to each column of a matrix or data frame.

\code{\link{mcse.multi}}, for a multivariate estimate of the Monte Carlo standard error.

\code{\link{mcse.q}} and \code{\link{mcse.q.mat}}, which compute standard errors for quantiles.
}
