% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mdgc.R
\name{mdgc_fit}
\alias{mdgc_fit}
\title{Estimate the Model Parameters}
\usage{
mdgc_fit(
  ptr,
  vcov,
  mea,
  lr = 0.001,
  rel_eps = 0.001,
  maxit = 25L,
  batch_size = NULL,
  method = c("svrg", "adam", "aug_Lagran"),
  seed = 1L,
  epsilon = 1e-08,
  beta_1 = 0.9,
  beta_2 = 0.999,
  n_threads = 1L,
  do_reorder = TRUE,
  abs_eps = -1,
  maxpts = 10000L,
  minvls = 100L,
  verbose = FALSE,
  decay = 0.98,
  conv_crit = 1e-06,
  use_aprx = FALSE,
  mu = 1,
  lambda = NULL
)
}
\arguments{
\item{ptr}{returned object from \code{\link{get_mdgc_log_ml}}.}

\item{vcov, mea}{starting value for the covariance matrix and the
non-zero mean entries.}

\item{lr}{learning rate.}

\item{rel_eps}{relative error for each marginal likelihood factor.}

\item{maxit}{maximum number of iteration.}

\item{batch_size}{number of observations in each batch.}

\item{method}{estimation method to use. Can be \code{"svrg"}, \code{"adam"},
or \code{"aug_Lagran"}.}

\item{seed}{fixed seed to use. Use \code{NULL} if the seed should not be
fixed.}

\item{epsilon, beta_1, beta_2}{ADAM parameters.}

\item{n_threads}{number of threads to use.}

\item{do_reorder}{logical for whether to use a heuristic variable
reordering. \code{TRUE} is likely the best option.}

\item{abs_eps}{absolute convergence threshold for each marginal likelihood
factor.}

\item{maxpts}{maximum number of samples to draw for each marginal likelihood
term.}

\item{minvls}{minimum number of samples.}

\item{verbose}{logical for whether to print output during the estimation.}

\item{decay}{the learning rate used by SVRG is given by \code{lr * decay^iteration_number}.}

\item{conv_crit}{relative convergence threshold.}

\item{use_aprx}{logical for whether to use an approximation of
\code{\link{pnorm}} and \code{\link{qnorm}}. This may yield a
noticeable reduction in the computation time.}

\item{mu}{starting value for the penalty in the augmented Lagrangian
method.}

\item{lambda}{starting values for the Lagrange multiplier estimates.
\code{NULL} yields a default.}
}
\value{
An \code{\link{list}} with the following elements:
\item{result}{\code{\link{list}} with two elements: \code{vcov} is the
estimated covariance matrix and \code{mea} is the estimated non-zero mean
terms.}
\item{estimates}{If present, the estimated parameters after each iteration.}
\item{fun_vals}{If present, the output of \code{\link{mdgc_log_ml}} after
each iteration.}
\item{mu,lambda}{If present, the \code{mu} and \code{lambda} values at the
end.}

The elements that may be present depending on the chosen \code{method}.
}
\description{
Estimates the covariance matrix and the non-zero mean terms.
The \code{lr} parameter and the \code{batch_size} parameter are likely
data dependent.
Convergence should be monitored e.g. by using \code{verbose = TRUE}
with \code{method = "svrg"}.

See the README at \url{https://github.com/boennecd/mdgc} for examples.
}
\examples{
\donttest{
# randomly mask data
set.seed(11)
masked_data <- iris
masked_data[matrix(runif(prod(dim(iris))) < .10, NROW(iris))] <- NA

# use the functions in the package
library(mdgc)
obj <- get_mdgc(masked_data)
ptr <- get_mdgc_log_ml(obj)
start_vals <- mdgc_start_value(obj)

fit <- mdgc_fit(ptr, start_vals, obj$means, rel_eps = 1e-2, maxpts = 10000L,
                minvls = 1000L, use_aprx = TRUE, batch_size = 100L, lr = .001,
                maxit = 100L, n_threads = 2L)
fit$result$vcov
fit$result$mea
}

}
\references{
Kingma, D.P., & Ba, J. (2015). \emph{Adam: A Method for Stochastic Optimization}. abs/1412.6980.

Johnson, R., & Zhang, T. (2013). \emph{Accelerating stochastic gradient descent using predictive variance reduction}. In Advances in neural information processing systems.
}
\seealso{
\code{\link{mdgc_log_ml}}, \code{\link{mdgc_start_value}},
\code{\link{mdgc_impute}}.
}
