% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/waas.R
\name{waas}
\alias{waas}
\title{Weighted Average of Absolute Scores}
\usage{
waas(
  .data,
  env,
  gen,
  rep,
  resp,
  block = NULL,
  mresp = NULL,
  wresp = NULL,
  prob = 0.05,
  naxis = NULL,
  ind_anova = FALSE,
  verbose = TRUE
)
}
\arguments{
\item{.data}{The dataset containing the columns related to Environments,
Genotypes, replication/block and response variable(s).}

\item{env}{The name of the column that contains the levels of the
environments.}

\item{gen}{The name of the column that contains the levels of the genotypes.}

\item{rep}{The name of the column that contains the levels of the
replications/blocks.}

\item{resp}{The response variable(s). To analyze multiple variables in a
single procedure a vector of variables may be used. For example \code{resp
  = c(var1, var2, var3)}.}

\item{block}{Defaults to \code{NULL}. In this case, a randomized complete
block design is considered. If block is informed, then a resolvable
alpha-lattice design (Patterson and Williams, 1976) is employed.
\strong{All effects, except the error, are assumed to be fixed.}}

\item{mresp}{The new maximum value after rescaling the response variable. By
default, all variables in \code{resp} are rescaled so that de maximum value
is 100 and the minimum value is 0 (i.e., \code{mresp = 100}). It must be a
numeric vector of the same length of \code{resp} if rescaling is assumed to
be different across variables, e.g., if for the first variable smaller
values are better and for the second one, higher values are better, then
\code{mresp = c(0, 100)} must be used. Numeric value of length 1 will be
recycled with a warning message.}

\item{wresp}{The weight for the response variable(s) for computing the WAASBY
index. By default, all variables in \code{resp} have equal weights for mean
performance and stability (i.e., \code{wresp = 50}). It must be a numeric
vector of the same length of \code{resp} to assign different weights across
variables, e.g., if for the first variable equal weights for mean
performance and stability are assumed and for the second one, a higher
weight for mean performance (e.g. 65) is assumed, then \code{wresp = c(50,
  65)} must be used. Numeric value of length 1 will be recycled with a
warning message.}

\item{prob}{The p-value for considering an interaction principal component
axis significant.}

\item{naxis}{The number of IPCAs to be used for computing the WAAS index.
Default is \code{NULL} (Significant IPCAs are used). If values are
informed, the number of IPCAS will be used independently on its
significance. Note that if two or more variables are included in
\code{resp}, then \code{naxis} must be a vector.}

\item{ind_anova}{Logical argument set to \code{FALSE}. If \code{TRUE} an
within-environment ANOVA is performed.}

\item{verbose}{Logical argument. If \code{verbose = FALSE} the code is run
silently.}
}
\value{
An object of class \code{waas} with the following items for each
variable:
\itemize{
\item \strong{individual} A within-environments ANOVA considering a fixed-effect
model.
\item \strong{model} A data frame with the response variable, the scores of all
Principal Components, the estimates of Weighted Average of Absolute Scores,
and WAASY (the index that consider the weights for stability and productivity
in the genotype ranking.
\item \strong{MeansGxE} The means of genotypes in the environments
\item \strong{PCA} Principal Component Analysis.
\item \strong{anova} Joint analysis of variance for the main effects and
Principal Component analysis of the interaction effect.
\item \strong{Details} A list summarizing the results. The following information
are showed. \code{WgtResponse}, the weight for the response variable in
estimating WAASB, \code{WgtWAAS} the weight for stability, \code{Ngen} the
number of genotypes, \code{Nenv} the number of environments, \code{OVmean}
the overall mean, \code{Min} the minimum observed (returning the genotype and
environment), \code{Max} the maximum observed, \code{Max} the maximum
observed, \code{MinENV} the environment with the lower mean, \code{MaxENV}
the environment with the larger mean observed, \code{MinGEN} the genotype
with the lower mean, \code{MaxGEN} the genotype with the larger.
\item \strong{augment:} Information about each observation in the dataset. This
includes predicted values in the \code{fitted} column, residuals in the
\code{resid} column, standardized residuals in the \code{stdres} column,
the diagonal of the 'hat' matrix in the \code{hat}, and standard errors for
the fitted values in the \code{se.fit} column.
\item \strong{probint} The p-value for the genotype-vs-environment interaction.
}
}
\description{
Compute the Weighted Average of Absolute Scores for AMMI analysis (Olivoto et
al., 2019).
}
\details{
This function compute the weighted average of absolute scores, estimated as
follows:
\loadmathjax
\mjsdeqn{WAAS_i = \sum_{k = 1}^{p} |IPCA_{ik} \times EP_k|/ \sum_{k =
1}^{p}EP_k}

where \mjseqn{WAAS_i} is the weighted average of absolute scores of the
\emph{i}th genotype; \mjseqn{IPCA_{ik}} is the score of the \emph{i}th genotype
in the \emph{k}th IPCA; and \mjseqn{EP_k} is the explained variance of the \emph{k}th
IPCA for \emph{k = 1,2,..,p}, considering \emph{p} the number of significant
PCAs, or a declared number of PCAs. For example if \code{prob = 0.05}, all
axis that are significant considering this probability level are used. The
number of axis can be also informed by declaring \code{naxis = x}. This will
override the number of significant axes according to the argument code{prob}.
}
\examples{
\donttest{
library(metan)
#===============================================================#
# Example 1: Analyzing all numeric variables considering p-value#
# <= 0.05 to compute the WAAS.                                  #
#===============================================================#
model <- waas(data_ge,
             env = ENV,
             gen = GEN,
             rep = REP,
             resp = everything())
# Residual plot (first variable)
plot(model)

# Get the WAAS index
get_model_data(model, "WAAS")

# Plot WAAS and response variable
plot_scores(model, type = 3)


#===============================================================#
# Example 2: Declaring the number of axis to be used for        #
# computing WAAS and assigning a larger weight for the response #
# variable when computing the WAASBY index.                     #
#===============================================================#

model2 <- waas(data_ge,
               env = ENV,
               gen = GEN,
               rep = REP,
               resp = everything(),
               naxis = 1, # Only to compare with PC1
               wresp = 60)
# Get the WAAS index (it will be |PC1|)
get_model_data(model2)

# Get values for IPCA1
get_model_data(model2, "PC1")


#===============================================================#
# Example 3: Analyzing GY and HM assuming a random-effect model.#
# Smaller values for HM and higher values for GY are better.    #
# To estimate WAASBY, higher weight for the GY (60\%) and lower  #
# weight for HM (40\%) are considered for mean performance.      #
#===============================================================#

model3 <- waas(data_ge,
               env = ENV,
               gen = GEN,
               rep = REP,
               resp = c(GY, HM),
               mresp = c(100, 0),
               wresp = c(60, 40))


# Get the ranks for the WAASY index
get_model_data(model3, what = "OrWAASY")
}

}
\references{
Olivoto, T., A.D.C. L{\'{u}}cio, J.A.G. da silva, V.S. Marchioro,
V.Q. de Souza, and E. Jost. 2019a. Mean performance and stability in
multi-environment trials I: Combining features of AMMI and BLUP techniques.
Agron. J. 111:2949-2960. \doi{10.2134/agronj2019.03.0220}
}
\seealso{
\code{\link{waas_means}} \code{\link{waasb}} \code{\link{get_model_data}}
}
\author{
Tiago Olivoto \email{tiagoolivoto@gmail.com}
}
