\name{MNARargument}
\alias{MNARargument}

\title{
Function providing modified arguments for imputation of Missing Not At Random (MNAR) outcomes using \code{\link[mice:mice]{mice}()} function of the '\code{\link{mice}}' package
}
\description{
Imputation models for Missing Not At Random (MNAR) binary or continuous outcomes develloped in this package use sample selection models. It is necessary, inside the imputation model, to specify a selection (i.e. missing data mechanism) and an outcome equation. The previous could be the model of interest (i.e. the post-imputation analysis model). 

\code{MNARargument} adaptes \code{\link[mice:mice]{mice}()} arguments:
\enumerate{
\item \code{data}: Indicator of MNAR outcome missingness is included
\item \code{method}: For the MNAR outcome (\code{varMNAR}), MNAR imputation model is specified
\item \code{predictorMatrix} is modified to include MNAR indicator of missingness in other variable imputation model
}
Finally two new arguments are provided: \code{JointModelEq}, defining selection and outcome equation of the sample selection model;  and \code{control} only for internal use.

The procedure is the following:
\enumerate{
\item Use \code{\link[miceMNAR:generate_JointModelEq]{generate_JointModelEq}()} to construct an empty matrix of variable names allowing to specify selection and outcome equation
\item Fulfill the previous empty matrix adequately to selection and outcome equation specification of the sample selection model
\item Generate an object using \code{MNARargument()} function
\item Include in the \code{\link[mice:mice]{mice}()} function the five arguments of the previous object generated by \code{MNARargument()} 
}
}
\usage{
MNARargument(data, method = NULL, predictorMatrix = NULL, varMNAR, JointModelEq = NULL)
}

\arguments{
  \item{data}{
The dataset used for classical \code{\link[mice:mice]{mice}()} and additional variables necessary for MNAR imputation models.
}
  \item{method}{
The \code{\link[mice:mice]{mice}()} \code{method} argument.
}
  \item{predictorMatrix}{
The \code{\link[mice:mice]{mice}()} \code{predictorMatrix} argument.
}
  \item{varMNAR}{
The name of MNAR outcome to be imputed.
}
  \item{JointModelEq}{
Matrix indicating variables included in selection and outcome equations of MNAR outcome imputation models.
}
}
\details{
Be careful to not define the same selection and outcome equations for MNAR imputation models. A constraint of the sample selection model implies the inclusion of different sets of covariates, which may or not be nested in the selection equation and the outcome equation, to avoid collinearity issues. It has been recommended to include at least a supplementary variable in the selection equation. This variable should be known to be unlinked directly to the outcome.
}
\value{
  \item{data_mod}{Modified dataset including indicator of missingness for MNAR outcomes. Indicators of missingness are coded as "ind_" adding the name of MNAR outcomes.}
  \item{method}{Modified \code{\link[mice:mice]{mice}()} \code{method} argument using \code{\link[miceMNAR:mice.impute.hecknorm]{mice.impute.hecknorm}()} and \code{\link[miceMNAR:mice.impute.heckprob]{mice.impute.heckprob}()} as imputation methods respectively for continuous and binary outcomes.}
  \item{predictorMatrix}{Modified \code{\link[mice:mice]{mice}()} \code{predictorMatrix} argument including indicator of MNAR outcomes missingness as predictors for MAR covariates.}
  \item{JointModelEq}{For internal use: Modified \code{JointModelEq} entry argument.}
  \item{control}{For internal use: MNAR outcomes.}
}
\references{
Galimard, J.E., Chevret, S., Curis, E., and Resche-Rigon, M. (2018). Heckman imputation models for binary or continuous MNAR missing outcomes and MAR missing predictors. BMC Medical Research Methodology (In press).
%
Galimard, J.-E., Chevret, S., Protopopescu, C., and Resche-Rigon, M. (2016) A multiple imputation approach for MNAR mechanisms compatible with Heckman's model. Statistics In  Medicine, 35: 2907-2920. doi:10.1002/sim.6902.
}

\author{
Jacques-Emmanuel Galimard
}

\note{For MNAR continuous outcome, the Heckman's one step estimator is selected by default. However, the two-step estimator is available using \code{\link[miceMNAR:mice.impute.hecknorm2step]{mice.impute.hecknorm2step}()}. To use it, the \code{method} argument should be modified before inclusion in the \code{\link[mice:mice]{mice}()} function.
}

\section{Warning}{
This package is only validated for the imputation of MNAR outcome. However, it is implemented to impute several MNAR variables in the same process. Such implementation must be realised carefully.
}

\seealso{
\code{\link{mice}}
\code{\link[GJRM:copulaSampleSel]{copulaSampleSel}}
\code{\link[GJRM:copulaSampleSel]{SemiParBIV}}
\code{\link[GJRM:hiv]{hiv}}
\code{\link[sampleSelection:selection]{selection}}
\code{\link[miceMNAR:generate_JointModelEq]{generate_JointModelEq}}
}
\examples{

require(GJRM)
require(mvtnorm)
require(pbivnorm)
require(sampleSelection)

# Import dataset with a suspected MNAR mechanism
data("hiv") 

# We select only one region (lusuka) and 5 variables
lusuka <- hiv[hiv$region==5,c("hiv", "age", "marital", "condom", "smoke")]

# Categorical variables have to be recoded as factor
lusuka$hiv <- as.factor(lusuka$hiv)

#############################################
#### Missing data only on a binary outcome ##
#############################################

# Specify a selection (missing data mechanism) and an outcome equation (analyse model)

# Generate an empty matrix

JointModelEq <- generate_JointModelEq(data=lusuka,varMNAR = "hiv")

# Fill in with 1 for variable included in equations
JointModelEq[,"hiv_var_sel"] <- c(0,1,1,1,1)
JointModelEq[,"hiv_var_out"] <- c(0,1,1,1,0)

# Generation of argument for MNAR imputation model in "mice()" function
arg <- MNARargument(data=lusuka,varMNAR="hiv",JointModelEq=JointModelEq)

# Imputation using mice() function
# Values returned have to be included in the "mice()" function as argument:

imputation <- mice(data = arg$data_mod,
                 method = arg$method,
                 predictorMatrix = arg$predictorMatrix,
                 JointModelEq=arg$JointModelEq,
                 control=arg$control,
                 maxit=1,m=5)

# Because of missing data only on one variable, fix maxit=1

# Estimation on each imputed dataset and pooling               
analysis <- with(imputation,glm(hiv~age+condom+marital,family=binomial(link="probit")))
result <- pool(analysis)
summary(result)

##########################################################
#### Missing data on a binary outcome and one covariate ##
##########################################################

# Generate missing values on the variable "condom" 
# According to a MAR mechanism using a probit model
prob <- pnorm((35.5-lusuka$age)/10.74) # Depending on "age"
lusuka$condom[rbinom(nrow(lusuka),size=1, prob=prob)==0] <- NA

JointModelEq <- generate_JointModelEq(data=lusuka,varMNAR = c("hiv"))
JointModelEq[,"hiv_var_sel"] <- c(0,1,1,1,1)
JointModelEq[,"hiv_var_out"] <- c(0,1,1,1,0)

arg <- MNARargument(data=lusuka,varMNAR=c("hiv"),JointModelEq=JointModelEq)

\dontrun{# Imputation using mice function
imputation <- mice(data = arg$data_mod,
                 method = arg$method,
                 predictorMatrix = arg$predictorMatrix,
                 JointModelEq=arg$JointModelEq,
                 control=arg$control, 
                 maxit=5,m=5)

# As classically, estimation on each imputed datasets and pooling               
analysis <- with(imputation,glm(hiv~age+condom+marital,family=binomial(link="probit")))
result <- pool(analysis)
summary(result)}

#################################################
#### Missing data only on a continuous outcome ##
#################################################

# Generation of a simulated dataset with MNAR mechanism on a continuous outcome

X1 <- rnorm(500,0,1)
X2 <- rbinom(500,1,0.5)
X3 <- rnorm(500,1,0.5)
  
errors <- rmvnorm(500,mean=c(0,0),sigma=matrix(c(1,0.3,0.3,1),nrow=2,byrow=TRUE))

Y <- X1+X2+errors[,1]
Ry <- ifelse(0.66+1*X1-0.5*X2+X3+errors[,2]>0,1,0)

Y[Ry==0] <- NA
  
simul_data <- data.frame(Y,X1,X2,X3)

JointModelEq <- generate_JointModelEq(data=simul_data,varMNAR = "Y")

JointModelEq[,"Y_var_sel"] <- c(0,1,1,1)
JointModelEq[,"Y_var_out"] <- c(0,1,1,0)

arg <- MNARargument(data=simul_data,varMNAR="Y",JointModelEq=JointModelEq)

imputation2 <- mice(data = arg$data_mod,
                 method = arg$method,
                 predictorMatrix = arg$predictorMatrix,
                 JointModelEq=arg$JointModelEq,
                 control=arg$control,
                 maxit=1,m=5)

analysis2 <- with(imputation,lm(Y~X1+X2+X3))
result2 <- pool(analysis2)
summary(result2)

#############################
## Using 2-step estimation ##
#############################

arg <- MNARargument(data=simul_data,varMNAR="Y",JointModelEq=JointModelEq)
arg$method["Y"] <- "hecknorm2step"

\dontrun{imputation3 <- mice(data = arg$data_mod,
                 method = arg$method,
                 predictorMatrix = arg$predictorMatrix,
                 JointModelEq=arg$JointModelEq,
                 control=arg$control,
                 maxit=1,m=5)

analysis3 <- with(imputation3,lm(Y~X1+X2+X3))
result3 <- pool(analysis3)
summary(result3)}
}
