% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MP.R
\name{MP}
\alias{LPA}
\alias{MP}
\title{Moving Particles}
\usage{
MP(dimension, lsf, N = 100, N.batch = 1, p, q, lower.tail = TRUE,
  Niter_1fold, alpha = 0.05, compute_confidence = FALSE, verbose = 0,
  chi2 = FALSE, breaks = N.batch/5, ...)
}
\arguments{
\item{dimension}{the dimension of the input space.}

\item{lsf}{the function defining the RV of interest Y = lsf(X).}

\item{N}{the total number of particles,}

\item{N.batch}{the number of parallel batches for the algorithm. Each batch will then
have \code{N/N.batch} particles. Typically this could be \code{detectCores()} or some
other machine-derived parameters.}

\item{p}{a given probability to estimate the corresponding quantile (as in qxxxx functions).}

\item{q}{a given quantile to estimate the corresponding probability (as in pxxxx functions).}

\item{lower.tail}{as for pxxxx functions, TRUE for estimating P(lsf(X) < q), FALSE
for P(lsf(X) > q).}

\item{Niter_1fold}{a function = fun(N) giving the deterministic number of iterations
for the first pass.}

\item{alpha}{when using default \code{Niter_1fold} function, this is the risk not to have
simulated enough samples to produce a quantile estimator.}

\item{compute_confidence}{if \code{TRUE}, the algorithm runs a little bit longer to produces
a 95\% interval on the quantile estimator.}

\item{verbose}{to control level of print (either 0, or 1, or 2).}

\item{chi2}{for a chi2 test on the number of events.}

\item{breaks}{for the final histogram is \code{chi2 == TRUE}.}

\item{...}{further arguments past to \code{\link{IRW}}.}
}
\value{
An object of class \code{list} containing the outputs described below:
\item{p}{the estimated probability or the reference for the quantile estimate.}
\item{q}{the estimated quantile or the reference for the probability estimate.}
\item{ecdf_MP}{the empirical cdf.}
\item{L_max}{the farthest state reached by the random process. Validity range
for the \code{ecdf_MP} is then (-Inf, L_max] or [L_max, Inf).}
\item{times}{the \emph{times} of the random process.}
\item{Ncall}{the total number of calls to the \code{lsf}.}
\item{particles}{the \code{N} particles in their final state.}
\item{LSF_particles}{the value of the \code{lsf(particles)}.}
\item{moves}{a vector containing the number of moves for each batch.}
\item{p_int}{a 95\% confidence intervalle on the probability estimate.}
\item{q_int}{a 95\% confidence intervall on the quantile estimate.}
}
\description{
This function runs the Moving Particles algorithm for estimating extreme probability
and quantile.
}
\details{
\code{MP} is a wrap up of \code{\link{IRW}} for probability and quantile estimation.
 By construction, the several calls to \code{\link{IRW}} are parallel (\pkg{foreach})
and so is the algorithm. Especially, with \code{N.batch=1}, this is the Last Particle
Algorithm, which is a specific version of \code{\link{SubsetSimulation}} with \code{p_0 = 1-1/N}.
However, note that this algorithm not only gives a quantile or a probability estimate
but also an estimate of the whole cdf until the given threshold \code{q}.

The probability estimator only requires to generate several random walks as it is the estimation
of the parameter of a Poisson random variable. The quantile estimator is a little bit more complicated
and requires a 2-passes algorithm. It is thus not exactly fully parallel as cluster/cores have to
communicate after the first pass. During the first pass, particles are moved a given number of
times, during the second pass particles are moved until the farthest event reach during the first
pass. Hence, the random process is completely simulated until this given state.

For an easy user experiment, all the parameters are defined by default with the optimised values
as described in the reference paper (see References below) and a typical use will only specify
\code{N} and \code{N.batch}.
}
\note{
The \code{alpha} parameter is set to 0.05 by default. Indeed it should not be
set too small as it is defined approximating the Poisson distribution with the Gaussian one.
However if no estimate is produce then the algorithm can be restarted for the few missing events.
In any cases, setting \code{Niter_1fold = -N/N.batch*log(p)} gives 100\% chances to produces
a quantile estimator.
}
\examples{
\dontrun{
# Estimate some probability and quantile with the parabolic lsf
p.est <- MP(2, kiureghian, N = 100, q = 0) # estimate P(lsf(X) < 0)
p.est <- MP(2, kiureghian, N = 100, q = 7.8, lower.tail = FALSE) # estimate P(lsf(X) > 7.8)

q.est <- MP(2, kiureghian, N = 100, p = 1e-3) # estimate q such that P(lsf(X) < q) = 1e-3
q.est <- MP(2, kiureghian, N = 100, p = 1e-3, lower.tail = FALSE) # estimate q such
# that P(lsf(X) > q) = 1e-3


# plot the empirical cdf
plot(xplot <- seq(-3, p.est$L_max, l = 100), sapply(xplot, p.est$ecdf_MP))

# check validity range
p.est$ecdf_MP(p.est$L_max - 1)
# this example will fail because the quantile is greater than the limit
tryCatch({
   p.est$ecdf_MP(p.est$L_max + 0.1)},
   error = function(cond) message(cond))

# Run in parallel
library(doParallel)
registerDoParallel()
p.est <- MP(2, kiureghian, N = 100, q = 0, N.batch = getDoParWorkers())
}

}
\author{
Clement WALTER \email{clement.walter@cea.fr}
}
\references{
\itemize{
   \item A. Guyader, N. Hengartner and E. Matzner-Lober:\cr
    \emph{Simulation and estimation of extreme quantiles and extreme
    probabilities}\cr
    Applied Mathematics \& Optimization, 64(2), 171-196.\cr
   \item C. Walter:\cr
   \emph{Moving Particles: a parallel optimal Multilevel Splitting
   method with application in quantiles estimation and meta-model
   based algorithms}\cr
   Structural Safety, 55, 10-25.\cr
   \item E. Simonnet:\cr
   \emph{Combinatorial analysis of the adaptive last particle method}\cr
   Statistics and Computing, 1-20.\cr
 }
}
\seealso{
\code{\link{SubsetSimulation}}
\code{\link{MonteCarlo}}
\code{\link{IRW}}
}

