% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mkinpredict.R
\name{mkinpredict}
\alias{mkinpredict}
\alias{mkinpredict.mkinmod}
\alias{mkinpredict.mkinfit}
\title{Produce predictions from a kinetic model using specific parameters}
\usage{
mkinpredict(x, odeparms, odeini, outtimes, ...)

\method{mkinpredict}{mkinmod}(
  x,
  odeparms = c(k_parent_sink = 0.1),
  odeini = c(parent = 100),
  outtimes = seq(0, 120, by = 0.1),
  solution_type = "deSolve",
  use_compiled = "auto",
  method.ode = "lsoda",
  atol = 1e-08,
  rtol = 1e-10,
  map_output = TRUE,
  na_stop = TRUE,
  ...
)

\method{mkinpredict}{mkinfit}(
  x,
  odeparms = x$bparms.ode,
  odeini = x$bparms.state,
  outtimes = seq(0, 120, by = 0.1),
  solution_type = "deSolve",
  use_compiled = "auto",
  method.ode = "lsoda",
  atol = 1e-08,
  rtol = 1e-10,
  map_output = TRUE,
  ...
)
}
\arguments{
\item{x}{A kinetic model as produced by \code{\link{mkinmod}}, or a kinetic
fit as fitted by \code{\link{mkinfit}}. In the latter case, the fitted
parameters are used for the prediction.}

\item{odeparms}{A numeric vector specifying the parameters used in the
kinetic model, which is generally defined as a set of ordinary
differential equations.}

\item{odeini}{A numeric vector containing the initial values of the state
variables of the model. Note that the state variables can differ from the
observed variables, for example in the case of the SFORB model.}

\item{outtimes}{A numeric vector specifying the time points for which model
predictions should be generated.}

\item{\dots}{Further arguments passed to the ode solver in case such a
solver is used.}

\item{solution_type}{The method that should be used for producing the
predictions. This should generally be "analytical" if there is only one
observed variable, and usually "deSolve" in the case of several observed
variables. The third possibility "eigen" is faster but not applicable to
some models e.g.  using FOMC for the parent compound.}

\item{use_compiled}{If set to \code{FALSE}, no compiled version of the
\code{\link{mkinmod}} model is used, even if is present.}

\item{method.ode}{The solution method passed via \code{\link{mkinpredict}}
to \code{\link{ode}} in case the solution type is "deSolve". The default
"lsoda" is performant, but sometimes fails to converge.}

\item{atol}{Absolute error tolerance, passed to \code{\link{ode}}. Default
is 1e-8, lower than in \code{\link{lsoda}}.}

\item{rtol}{Absolute error tolerance, passed to \code{\link{ode}}. Default
is 1e-10, much lower than in \code{\link{lsoda}}.}

\item{map_output}{Boolean to specify if the output should list values for
the observed variables (default) or for all state variables (if set to
FALSE). Setting this to FALSE has no effect for analytical solutions,
as these always return mapped output.}

\item{na_stop}{Should it be an error if deSolve::ode returns NaN values}
}
\value{
A matrix with the numeric solution in wide format
}
\description{
This function produces a time series for all the observed variables in a
kinetic model as specified by \code{\link{mkinmod}}, using a specific set of
kinetic parameters and initial values for the state variables.
}
\examples{

SFO <- mkinmod(degradinol = mkinsub("SFO"))
# Compare solution types
mkinpredict(SFO, c(k_degradinol = 0.3), c(degradinol = 100), 0:20,
      solution_type = "analytical")
mkinpredict(SFO, c(k_degradinol = 0.3), c(degradinol = 100), 0:20,
      solution_type = "deSolve")
mkinpredict(SFO, c(k_degradinol = 0.3), c(degradinol = 100), 0:20,
      solution_type = "deSolve", use_compiled = FALSE)
mkinpredict(SFO, c(k_degradinol = 0.3), c(degradinol = 100), 0:20,
      solution_type = "eigen")

# Compare integration methods to analytical solution
mkinpredict(SFO, c(k_degradinol = 0.3), c(degradinol = 100), 0:20,
      solution_type = "analytical")[21,]
mkinpredict(SFO, c(k_degradinol = 0.3), c(degradinol = 100), 0:20,
      method = "lsoda")[21,]
mkinpredict(SFO, c(k_degradinol = 0.3), c(degradinol = 100), 0:20,
      method = "ode45")[21,]
mkinpredict(SFO, c(k_degradinol = 0.3), c(degradinol = 100), 0:20,
      method = "rk4")[21,]
# rk4 is not as precise here

# The number of output times used to make a lot of difference until the
# default for atol was adjusted
mkinpredict(SFO, c(k_degradinol = 0.3), c(degradinol = 100),
      seq(0, 20, by = 0.1))[201,]
mkinpredict(SFO, c(k_degradinol = 0.3), c(degradinol = 100),
      seq(0, 20, by = 0.01))[2001,]

# Comparison of the performance of solution types
SFO_SFO = mkinmod(parent = list(type = "SFO", to = "m1"),
                  m1 = list(type = "SFO"), use_of_ff = "max")
if(require(rbenchmark)) {
  benchmark(replications = 10, order = "relative", columns = c("test", "relative", "elapsed"),
    eigen = mkinpredict(SFO_SFO,
      c(k_parent = 0.15, f_parent_to_m1 = 0.5, k_m1 = 0.01),
      c(parent = 100, m1 = 0), seq(0, 20, by = 0.1),
      solution_type = "eigen")[201,],
    deSolve_compiled = mkinpredict(SFO_SFO,
      c(k_parent = 0.15, f_parent_to_m1 = 0.5, k_m1 = 0.01),
      c(parent = 100, m1 = 0), seq(0, 20, by = 0.1),
      solution_type = "deSolve")[201,],
    deSolve = mkinpredict(SFO_SFO,
      c(k_parent = 0.15, f_parent_to_m1 = 0.5, k_m1 = 0.01),
      c(parent = 100, m1 = 0), seq(0, 20, by = 0.1),
      solution_type = "deSolve", use_compiled = FALSE)[201,],
    analytical = mkinpredict(SFO_SFO,
      c(k_parent = 0.15, f_parent_to_m1 = 0.5, k_m1 = 0.01),
      c(parent = 100, m1 = 0), seq(0, 20, by = 0.1),
      solution_type = "analytical", use_compiled = FALSE)[201,])
}

\dontrun{
  # Predict from a fitted model
  f <- mkinfit(SFO_SFO, FOCUS_2006_C, quiet = TRUE)
  f <- mkinfit(SFO_SFO, FOCUS_2006_C, quiet = TRUE, solution_type = "deSolve")
  head(mkinpredict(f))
}

}
\author{
Johannes Ranke
}
