% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/iea_model.R
\name{iea_model}
\alias{iea_model}
\title{Independent edge assignment model for multigraphs}
\usage{
iea_model(
  adj,
  type = "multigraph",
  model = "IEAS",
  K = 0,
  apx = FALSE,
  p.seq = NULL
)
}
\arguments{
\item{adj}{matrix of integers representing graph adjacency matrix.}

\item{type}{equals \code{'graph'} if adjacency matrix is for graphs (default),
equals \code{'multigraph'} if it is the equivalence of the adjacency matrix for multigraphs
(with matrix diagonal representing loops double counted).}

\item{model}{character string representing which IEA model: either \code{'IEAS'} (default) or \code{'ISA'}.}

\item{K}{upper limit for the complexity statistics \emph{R(k)},
\emph{k}=(0,1,...,\code{K}), representing the sequence of frequencies of edge multiplicities
(default is maximum observed in adjacency matrix).}

\item{apx}{logical (default = \code{'FALSE'}). if \code{'TRUE'}, the IEA model is used to approximate
the statistics under the random stub matching model given observed degree sequence.}

\item{p.seq}{if \code{model = ISA} and \code{apx = FALSE}, then specify this numerical vector of
stub assignment probabilities.}
}
\value{
\item{nr.multigraphs}{Number of unique multigraphs possible.}
\item{M}{Summary and interval estimates for \emph{number of loops} (\code{M1}) and
\emph{number of multiple edges} (\code{M2}).}
\item{R}{Summary and interval estimates for frequencies of edge multiplicities
\code{R1},\code{R2},...,\code{RK}, where \code{K} is a function argument.}
}
\description{
Summary of estimated statistics for analyzing
global structure of random multigraphs under the independent edge assignment model
given observed adjacency matrix.

Two versions of the IEA model are implemented,
both of which can be used to approximate the RSM model:
\cr
  1. independent edge assignment of stubs (IEAS) given an edge probability sequence\cr
  2. independent stub assignment (ISA) given a stub probability sequence \cr
}
\details{
When using the IEAS model: \cr If the IEAS model is used
as an approximation to the RSM model, then the edge assignment probabilities are estimated
by using the observed degree sequence. Otherwise, the edge assignment probabilities are
estimated by using the observed edge multiplicities  (maximum likelihood estimates). \cr

When using the ISA model: \cr If the ISA model is used
as an approximation to the RSM model, then the stub assignment probabilities are estimated by using
the observed degree sequence over \emph{2m}.  Otherwise, a sequence containing the stub assignment
probabilities (for example based on prior belief) should be given as argument \code{p.seq}.
}
\examples{
# Adjacency matrix of a small graph on 3 nodes
A <-  matrix(c(1, 1, 0,
               1, 2, 2,
               0, 2, 0),
             nrow = 3, ncol = 3)

# When the IEAS model is used
iea_model(adj = A , type = 'graph', model = 'IEAS', K = 0, apx = FALSE)

# When the IEAS model is used to approximate the RSM model
iea_model(adj = A , type = 'graph', model = 'IEAS', K = 0, apx = TRUE)

# When the ISA model is used to approximate the RSM model
iea_model(adj = A , type = 'graph', model = 'ISA', K = 0, apx = TRUE)

# When the ISA model is used with a pre-specified stub assignment probabilities
iea_model(adj = A , type = 'graph', model = 'ISA', K = 0, apx = FALSE, p.seq = c(1/3, 1/3, 1/3))
}
\references{
Shafie, T. (2015). A Multigraph Approach to Social Network Analysis. \emph{Journal of Social Structure}, 16.
\cr

Shafie, T. (2016). Analyzing Local and Global Properties of Multigraphs. \emph{The Journal of Mathematical Sociology}, 40(4), 239-264.
}
\seealso{
\code{\link{get_degree_seq}}, \code{\link{get_edge_multip_seq}}, \code{\link{iea_model}}
}
\author{
Termeh Shafie
}
