% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/context.R
\name{request}
\alias{request}
\alias{request_signal}
\title{Request over Context (RPC Client for Req/Rep Protocol)}
\usage{
request(
  context,
  data,
  send_mode = c("serial", "raw"),
  recv_mode = c("serial", "character", "complex", "double", "integer", "logical",
    "numeric", "raw", "string"),
  timeout = NULL
)

request_signal(
  context,
  data,
  send_mode = c("serial", "raw"),
  recv_mode = c("serial", "character", "complex", "double", "integer", "logical",
    "numeric", "raw", "string"),
  timeout = NULL,
  cv
)
}
\arguments{
\item{context}{a Context.}

\item{data}{an object (if send_mode = 'raw', a vector).}

\item{send_mode}{[default 'serial'] to send serialised R objects, or 'raw' to
send atomic vectors of any type as a raw byte vector. Use 'serial' to
ensure perfect reproducibility within R, although 'raw' must be used when
interfacing with external applications which do not understand R
serialisation. Alternatively, for performance, specify an integer position
in the vector of choices i.e. 1L for 'serial' or 2L for 'raw'.}

\item{recv_mode}{[default 'serial'] mode of vector to be received - one of
'serial', 'character', 'complex', 'double', 'integer', 'logical',
'numeric', 'raw', or 'string'. The default 'serial' means a serialised R
object, for the other modes, the raw vector received will be converted
into the respective mode. Note that 'string' is defined here as a character
scalar and is a faster alternative to 'character' for receiving a single
string. Alternatively, for performance, specify an integer position in
the vector of choices e.g. 1L for 'serial', 2L for 'character' etc.}

\item{timeout}{[default NULL] integer value in milliseconds or NULL, which
applies a socket-specific default, usually the same as no timeout.}

\item{cv}{\strong{For the signalling version}: a 'conditionVariable' to
signal when the async receive is complete.}
}
\value{
A 'recvAio' (object of class 'recvAio') (invisibly).
}
\description{
Implements a caller/client for the req node of the req/rep protocol. Sends
    data to the rep node (executor/server) and returns an Aio, which can be
    called for the value when required.

A signalling version of the function takes a 'conditionVariable' as an
    additional argument and signals it when the async receive is complete.
}
\details{
Sending the request and receiving the result are both performed async,
    hence the function will return immediately with a 'recvAio' object. Access
    the return value at \code{$data}.

    This is designed so that the process on the server can run concurrently
    without blocking the client.

    Optionally use \code{\link{call_aio}} on the 'recvAio' to call (and wait
    for) the result.

    If an error occured in the server process, a nul byte \code{00} will be
    received. This allows an error to be easily distinguished from a NULL
    return value. \code{\link{is_nul_byte}} can be used to test for a nul byte.

    It is recommended to use a new context for each request to ensure
    consistent state tracking. For safety, the context used for the request
    is closed when all references to the returned 'recvAio' are removed and
    the object is garbage collected.

\strong{For the signalling version}: when the receive is complete,
    the supplied 'conditionVariable' is signalled by incrementing its value
    by 1. This happens asynchronously and independently of the R execution
    thread.
}
\examples{
req <- socket("req", listen = "tcp://127.0.0.1:6546")
rep <- socket("rep", dial = "tcp://127.0.0.1:6546")

# works if req and rep are running in parallel in different processes
reply(.context(rep), execute = function(x) x + 1, timeout = 50)
aio <- request(.context(req), data = 2022)
aio$data

close(req)
close(rep)

# Signalling a condition variable

req <- socket("req", listen = "tcp://127.0.0.1:6546")
ctxq <- context(req)
cv <- cv()
aio <- request_signal(ctxq, data = 2022, cv = cv)
until(cv, 10L)
close(req)

# The following should be run in another process
# rep <- socket("rep", dial = "tcp://127.0.0.1:6546")
# ctxp <- context(rep)
# reply(ctxp, execute = function(x) x + 1)
# close(rep)

}
