% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fn.R
\name{fn}
\alias{fn}
\alias{..}
\title{Low-cost anonymous functions}
\usage{
fn(..., ..env = parent.frame())

..(..., ..env = parent.frame())
}
\arguments{
\item{...}{Function declaration, which supports
\link[rlang:quasiquotation]{quasiquotation}.}

\item{..env}{Environment in which to create the function (i.e., the
function’s \link[base:environment]{enclosing environment}).}
}
\value{
A function whose enclosing environment is \code{..env}.
}
\description{
\code{fn()} enables you to create (anonymous) functions, of arbitrary call
signature. It is a drop-in replacement for the usual
\code{function(<arguments>) <body>} invocation, but costs less:
\itemize{
\item It is \strong{shorter}:
\preformatted{
    fn(x, y = 1 ~ x + y)
    function(x, y = 1) x + y}
are equivalent.
\item It is \strong{safer}: by enabling (Tidyverse)
\link[rlang:quasiquotation]{quasiquotation}, \code{fn()} allows you to
\dQuote{burn in} values, which can guard your function from unexpected
scope changes (see \emph{Examples}).
}
To reduce visual noise, \code{..()} is provided as an alias of \code{fn()}.
}
\section{Function declarations}{
 A \emph{function declaration} is an expression
that specifies a function’s arguments and body, as a comma-separated
expression of the form\preformatted{    arg1, arg2, ..., argN ~ body
}

or\preformatted{    arg1, arg2, ..., argN, ~ body
}

(Note in the second form that the body is a one-sided formula. This
distinction is relevant for argument \link[rlang:UQS]{splicing}; see below.)

To the left of \code{~}, you write a conventional function-argument declaration,
just as in \code{function(<arguments>)}: each of \code{arg1}, \code{arg2}, \dots, \code{argN}
is either a bare argument (e.g., \code{x} or \code{...}) or an argument with default
value (e.g., \code{x = 1}). To the right of \code{~}, you write the function body,
i.e., an expression of the arguments.

\subsection{Quasiquotation}{
All parts of a function declaration support (Tidyverse)
\link[rlang:quasiquotation]{quasiquotation}:
\itemize{
\item To unquote values (of arguments or parts of the body), use \code{!!}
or \code{UQ()}:
\preformatted{
    z <- 0
    fn(x, y = !! z ~ x + y)
    fn(x ~ x > !! z)}
\item To unquote argument names (with default value), use \code{:=}
(definition operator):
\preformatted{
    arg <- "y"
    fn(x, !! arg := 0 ~ x + !! as.name(arg))}
\item To splice in a (formal) list of arguments, use \code{!!!} or \code{UQS()}:
\preformatted{
    fn(!!! alist(x, y = 0), ~ x + y)}
(Note that the body, in this case, must be given as a one-sided
formula.)
}
}
}

\examples{
fn(x ~ x + 1)
fn(x, y ~ x + y)
fn(x, y = 2 ~ x + y)
fn(x, y = 1, ... ~ log(x + y, ...))

## to specify '...' in the middle, write '... = '
fn(x, ... = , y ~ log(x + y, ...))

## use one-sided formula for constant functions or commands
fn(~ NA)
fn(~ message("!"))

## unquoting is supported (using `!!` or UQ() from rlang)
zero <- 0
fn(x = UQ(zero) ~ x > !! zero)

## formals and function bodies can also be spliced in
f <- function(x, y) x + y
g <- function(y, x, ...) x - y
frankenstein <- fn(!!! formals(f), ~ !! body(g))
stopifnot(identical(frankenstein, function(x, y) x - y))

## unquoting protects against changes in a function’s scope
x <- "x"
f <- function() x
f_solid <- fn(~ !! x)
# both return the same value of x
f()
f_solid()
# but if the binding `x` is (unwittingly) changed, f() changes ...
x <- sin
f()
# ... while f_solid() remains unaffected
f_solid()

}
\seealso{
\code{\link[=as_fn]{as_fn()}}
}
