% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vuongtest.R
\name{vuongtest}
\alias{vuongtest}
\title{Vuong Tests for Model Comparison}
\usage{
vuongtest(object1, object2, nested = FALSE, adj = "none",
  ll1 = llcont, ll2 = llcont, score1 = NULL, score2 = NULL,
  vc1 = vcov, vc2 = vcov)
}
\arguments{
\item{object1}{a model object}

\item{object2}{a model object}

\item{nested}{if \code{TRUE}, models are assumed to be nested}

\item{adj}{Should an adjusted test statistic be calculated?  Defaults to \dQuote{none}, with possible adjustments being \dQuote{aic} and \dQuote{bic}}

\item{ll1}{an optional function for computing log-likelihood contributions of object1}

\item{ll2}{an optional function for computing log-likelihood contributions of object2}

\item{score1}{an optional function for computing scores of object 1}

\item{score2}{an optional function for computing scores of object 2}

\item{vc1}{an optional function for computing the asymptotic covariance matrix of the object1 parameters}

\item{vc2}{an optional function for computing the asymptotic covariance matrix of the object2 parameters}
}
\value{
an object of class \code{vuongtest} containing test results.
}
\description{
Test pairs of models using Vuong's (1989) <DOI:10.2307/1912557> theory.  This includes
a test of model distinguishability and a test of model fit.
}
\details{
For non-nested models, the test of distinguishability indicates whether or
not the models can possibly be distinguished on the basis of the observed
data.  The LRT then indicates whether or not one model fits better
than another.

For nested models (\code{nested=TRUE}), both tests serve as robust
alternatives to the classical likelihood ratio tests.  In this case,
the \code{adj} argument is ignored.

Users should take care to ensure that the two models have
the same dependent variable (or, for lavaan objects, identical
modeled variables), with observations ordered identically within
each model object.  Assuming the same data matrix is used to fit each model,
observation ordering should generally be identical.  There are currently
no checks for this, however.
}
\examples{
\dontrun{
## Count regression comparisons
require(MASS)
house1 <- glm(Freq ~ Infl + Type + Cont, family=poisson, data=housing)
house2 <- glm(Freq ~ Infl + Sat, family=poisson, data=housing)
house3 <- glm(Freq ~ Infl, family=poisson, data=housing)
## house3 is nested within house1 and house2
anova(house3, house1, test="Chisq")
anova(house3, house2, test="Chisq")

## house 2 is not nested in house1, so this test is invalid
anova(house2, house1, test="Chisq")

## Use vuongtest() instead
vuongtest(house2, house1)

## Application to models with different distributional assumptions
require(pscl)
bio1 <- glm(art ~ fem + mar + phd + ment, family=poisson, data=bioChemists)
bio2 <- hurdle(art ~ fem + mar + phd + ment, data=bioChemists)
bio3 <- zeroinfl(art ~ fem + mar + phd + ment, data=bioChemists)
vuongtest(bio2, bio1)
vuongtest(bio3, bio1)
vuongtest(bio1, bio2)
vuongtest(bio1, bio3)
vuongtest(bio3, bio2)

## Application to latent variable models
require(lavaan)
HS.model <- 'visual  =~ x1 + x2 + x3
              textual =~ x4 + x5 + x6
              speed   =~ x7 + x8 + x9 '
fit1 <- cfa(HS.model, data=HolzingerSwineford1939)
fit2 <- cfa(HS.model, data=HolzingerSwineford1939, group="school")
vuongtest(fit1, fit2)

## Supplying custom vcov function
require(lme4)
require(merDeriv)

fm1 <- lmer(Reaction ~ Days + (Days | Subject), sleepstudy, REML=FALSE)
fm2 <- lmer(Reaction ~ Days + (Days || Subject), sleepstudy, REML=FALSE)

vcl <- function(obj) vcov(obj, full=TRUE)
vuongtest(fm1, fm2, vc1=vcl, vc2=vcl, nested=TRUE)

}

}
\references{
Vuong, Q. H. (1989).  Likelihood ratio tests for model selection and non-nested hypotheses.  \emph{Econometrica, 57}, 307-333. <DOI:10.2307/1912557>

Merkle, E. C., You, D., & Preacher, K. (2016). Testing non-nested structural equation models.  \emph{Psychological Methods, 21}, 151-163. <DOI:10.1037/met0000038>
}
\author{
Ed Merkle and Dongjun You
}
