% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/occumb.R
\name{occumb}
\alias{occumb}
\title{Model-fitting function.}
\usage{
occumb(
  formula_phi = ~1,
  formula_theta = ~1,
  formula_psi = ~1,
  formula_phi_shared = ~1,
  formula_theta_shared = ~1,
  formula_psi_shared = ~1,
  prior_prec = 1e-04,
  prior_ulim = 10000,
  data,
  n.chains = 4,
  n.adapt = NULL,
  n.burnin = 10000,
  n.thin = 10,
  n.iter = 20000,
  parallel = FALSE,
  ...
)
}
\arguments{
\item{formula_phi}{A right-hand side formula describing species-specific
effects of sequence relative dominance (\eqn{\phi}).}

\item{formula_theta}{A right-hand side formula describing species-specific
effects of sequence capture probability (\eqn{\theta}).}

\item{formula_psi}{A right-hand side formula describing species-specific
effects of occupancy probability (\eqn{\psi}).}

\item{formula_phi_shared}{A right-hand side formula describing effects of
sequence relative dominance (\eqn{\phi}) that are common across
species. The intercept term is ignored (see Details).}

\item{formula_theta_shared}{A right-hand side formula describing effects of
sequence capture probability (\eqn{\theta}) that are common across
species.
The intercept term is ignored (see Details).}

\item{formula_psi_shared}{A right-hand side formula describing effects of
occupancy probability (\eqn{\psi}) that are common across species.
The intercept term is ignored (see Details).}

\item{prior_prec}{Precision of normal prior distribution for the
community-level average of species-specific parameters and effects
common across species.}

\item{prior_ulim}{Upper limit of uniform prior distribution for the
standard deviation of species-specific parameters.}

\item{data}{A dataset supplied as an \code{\link{occumbData}} class object.}

\item{n.chains}{Number of Markov chains to run.}

\item{n.adapt}{Number of iterations to run in the JAGS adaptive phase.}

\item{n.burnin}{Number of iterations at the beginning of the chain to discard.}

\item{n.thin}{Thinning rate. Must be a positive integer.}

\item{n.iter}{Total number of iterations per chain (including burn-in).}

\item{parallel}{If TRUE, run MCMC chains in parallel on multiple CPU cores.}

\item{...}{Additional arguments passed to \code{\link[jagsUI]{jags}()} function.}
}
\value{
An S4 object of the \code{occumbFit} class containing the results of
the model fitting and the supplied dataset.
}
\description{
\code{occumb()} fits the community site-occupancy model for eDNA
metabarcoding (Fukaya et al. 2022) and returns a model-fit object containing
posterior samples.
}
\details{
\code{occumb()} allows the fitting of a range of community site
occupancy models, including covariates at different levels of the data
generation process.
The most general form of the model can be written as follows (the notation
follows that of the original article; see Fukaya et al. (2022) or
\href{https://fukayak.github.io/occumb/articles/model_specification.html}{the package vignette}).

Sequence read counts:
\deqn{(y_{1jk}, ..., y_{Ijk}) \sim \textrm{Multinomial}((\pi_{1jk}, ...,  \pi_{Ijk}), N_{jk}),}{y[1:I, j, k] ~ Multinomial(pi[1:I, j, k], N[j, k]),}
\deqn{\pi_{ijk} = \frac{u_{ijk}r_{ijk}}{\sum_m u_{mjk}r_{mjk}},}{pi[i, j, k] = (u[i, j, k] * r[i, j, k]) / sum(u[1:I, j, k] * r[1:I, j, k]),}

Relative frequency of species sequences:
\deqn{r_{ijk} \sim \textrm{Gamma}(\phi_{ijk}, 1),}{r[i, j, k] ~ Gamma(phi[i, j, k], 1),}

Capture of species sequences:
\deqn{u_{ijk} \sim \textrm{Bernoulli}(z_{ij}\theta_{ijk}),}{u[i, j, k] ~ Bernoulli(z[i, j] * theta[i, j, k]),}

Site occupancy of species:
\deqn{z_{ij} \sim \textrm{Bernoulli}(\psi_{ij}),}{z[i, j] ~ Bernoulli(psi[i, j]),}
where the variations of \eqn{\phi}{phi}, \eqn{\theta}{theta}, and
\eqn{\psi}{psi} are modeled by specifying model formulas in
\code{formula_phi}, \code{formula_theta}, \code{formula_psi},
\code{formula_phi_shared}, \code{formula_theta_shared}, and
\code{formula_psi_shared.}
Each parameter may have species-specific effects and effects that are common
across species, where the former is specified by \code{formula_phi},
\code{formula_theta}, and \code{formula_psi}, whereas
\code{formula_phi_shared}, \code{formula_theta_shared}, and
\code{formula_psi_shared} specify the latter.
As species-specific intercepts are specified by default, the intercept
terms in \code{formula_phi_shared}, \code{formula_theta_shared}, and
\code{formula_psi_shared} are always ignored.
Covariate terms must be found in the names of the list elements stored
in the \code{spec_cov}, \code{site_cov}, or \code{repl_cov} slots in the dataset
object provided with the \code{data} argument.
Covariates are modeled using the log link function for \eqn{\phi}{phi}
and logit link function for \eqn{\theta}{theta} and \eqn{\psi.}{psi.}

The two arguments, \code{prior_prec} and \code{prior_ulim}, control the
prior distribution of parameters. For the community-level average of
species-specific effects and effects common across species, a normal
prior distribution with a mean of 0 and precision (i.e., the inverse of the
variance) \code{prior_prec} is specified. For the standard deviation of
species-specific effects, a uniform prior distribution with a lower limit of
zero and an upper limit of \code{prior_ulim} is specified. For the correlation
coefficient of species-specific effects, a uniform prior distribution in the
range of \eqn{-}1 to 1 is specified by default.

See \href{https://fukayak.github.io/occumb/articles/model_specification.html}{the package vignette}
for details on the model specifications in \code{occumb()}.

The \code{data} argument requires a dataset object to be generated using
\code{ocumbData()}; see the document of \code{\link{occumbData}()}.

The model is fit using the \code{\link[jagsUI]{jags}()} function of the
\href{https://cran.r-project.org/package=jagsUI}{jagsUI} package, where
Markov chain Monte Carlo (MCMC) methods are used to
obtain posterior samples of the parameters and latent variables.
Arguments \code{n.chains}, \code{n.adapt}, \code{n.burnin}, \code{n.thin},
\code{n.iter}, and \code{parallel} are passed on to arguments of the
same name in the \code{\link[jagsUI]{jags}()} function.
See the document of \href{https://cran.r-project.org/package=jagsUI}{jagsUI}'s
\code{\link[jagsUI]{jags}()} function for details.
A set of random initial values is used to perform an MCMC run.
}
\section{References}{

K. Fukaya, N. I. Kondo, S. S. Matsuzaki and T. Kadoya (2022)
Multispecies site occupancy modelling and study design for spatially
replicated environmental DNA metabarcoding. \emph{Methods in Ecology
and Evolution} \strong{13}:183--193.
\doi{10.1111/2041-210X.13732}
}

\examples{
# Generate the smallest random dataset (2 species * 2 sites * 2 reps)
I <- 2 # Number of species
J <- 2 # Number of sites
K <- 2 # Number of replicates
data <- occumbData(
    y = array(sample.int(I * J * K), dim = c(I, J, K)),
    spec_cov = list(cov1 = rnorm(I)),
    site_cov = list(cov2 = rnorm(J),
                    cov3 = factor(1:J)),
    repl_cov = list(cov4 = matrix(rnorm(J * K), J, K)))

\donttest{
# Fitting a null model (includes only species-specific intercepts)
res0 <- occumb(data = data)

# Add species-specific effects of site covariates in occupancy probabilities
res1 <- occumb(formula_psi = ~ cov2, data = data)        # Continuous covariate
res2 <- occumb(formula_psi = ~ cov3, data = data)        # Categorical covariate
res3 <- occumb(formula_psi = ~ cov2 * cov3, data = data) # Interaction
# Add species covariate in the three parameters
# Note that species covariates are modeled as common effects
res4 <- occumb(formula_phi_shared = ~ cov1, data = data)   # phi
res5 <- occumb(formula_theta_shared = ~ cov1, data = data) # theta
res6 <- occumb(formula_psi_shared = ~ cov1, data = data)   # psi
# Add replicate covariates
# Note that replicate covariates can only be specified for theta and phi
res7 <- occumb(formula_phi = ~ cov4, data = data)   # phi
res8 <- occumb(formula_theta = ~ cov4, data = data) # theta
# Specify the prior distribution and MCMC settings explicitly
res9 <- occumb(data = data, prior_prec = 1E-2, prior_ulim = 1E2,
               n.chains = 1, n.burnin = 1000, n.thin = 1, n.iter = 2000)
res10 <- occumb(data = data, parallel = TRUE, n.cores = 2) # Run MCMC in parallel
}
}
