% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_phylo_persistence.R
\name{plot_phylo_persistence}
\alias{plot_phylo_persistence}
\title{Plot a phylogram to visualize a project prioritization}
\usage{
plot_phylo_persistence(
  x,
  solution,
  n = 1,
  symbol_hjust = 0.007,
  return_data = FALSE
)
}
\arguments{
\item{x}{project prioritization \code{\link{problem}}.}

\item{solution}{\code{\link[base]{data.frame}} or
\code{\link[tibble]{tibble}} table containing the solutions. Here,
rows correspond to different solutions and columns correspond to
different actions. Each column in the argument to \code{solution} should
be named according to a different action in \code{x}.
Cell values indicate if an action is funded in a given solution or not,
and should be either zero or one. Arguments to \code{solution} can
contain additional columns, and they will be ignored.}

\item{n}{\code{integer} solution number to visualize.
Since each row in the argument to \code{solutions} corresponds to a
different solution, this argument should correspond to a row in
the argument to \code{solutions}. Defaults to 1.}

\item{symbol_hjust}{\code{numeric} horizontal adjustment parameter to
manually align the asterisks and dashes in the plot. Defaults to
\code{0.007}. Increasing this parameter will shift the symbols further
right. Please note that this parameter may require some tweaking
to produce visually appealing publication quality plots.}

\item{return_data}{\code{logical} should the underlying data used to create
the plot be returned instead of the plot? Defaults to \code{FALSE}.}
}
\value{
A \code{\link[ggtree]{ggtree}} object, or a
  \code{\link[tidytree]{treedata}} object if \code{return_data} is
  \code{TRUE}.
}
\description{
Create a plot showing a phylogenetic tree (i.e. a "phylogram") to visualize
the probability that phylogenetic branches are expected to persist
into the future under a solution to a project prioritization problem.
}
\details{
This function requires the \pkg{ggtree} (Yu \emph{et al.} 2017).
  Since this package is distributed exclusively
  through \href{https://bioconductor.org}{Bioconductor}, and is not
  available on the
  \href{https://cran.r-project.org/}{Comprehensive R Archive Network},
  please execute the following command to install it:
  \code{source("https://bioconductor.org/biocLite.R");biocLite("ggtree")}.
  If the installation process fails, please consult the package's \href{https://bioconductor.org/packages/release/bioc/html/ggtree.html}{online documentation}.

  In this plot, each phylogenetic branch is colored according to probability
  that it is expected to persist into the future (see Faith 2008).
  Features that directly benefit from at least a single
  completely funded project with a non-zero cost are depicted with an
  asterisk symbol. Additionally, features that indirectly benefit from funded
  projects---because they are associated with partially funded projects that
  have non-zero costs and share actions with at least one completely funded
  project---are depicted with an open circle symbol.
}
\examples{
# set seed for reproducibility
set.seed(500)

# load the ggplot2 R package to customize plots
library(ggplot2)

data(sim_projects, sim_features, sim_actions)

# build problem
p <- problem(sim_projects, sim_actions, sim_features,
             "name", "success", "name", "cost", "name") \%>\%
     add_max_phylo_div_objective(budget = 400, sim_tree) \%>\%
     add_binary_decisions() \%>\%
     add_heuristic_solver(number_solutions = 10)

\donttest{
# solve problem
s <- solve(p)

# plot the first solution
plot(p, s)

# plot the second solution
plot(p, s, n = 2)

# since this function returns a ggplot2 plot object, we can customize the
# appearance of the plot using standard ggplot2 commands!
# for example, we can add a title
plot(p, s) + ggtitle("solution")

# we could also also set the minimum and maximum values in the color ramp to
# correspond to those in the data, rather than being capped at 0 and 1
plot(p, s) +
scale_color_gradientn(name = "Probability of\npersistence",
                      colors = viridisLite::inferno(150, begin = 0,
                                                    end = 0.9,
                                                    direction = -1)) +
ggtitle("solution")

# we could also change the color ramp
plot(p, s) +
scale_color_gradient(name = "Probability of\npersistence",
                     low = "red", high = "black") +
ggtitle("solution")

# we could even hide the legend if desired
plot(p, s) +
scale_color_gradient(name = "Probability of\npersistence",
                     low = "red", high = "black") +
theme(legend.position = "hide") +
ggtitle("solution")

# we can also obtain the raw plotting data using return_data=TRUE
plot_data <- plot(p, s, return_data = TRUE)
print(plot_data)
}
}
\references{
Faith DP (2008) Threatened species and the potential loss of
phylogenetic diversity: conservation scenarios based on estimated extinction
probabilities and phylogenetic risk analysis. \emph{Conservation Biology},
\strong{22}: 1461--1470.

Yu G, Smith DK, Zhu H, Guan Y, & Lam TTY (2017) ggtree: an
R package for visualization and annotation of phylogenetic trees with their
covariates and other associated data. \emph{Methods in Ecology and
Evolution}, \strong{8}: 28--36.
}
