#' Validate the library.
#' @description Compare current and expected packages under `.libPaths`.
#' Checking the healthy of the library, which packages are newest one.
#' Optionally added life duration of each package.
#' @param lib.loc character. Default: NULL
#' @param fields character vector with possible values `c("Depends", "Imports", "LinkingTo", "Suggests")`. Default: `c("Depends", "Imports", "LinkingTo")`
#' @param lifeduration logical if to add life duration column, might take some time. Default: FALSE
#' @param checkred logical if to add R CRAN check page status, any WARNING or ERROR will give TRUE. Default FALSE
#' @return data.frame with 5/6/7 columns.
#' \describe{
#' \item{Package}{character package names.}
#' \item{Version.expected.min}{character expected by DESCRIPTION files minimal version. "" means not specified so the newest version.}
#' \item{Version.have}{character installed package version.}
#' \item{version_status}{ numeric -1/0/1 which comes from `utils::compareVersion` function.
#' 0 means that we have the same version as required by DESCRIPTION files. -1 means we have too low version installed, this is an error. 1 means we have higher version.}
#' \item{life_duration}{(Optional) integer number of days package was released.}
#' \item{newest}{ logical if the installed version is the newest one.}
#' \item{checkred}{(Optional) logical if the newest package contains any errors or warnings on CRAN check page.}
#' }
#' @note Version.expected.min column not count packages which are not a dependency for any package, so could not be find in DESCRIPTION files.
#' When turn on the `lifeduration` and/or `checkred` options, calculations might be time consuming.
#' Please as a courtesy to the R CRAN, don't overload their server by constantly using this function with `lifeduration` or `checkred` turned on.
#' Results are cached for 1 hour with `memoise` package, memory cache.
#' @export
#' @examples
#' lib_validate()
#'
lib_validate <- function(lib.loc = NULL,
                         fields = c("Depends", "Imports", "LinkingTo"),
                         lifeduration = FALSE,
                         checkred = FALSE) {
  stopifnot(is.null(lib.loc) || all(lib.loc %in% .libPaths()))
  stopifnot(all(fields %in% c("Depends", "Imports", "Suggests", "LinkingTo")))
  stopifnot(is.logical(lifeduration))
  installed_agg <- installed_agg_fun(lib.loc, fields)

  res_agg <- installed_descriptions(lib.loc, fields)

  result <- merge(res_agg,
    rbind(
      installed_agg[, c("Package", "Version")],
      data.frame(
        Package = "R",
        Version = paste0(R.Version()[c("major", "minor")], collapse = "."), stringsAsFactors = FALSE
      )
    ),
    by = "Package",
    all = TRUE,
    suffix = c(".expected.min", ".have")
  )

  result$version_status <- apply(result, 1, function(x) utils::compareVersion(x["Version.have"], x["Version.expected.min"]))

  result <- result[!is.na(result$Package) & !(result$Package %in% c("NA", pacs_base())), ]

  if (lifeduration) {
    cat("Please wait, Packages life durations are assessed.\n")
    result$life_duration <- apply(result, 1, function(x) pac_lifeduration(x["Package"], x["Version.have"]))
  }

  result$newest <- apply(result, 1, function(x) is_last_release(x["Package"], x["Version.have"]))

  if (checkred) {
    cat("Please wait, Packages CRAN check statuses are assessed.\n")
    result$checkred <- vapply(seq_len(nrow(result)), function(x) result$newest[x] && pac_checkred(result$Package[x]), logical(1))
  }

  result
}

#' Validate a specific package
#' @description Checking the healthy of the specific packages.
#' @param pac character a package name.
#' @param lib.loc character. Default: NULL
#' @param fields character vector with possible values `c("Depends", "Imports", "LinkingTo", "Suggests")`. Default: `c("Depends", "Imports", "LinkingTo")`
#' @param lifeduration logical if to add life duration column, might take some time. Default: FALSE
#' @param checkred logical if to add R CRAN check page status, any WARNING or ERROR will give TRUE. Default FALSE
#' @return data.frame with 5/6/7 columns.
#' \describe{
#' \item{Package}{character package names.}
#' \item{Version.expected.min}{character expected by DESCRIPTION files minimal version. "" means not specified so the newest version.}
#' \item{Version.have}{character installed package version.}
#' \item{version_status}{ numeric -1/0/1 which comes from `utils::compareVersion` function.
#' 0 means that we have the same version as required by DESCRIPTION files. -1 means we have too low version installed, this is an error. 1 means we have higher version.}
#' \item{life_duration}{(Optional) integer number of days package was released.}
#' \item{newest}{ logical if the installed version is the newest one.}
#' \item{checkred}{(Optional) logical if the newest package contains any errors or warnings on CRAN check page.}
#' }
#' @note Version.expected.min column not count packages which are not a dependency for any package, so could not be find in DESCRIPTION files.
#' When turn on the `lifeduration` and/or `checkred` options, calculations might be time consuming.
#' Please as a courtesy to the R CRAN, don't overload their server by constantly using this function with `lifeduration` or `checkred` turned on.
#' Results are cached for 1 hour with `memoise` package, memory cache.
#' @export
#' @examples
#' pac_validate("memoise")
#'
pac_validate <- function(pac, lib.loc = NULL,
                         fields = c("Depends", "Imports", "LinkingTo"),
                         lifeduration = FALSE,
                         checkred = FALSE) {
  stopifnot(is.null(lib.loc) || all(lib.loc %in% .libPaths()))
  stopifnot(all(fields %in% c("Depends", "Imports", "Suggests", "LinkingTo")))
  stopifnot((length(pac) == 1) && is.character(pac))

  descriptions_pac <- pac_deps(pac, lib.loc = lib.loc, fields = fields, description_v = TRUE)
  installed_pac <- pac_deps(pac, lib.loc = lib.loc, fields = fields)

  result <- merge(descriptions_pac,
    installed_pac,
    by = "Package",
    all = TRUE,
    suffix = c(".expected.min", ".have")
  )

  result$version_status <- apply(result, 1, function(x) utils::compareVersion(x["Version.have"], x["Version.expected.min"]))

  result <- result[!is.na(result$Package) & !(result$Package %in% c("NA", pacs_base())), ]

  if (lifeduration) {
    cat("Please wait, Packages life durations are assessed.\n")
    result$life_duration <- apply(result, 1, function(x) pac_lifeduration(x["Package"], x["Version.have"]))
  }

  result$newest <- apply(result, 1, function(x) is_last_release(x["Package"], x["Version.have"]))

  if (checkred) {
    cat("Please wait, Packages CRAN check statuses are assessed.\n")
    result$checkred <- vapply(seq_len(nrow(result)), function(x) result$newest[x] && pac_checkred(result$Package[x]), logical(1))
  }

  result
}

#' Validate specific packages
#' @description Checking the healthy of the specific packages.
#' @param pacs character vector packages names.
#' @param lib.loc character. Default: NULL
#' @param fields character vector with possible values `c("Depends", "Imports", "LinkingTo", "Suggests")`. Default: `c("Depends", "Imports", "LinkingTo")`
#' @param lifeduration logical if to add life duration column, might take some time. Default: FALSE
#' @param checkred logical if to add R CRAN check page status, any WARNING or ERROR will give TRUE. Default FALSE
#' @return data.frame with 5/6/7 columns.
#' \describe{
#' \item{Package}{character package names.}
#' \item{Version.expected.min}{character expected by DESCRIPTION files minimal version. "" means not specified so the newest version.}
#' \item{Version.have}{character installed package version.}
#' \item{version_status}{ numeric -1/0/1 which comes from `utils::compareVersion` function.
#' 0 means that we have the same version as required by DESCRIPTION files. -1 means we have too low version installed, this is an error. 1 means we have higher version.}
#' \item{life_duration}{(Optional) integer number of days package was released.}
#' \item{newest}{ logical if the installed version is the newest one.}
#' \item{checkred}{(Optional) logical if the newest package contains any errors or warnings on CRAN check page.}
#' }
#' @note Version.expected.min column not count packages which are not a dependency for any package, so could not be find in DESCRIPTION files.
#' When turn on the `lifeduration` and/or `checkred` options, calculations might be time consuming.
#' Please as a courtesy to the R CRAN, don't overload their server by constantly using this function with `lifeduration` or `checkred` turned on.
#' Results are cached for 1 hour with `memoise` package, memory cache.
#' @export
#' @examples
#' pacs_validate(c("memoise", "rlang"))
#'
pacs_validate <- function(pacs,
                          lib.loc = NULL,
                          fields = c("Depends", "Imports", "LinkingTo"),
                          lifeduration = FALSE,
                          checkred = FALSE) {
  stopifnot(is.null(lib.loc) || all(lib.loc %in% .libPaths()))
  stopifnot(all(fields %in% c("Depends", "Imports", "Suggests", "LinkingTo")))
  stopifnot(is.null(pacs) || is.character(pacs))

  descriptions_pac <- pacs_deps(pacs, lib.loc = lib.loc, fields = fields, description_v = TRUE)
  installed_pac <- pacs_deps(pacs, lib.loc = lib.loc, fields = fields)

  result <- merge(descriptions_pac,
    installed_pac,
    by = "Package",
    all = TRUE,
    suffix = c(".expected.min", ".have")
  )

  result$version_status <- apply(result, 1, function(x) utils::compareVersion(x["Version.have"], x["Version.expected.min"]))

  result <- result[!is.na(result$Package) & !(result$Package %in% c("NA", pacs_base())), ]

  if (lifeduration) {
    cat("Please wait, Packages life durations are assessed.\n")
    result$life_duration <- apply(result, 1, function(x) pac_lifeduration(x["Package"], x["Version.have"]))
  }

  result$newest <- apply(result, 1, function(x) is_last_release(x["Package"], x["Version.have"]))

  if (checkred) {
    cat("Please wait, Packages CRAN check statuses are assessed.\n")
    result$checkred <- vapply(seq_len(nrow(result)), function(x) result$newest[x] && pac_checkred(result$Package[x]), logical(1))
  }

  result
}
