#' Generate time bins
#'
#' A function to generate time bins for a given study interval. This function
#' is flexible in that either stage-level or higher stratigraphic-level
#' (e.g. period) time bins can be called.
#' In addition, near equal-length time bins can be generated by grouping stages
#' together. For example, for 10 Myr as a target bin size, the
#' function will generate groups of bins that have a mean bin length close to
#' 10 Myr.
#' However, users may also want to consider grouping stages based on other
#' reasoning e.g. availability of outcrop (see Dean et al. 2020).
#'
#' @param interval \code{character or numeric}. Interval name available in
#'   \code{\link{GTS2020}} or \code{\link{GTS2012}}. If a single interval name
#'   is provided, this interval is returned. If two interval names are
#'   provided, these intervals and those existing between are returned. If a
#'   single numeric age is provided, the interval that covers this age is
#'   returned. If two numeric ages are provided, the intervals occurring in
#'   the range of these ages are returned. Defaults to "Phanerozoic".
#' @param rank \code{character}. Which stratigraphic rank is desired? Choose
#'   from: "stage", "epoch", "period", "era", and "eon". If `scale` is a
#'   dataframe, this argument is ignored.
#' @param size \code{numeric}. If equal-length time bins are desired, specify
#'   the length in millions of years (Myr) of the time bins desired.
#' @param assign \code{numeric}. A numeric vector of age estimates to use to
#'   assign to requested bins. If assign is specified, a numeric vector is
#'   returned of the midpoint age of the specified bins. Note this is the
#'   simplified approach of assignment in `palaeoverse` included for data with
#'   'known' point-age estimates. For a wider range of binning methods, see
#'   \code{\link[palaeoverse:bin_time]{palaeoverse::bin_time()}}.
#' @param scale \code{character} or \code{data.frame}. Specify the desired
#'   geological timescale to be used "GTS2020" (default), "GTS2012" or a
#'   user-input `data.frame`. If a `data.frame` is provided, it must contain
#'   at least the following named columns: "interval_name", "max_ma", and
#'   "min_ma". Column names "name", "max_age", and "min_age" are also accepted,
#'   but these are assumed to be equivalent to the aforementioned. As such,
#'   age data should be provided in Ma.
#' @param plot \code{logical}. Should a plot of time bins be generated?
#' @importFrom graphics polygon title
#' @importFrom stats sd
#'
#' @return A \code{dataframe} of time bins for the specified intervals or a
#' list with a \code{dataframe} of time bins and a named \code{numeric}
#' vector (bin number) of binned age estimates (midpoint of specified bins) if
#' assign is specified.
#'
#' @details This function uses either the Geological Time Scale 2020,
#' Geological Time Scale 2012, or a user-input `data.frame` (see `scale`
#' argument) to generate time bins. Interval data hosted by Macrostrat are
#' also compatiable and accessible via the deeptime R pacakge
#' (\code{\link[deeptime]{getScaleData}}).
#' Additional information on included Geological Time Scales and source can be
#' accessed via:
#' - \code{\link{GTS2020}}
#' - \code{\link{GTS2012}}
#'
#' Available intervals names are accessible via the `interval_name`
#' column in `GTS2012` and `GTS2020`. Data of the Geological Timescale 2020
#' and 2012 were compiled by Lewis A. Jones (2022-07-02).
#'
#' @section References:
#'
#' Dean, C.D., Chiarenza, A.A. and Maidment, S.C., 2020. Formation binning: a
#' new method for increased temporal resolution in regional studies, applied
#' to the Late Cretaceous dinosaur fossil record of North America.
#' Palaeontology, 63(6), 881-901. \doi{10.1111/pala.12492}.
#'
#' @section Developer(s):
#' Lewis A. Jones
#' @section Reviewer(s):
#' Kilian Eichenseer & William Gearty
#' @export
#' @examples
#' #Using numeric age
#' ex1 <- time_bins(interval = 10, plot = TRUE)
#'
#' #Using numeric age range
#' ex2 <- time_bins(interval = c(50, 100), plot = TRUE)
#'
#' #Using a single interval name
#' ex3 <- time_bins(interval = c("Maastrichtian"), plot = TRUE)
#'
#' #Using a range of intervals and near-equal duration bins
#' ex4 <- time_bins(interval = c("Fortunian", "Meghalayan"),
#'                  size = 10, plot = TRUE)
#'
#' #Assign bins based on given age estimates
#' ex5 <- time_bins(interval = c("Fortunian", "Meghalayan"),
#'                  assign = c(232, 167, 33))
#'
#' #Use user-input data.frame to generate near-equal length bins
#' scale <- data.frame(interval_name = 1:5,
#'                     min_ma = c(0, 18, 32, 38, 45),
#'                     max_ma = c(18, 32, 38, 45, 53))
#' ex6 <- time_bins(scale = scale, size = 20, plot = TRUE)
#'
#' #Use North American land mammal ages from deeptime/Macrostrat
#' scale <- deeptime::getScaleData(name = "North American land mammal ages")
#' ex7 <- time_bins(scale = scale, size = 10)
#'
time_bins <- function(interval = "Phanerozoic", rank = "stage",
           size = NULL, assign = NULL, scale = "GTS2020", plot = FALSE) {
    # Error handling
    if (!is.character(interval) && !is.numeric(interval)) {
      stop("`interval` must be of class 'character' or 'numeric'")
    }

    if (is.numeric(size) == FALSE && is.null(size) == FALSE) {
      stop("`size` should be a numeric or NULL")
    }

    if (is.logical(plot) == FALSE) {
      stop("`plot` should be logical (TRUE/FALSE)")
    }

    if (is.numeric(assign) == TRUE && any(assign < 0) == TRUE) {
      stop("Age estimates for `assign` should be non-negative values.
  You can transform your data using abs().")
    }

    if (is.character(scale) &&
        scale != "GTS2012" &&
        scale != "GTS2020") {
      stop("`scale` must be either GTS2012, GTS2020 or a data.frame")
    }

  if (is.data.frame(scale) &&
      any(!c("max_ma", "min_ma") %in% colnames(scale)) &&
      any(!c("max_age", "min_age") %in% colnames(scale))) {
    stop("`scale` does not contain named columns: max_ma and min_ma")
  }

    if (length(interval) > 2) {
      stop("`interval` must be a character or numeric vector of length 1 or 2")
    }

    if (length(rank) > 1) {
      stop("`rank` must be of length 1")
    }

    if ((rank %in% c("stage", "epoch", "period", "era", "eon")) == FALSE) {
      stop("`rank` must be either: stage, epoch, period, era, or eon")
    }

    # Grab data
    # Which geological timescale to use?
  if (is.character(scale)) {
    if (scale == "GTS2020") {
      df <- palaeoverse::GTS2020
    }
    if (scale == "GTS2012") {
      df <- palaeoverse::GTS2012
    }

    colnames(df)[which(colnames(df) == "interval_number")] <- "bin"

    #character string entered
    if (is.character(interval) && length(interval) == 1) {
      #rank ages
      rank_ages <- df[which(df$rank == rank),]
      w <- which(df$interval_name %in% interval)
      if (length(w) != length(interval)) {
        stop(
          paste0(
            "Check spelling of specified intervals.
  Available intervals are accessible via GTS2020 and GTS2012."
          )
        )
      }
      rank_ages <-
        rank_ages[which(rank_ages$max_ma > df$min_ma[w] &
                          rank_ages$min_ma < df$max_ma[w]),]
      df <- rank_ages
    }
    if (is.character(interval) && length(interval) == 2) {
      # rank ages
      rank_ages <- df[which(df$rank == rank),]
      w <- which(df$interval_name %in% interval)
      if (length(w) != length(interval)) {
        stop(
          paste0(
            "Check spelling of specified intervals.
  Available intervals are accessible via GTS2020 and GTS2012."
          )
        )
      }
      rank_ages <-
        rank_ages[which(rank_ages$max_ma > min(df$min_ma[w]) &
                          rank_ages$min_ma < max(df$max_ma[w])),]
      df <- subset(df, max_ma <= max(rank_ages$max_ma))
      df <- subset(df, min_ma >= min(rank_ages$min_ma))
    }

    #subset to rank
    df <- df[which(df$rank == rank),]

    #numeric ages entered
    if (is.numeric(interval) && length(interval) == 1) {
      if (interval > max(df$max_ma) || interval < min(df$min_ma)) {
        stop("Value does not appear in the range of available intervals:
    0 to 541")
      }
      int_index <-
        which(interval <= df$max_ma & interval >= df$min_ma)
      df <- df[int_index,]
    }

    if (is.numeric(interval) && length(interval) == 2) {
      max_int <- max(interval)
      min_int <- min(interval)

      if (max_int > max(df$max_ma) || min_int < min(df$min_ma)) {
        stop("Values do not appear in the range of available intervals:
          0 to 541")
      }

      int_index <-
        which(min_int <= df$max_ma & max_int >= df$min_ma)
      df <- df[int_index,]
    }
  } else {
    # Assign input scale to df
    df <- scale
    # Match column names to getScaleData (or other user-input data)
    col_indx <- match(c("name", "max_age", "min_age"), colnames(df))
    # Column names
    cnames <- c("interval_name", "max_ma", "min_ma")[which(!is.na(col_indx))]
    # Remove NA matches
    col_indx <- col_indx[!is.na(col_indx)]
    if (length(col_indx) >= 1) {
      colnames(df)[col_indx] <- cnames
    }

    # Add mid_ma and duration myr
    df$mid_ma <- (df$max_ma + df$min_ma) / 2
    df$duration_myr <- df$max_ma - df$min_ma
    # Rearrange columns
    colnme <- colnames(df)
    order_nme <- c("interval_name", "max_ma",
                   "mid_ma", "min_ma", "duration_myr")
    colnme <- colnme[!colnme %in% order_nme]
    order_nme <- c(colnme, order_nme)
    df <- df[, order_nme]

    # Order by mid_ma
    df <- df[order(df$mid_ma), ]
  }

    #are equal length time bins required?

    if (is.numeric(size) == TRUE) {
      # Update bin size for age range
      # How many bins should be generated?
      n_bins <- round((max(df$max_ma) - min(df$min_ma)) / size)
      size <- (max(df$max_ma) - min(df$min_ma)) / n_bins
      #track cumulative sum
      tracker <- list()
      for (i in seq_len(length.out = nrow(df))) {
        tracker[[i]] <- rep(NA, length.out = nrow(df))
        tracker[[i]][i:nrow(df)] <- abs(
          cumsum(df$duration_myr[i:nrow(df)]) - size)
      }

      #calculate upper and lower limits for each bin
      lower <- NULL
      upper <- NULL
      count <- 1
      while (count <= nrow(df)) {
        upper <- append(upper, which.min(tracker[[count]]))
        lower <- append(lower, (count))
        count <- which.min(tracker[[count]]) + 1
      }

      #generate bin information
      bin <- rev(seq_along(upper))
      max_ma <- df[upper, c("max_ma")]
      min_ma <- df[lower, c("min_ma")]
      mid_ma <- (max_ma + min_ma) / 2
      duration_myr <- max_ma - min_ma
      intervals <- vector("character")

      #resolve any edge effect
      if (duration_myr[length(duration_myr)] < size / 2) {
        upper[length(upper) - 1] <- upper[length(upper)]
        upper <- upper[-length(upper)]
        lower <- lower[-length(lower)]
        bin <- rev(seq_along(upper))
        max_ma <- df[upper, c("max_ma")]
        min_ma <- df[lower, c("min_ma")]
        mid_ma <- (max_ma + min_ma) / 2
        duration_myr <- max_ma - min_ma
      }

      #get interval names
      for (i in seq_along(upper)) {
        intervals[i] <- toString(df[lower[i]:upper[i], c("interval_name")])
      }

      #generate dataframe
      grouping_rank <- rank
      df <- cbind.data.frame(bin,
                             max_ma,
                             mid_ma,
                             min_ma,
                             duration_myr,
                             grouping_rank,
                             intervals)

      #message user
      message(
        paste0(
          "Target equal length time bins was set to ",
          round(size, digits = 2),
          " Myr. \nGenerated time bins have a mean length of ",
          round(mean(df$duration_myr), digits = 2),
          " Myr and a standard deviation of ",
          round(sd(df$duration_myr), digits = 2),
          " Myr."
        )
      )
    }

    if (plot == TRUE) {
      if (is.numeric(size) == TRUE) {
        df$colour <- c("#4292c6")
      }
      plot(
        1,
        type = "n",
        xlim = c(max(df$max_ma), min(df$min_ma)),
        ylim = c(0, max(df$duration_myr)),
        xlab = "Time (Ma)",
        ylab = "Duration (Myr)"
      )
      for (i in seq_len(length.out = nrow(df))) {
        polygon(
          x = c(df$min_ma[i], df$max_ma[i], df$max_ma[i], df$min_ma[i]),
          y = c(0, 0, df$duration_myr[i], df$duration_myr[i]),
          col = df$colour[i]
        )
      }
      if (is.numeric(size) == TRUE) {
        title(paste0(
          "Mean bin length = ",
          round(mean(df$duration_myr), digits = 2),
          " (standard deviation = ",
          round(sd(df$duration_myr), digits = 2),
          ")"
        ))
        df <- df[, -which(colnames(df) == "colour")]
      }

      if (is.numeric(size) == FALSE) {
        df$bin <- rev(seq_len(length.out = nrow(df)))
        df <-
          df[, c("bin",
                 "interval_name",
                 "max_ma",
                 "mid_ma",
                 "min_ma",
                 "duration_myr",
                 "rank")]
      }
      df <- df[order(df$bin, decreasing = FALSE), ]

    }

    if (!is.null(assign)) {
      if (is.numeric(assign)) {
        if (any(assign > max(df$max_ma) | assign < min(df$min_ma))) {
          stop("One or more ages is outside the specified time interval range")
        }
        tmp <- assign
        for (i in seq_len(length.out = nrow(df))) {
          assign[which(tmp <= df$max_ma[i] &
                         tmp >= df$min_ma[i])] <- df$mid_ma[i]
          names(assign)[which(tmp <= df$max_ma[i] &
                         tmp >= df$min_ma[i])] <- df$bin[i]
        }
        assign <- list(df, assign)
        names(assign) <- c("Bins", "Assignation")
        return(assign)
      } else {
        stop("`assign` should be a numeric")
      }
    }
  # Clean up
  if (is.data.frame(scale) && !is.null(size)) {
    df <- df[, -which(colnames(df) == "grouping_rank")]
  }

    return(df)
  }
