% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/exhaustationFunctions.R
\name{exhaustionFunctions}
\alias{exhaustionFunctions}
\alias{accioExhaustionCurve}
\alias{accioBestAcquisitionModel}
\alias{charExhaustPlot}
\title{Analyses of the exhaustion of Character States Over Evolutionary History}
\usage{
accioExhaustionCurve(phyloTree, charData, charTypes = "unordered",
  outgroup = NULL, firstAppearances = NULL, missingCharValue = "?",
  inapplicableValue = "-")

accioBestAcquisitionModel(exhaustion_info, changesType,
  models = c("exponential", "gamma", "lognormal", "zipf"))

charExhaustPlot(exhaustion_info, changesType, xlab = "Total Characters",
  ylab = NULL, main = NULL, xsize = 3)
}
\arguments{
\item{phyloTree}{A phylogenetic tree of class \code{phylo} as used by package \code{ape}.}

\item{charData}{A \code{data.frame} of morphological character
codings (a morphological 'matrix'), with taxon units
as rows and characters as columns.}

\item{charTypes}{A vector of length equal to
the number of chaacters in \code{charData}, with elements indicating
whether the corresponding character in \code{charData}
is \code{"unordered"} or \code{"ordered"}. However,
if \code{length(charTypes) = 1}, then it is repeated for all taxa.
The default value for this argument is \code{"unordered"}.}

\item{outgroup}{A string matching to one
of the tip labels as given by \code{tip.label},}

\item{firstAppearances}{A vector, with length equal to the
same number of taxa (rows) as in \code{charData}, in the same corresponding order.}

\item{missingCharValue}{The string value indicating a missing
character coding value, by default \code{"?"}.}

\item{inapplicableValue}{The string value indicating an
inapplicable character coding value, by default \code{"-"}.}

\item{exhaustion_info}{The list of results output
from function \code{accioExhaustionCurve}.}

\item{changesType}{A single character value, indicating the character change data
to be assessed from the result of the character
exhaustion analysis, must be one of either 'totalAcc' (to the total number of
accumulated character changes, ideal for modeling the size and distribution of
\emph{state} space) or 'charAlt' (to plot the total number of character alterations,
ideal for modeling the size and distribution of \emph{character} space).}

\item{models}{A vector of type \code{character} naming models to be fit.
Default is \code{c("exponential","gamma","lognormal","zipf")}.}

\item{xlab}{Label for the X axis; \code{"Total Characters"} by default.}

\item{ylab}{Label for the Y axis. If not provided by the user,
a label based on the \code{changesType} argument will be used.}

\item{main}{Main title label for the plot. If not provided by
the user, a label based on the \code{changesType} argument will be used.}

\item{xsize}{Parameter controling size of the axes,
which are forced to be symmetric.}
}
\value{
\code{accioExhaustionCurve} outputs a list containing two objects: first,
a matrix named \code{exhaustion} consisting of three columns: \code{"Steps"},
\code{"Novel_States"}, and \code{"Novel_Characters"}, respectively giving
the counts of these respective values for each branch (internode edge).
The second element of this list is named \code{State_Derivations} and is
a count of how often each state, across all characters, was derived relative
to the primitive position along each internode edge.

The output of \code{accioBestAcquisitionModel} is a list object containing
information on the best-fit model, the parameters of that model, the calculated
probabilition distribution for that model at the same intervals, for use in quantile plots.	

\code{charExhaustPlot} produces a plot, and outputs no data.
}
\description{
The following functions are for measuring and fitting various
distributions to the gradual exhaustion of unexpressed
character states, as originally described by Wagner (2000,
Evolution).
}
\details{
\code{accioExhaustionCurve} uses a Sankoff parsimony ancestral-reconstruction
algorithm (written by PJ Wagner, \emph{not} the one from \code{phangorn} used
elsewhere in paleotree) to calculate character changes across each branch
(internode edge) of a tree, and then reports the counts of character state

\code{accioBestAcquisitionModel} takes output from \code{accioExhaustionCurve},
calculates one of two character change indices, and then fits a series of user-selected
models to the dataset, returning details pertaining to the best-fit model.

\code{charExhaustPlot} is a wrapper for \code{accioBestAcquisitionModel} that
produces a plot of the observed character change data against the
expectation under the best-fit model.


The functions \code{accioBestAcquisitionModel}  and \code{charExhaustPlot} offer
users two different options for examining character change: \code{totalAcc}
fits models to the total accumulated number of state changes over the phylogeny,
thus using exhaustion to explor the size and distribution of character space. The
other option \code{charAlt} fits models to the number of character that alter from
primitive to derived over phylogeny, thus reflecting the size and distribution of state space.

\code{accioExhaustionCurve} can order its reconstruction of change by stratigraphic order of first appearances. It is unclear what this means.
}
\note{
This family of functios presented here were originally written 
by Peter J. Wagner, and then modified and adapted by David W.
Bapst for wider release  in a CRAN-distributed
package: \code{paleotree}. This makes the code presented here
a very different beast than typical paleotree code, for
example, there are fewer tests of correct input type, length, etc.
}
\examples{
\donttest{

# get data
data(SongZhangDicrano)

dicranoTree<-cladogramDicranoX13

# modify char data
charMat<-data.matrix(charMatDicrano)
charMat[is.na(charMatDicrano)]<-0
charMat<-(charMat-1)
colnames(charMat)<-NULL
# replace missing values
charMat[is.na(charMatDicrano)]<-"?"

# the 'outgroup' is Exigraptus, first taxon listed in the matrix
exhaustionResults <- accioExhaustionCurve(phyloTree=dicranoTree,
   charData=charMat, charTypes="unordered",
   outgroup="Exigraptus_uniformis")

# fits models to exhaustion for total accumulation
accioBestAcquisitionModel(exhaustion_info=exhaustionResults,
 changesType="totalAcc", 	
 models=c("exponential","gamma","lognormal","zipf")) 

# plot of exhausation of total accumulation of character states
charExhaustPlot(exhaustion_info=exhaustionResults,
	   changesType="totalAcc")

# plot of exhausation of character alterations
charExhaustPlot(exhaustion_info=exhaustionResults,
	   changesType="charAlt")

}	
}
\references{
Wagner, P. J. 2000. Exhaustion of morphologic character
states among fossil taxa. \emph{Evolution} 54(2):365-386.
}
\seealso{
Also see \code{paleotree} functions \code{\link{minCharChange}} and
\code{\link{ancPropStateMat}}, the latter of which is a wrapper
for \code{phangorn}'s function \code{ancestral.pars}.
}
\author{
Initially written by Peter J. Wagner, with modification
and documentation by David W. Bapst.
}
