% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit.R
\name{fit.model_spec}
\alias{fit.model_spec}
\alias{fit_xy.model_spec}
\title{Fit a Model Specification to a Dataset}
\usage{
\method{fit}{model_spec}(
  object,
  formula,
  data,
  case_weights = NULL,
  control = control_parsnip(),
  ...
)

\method{fit_xy}{model_spec}(object, x, y, case_weights = NULL, control = control_parsnip(), ...)
}
\arguments{
\item{object}{An object of class \code{model_spec} that has a chosen engine
(via \code{\link[=set_engine]{set_engine()}}).}

\item{formula}{An object of class \code{formula} (or one that can
be coerced to that class): a symbolic description of the model
to be fitted.}

\item{data}{Optional, depending on the interface (see Details
below). A data frame containing all relevant variables (e.g.
outcome(s), predictors, case weights, etc). Note: when needed, a
\emph{named argument} should be used.}

\item{case_weights}{An optional classed vector of numeric case weights. This
must return \code{TRUE} when \code{\link[hardhat:is_case_weights]{hardhat::is_case_weights()}} is run on it. See
\code{\link[hardhat:frequency_weights]{hardhat::frequency_weights()}} and \code{\link[hardhat:importance_weights]{hardhat::importance_weights()}} for
examples.}

\item{control}{A named list with elements \code{verbosity} and
\code{catch}. See \code{\link[=control_parsnip]{control_parsnip()}}.}

\item{...}{Not currently used; values passed here will be
ignored. Other options required to fit the model should be
passed using \code{set_engine()}.}

\item{x}{A matrix, sparse matrix, or data frame of predictors. Only some
models have support for sparse matrix input. See \code{parsnip::get_encoding()}
for details. \code{x} should have column names.}

\item{y}{A vector, matrix or data frame of outcome data.}
}
\value{
A \code{model_fit} object that contains several elements:
\itemize{
\item \code{lvl}: If the outcome is a factor, this contains
the factor levels at the time of model fitting.
\item \code{spec}: The model specification object
(\code{object} in the call to \code{fit})
\item \code{fit}: when the model is executed without error,
this is the model object. Otherwise, it is a \code{try-error}
object with the error message.
\item \code{preproc}: any objects needed to convert between
a formula and non-formula interface (such as the \code{terms}
object)
}
The return value will also have a class related to the fitted model (e.g.
\code{"_glm"}) before the base class of \code{"model_fit"}.
}
\description{
\code{fit()} and \code{fit_xy()} take a model specification, translate the required
code by substituting arguments, and execute the model fit
routine.
}
\details{
\code{fit()} and \code{fit_xy()} substitute the current arguments in the model
specification into the computational engine's code, check them
for validity, then fit the model using the data and the
engine-specific code. Different model functions have different
interfaces (e.g. formula or \code{x}/\code{y}) and these functions translate
between the interface used when \code{fit()} or \code{fit_xy()} was invoked and the one
required by the underlying model.

When possible, these functions attempt to avoid making copies of the
data. For example, if the underlying model uses a formula and
\code{fit()} is invoked, the original data are references
when the model is fit. However, if the underlying model uses
something else, such as \code{x}/\code{y}, the formula is evaluated and
the data are converted to the required format. In this case, any
calls in the resulting model objects reference the temporary
objects used to fit the model.

If the model engine has not been set, the model's default engine will be used
(as discussed on each model page). If the \code{verbosity} option of
\code{\link[=control_parsnip]{control_parsnip()}} is greater than zero, a warning will be produced.

If you would like to use an alternative method for generating contrasts when
supplying a formula to \code{fit()}, set the global option \code{contrasts} to your
preferred method. For example, you might set it to:
\code{options(contrasts = c(unordered = "contr.helmert", ordered = "contr.poly"))}.
See the help page for \code{\link[stats:contrast]{stats::contr.treatment()}} for more possible contrast
types.

For models with \code{"censored regression"} modes, an additional computation is
executed and saved in the parsnip object. The \code{censor_probs} element contains
a "reverse Kaplan-Meier" curve that models the probability of censoring. This
may be used later to compute inverse probability censoring weights for
performance measures.
}
\examples{
# Although `glm()` only has a formula interface, different
# methods for specifying the model can be used

library(dplyr)
library(modeldata)
data("lending_club")

lr_mod <- logistic_reg()

using_formula <-
  lr_mod \%>\%
  set_engine("glm") \%>\%
  fit(Class ~ funded_amnt + int_rate, data = lending_club)

using_xy <-
  lr_mod \%>\%
   set_engine("glm") \%>\%
  fit_xy(x = lending_club[, c("funded_amnt", "int_rate")],
         y = lending_club$Class)

using_formula
using_xy
}
\seealso{
\code{\link[=set_engine]{set_engine()}}, \code{\link[=control_parsnip]{control_parsnip()}}, \code{model_spec}, \code{model_fit}
}
