#' @title Adjust edge label
#' @description The edge label is assigned when creating the process map by `create_pmap()`, however, it can be changed by this function.
#' @usage adjust_edge_label(p, label = c(
#'    "amount",
#'    "mean_duration",
#'    "median_duration",
#'    "max_duration",
#'    "min_duration"
#'  ))
#' @param p The process map generated by `create_pmap()`
#' @param label Specify which attribute is used for the edge label.
#' @examples
#'  library(dplyr)
#'  p <- generate_eventlog() %>% create_pmap()
#'  p <- adjust_edge_label(p, label = "mean_duration")
#' @importFrom DiagrammeR   get_edge_df
#' @importFrom DiagrammeR   set_edge_attrs
#' @export
adjust_edge_label <- function(p, label = c(
    "amount",
    "mean_duration",
    "median_duration",
    "max_duration",
    "min_duration"
  )
) {
  edges <- DiagrammeR::get_edge_df(p)

  # Handle empty edge graph case
  if (nrow(edges) == 0) return(p)

  label <- match.arg(label)
  edge_label_value <- switch(
    label,
    amount = edges$amount,
    max_duration = edges$max_duration,
    mean_duration = edges$mean_duration,
    median_duration = edges$median_duration,
    min_duration = edges$min_duration
  )

  p <- DiagrammeR::set_edge_attrs(p, edge_attr = "label", values = paste0("   ", edge_label_value, "   "))
}
