# Dataset transformations

#' Kendall transformation
#' 
#' @param x Vector or data frame to be Kendall-transformed; allowed feature types are numeric, integer (treated as numeric), ordered factor, logical and unordered factor with two or less levels.
#' \code{NA} and non-finite values are allowed; \code{NaN} is treated as \code{NA}.
#' @return A transformed vector or data frame with transformed columns.
#' @references Kendall transformation M.B. Kursa, arXiv:2006.15991 (2020).
#' @examples
#' kTransform(data.frame(Asc=1:3,Desc=3:1,Vsh=c(2,1,2)))
#' @export
kTransform<-function(x)
 if(is.data.frame(x)) data.frame(.Call(C_kt,x)) else .Call(C_kt,x)

#' Inverse Kendall transform
#'
#' This function attempts to reverse Kendall transformation using a simple ranking agreement method, which always restores original ranking if the input corresponds to one, or a reasonable best-effort guess if not.
#' Namely, each objects gets a score based on its relation with each other object, 2 points for a win (\code{'>'}) and 1 point for a tie (\code{'='}); these scores are used to calculate ranks.
#' This function can also be directly given greater-than scores, for instance confidence scores from some classifier trained on Kendall-transformed data.
#' @param x A Kendall-transformed feature to be converted back into a ranking.
#' To be interpreted as a such, it must be a factor with levels being a subset of \code{'<'}, \code{'>'} or \code{'='}.
#' Alternatively, it may be a numeric vector of greater-than scores.
#' @return Vector of ranks corresponding to \code{x}.
#' @note An order of elements in \code{x} is crucial; if it is not the same as generated by the \code{\link{kTransform}}, results will be wrong.
#' This function cannot assert that the order is correct.
#' @references Kendall transformation M.B. Kursa, arXiv:2006.15991 (2020).
#' @examples
#' kInverse(kTransform(1:7))
#' @export
kInverse<-function(x){
 if(is.factor(x))
  if(all(levels(x)%in%c("<",">","=")))
   x<-factor(x,levels=c("<","=",">"))
  else stop("Factor does not seem to be a Kendall-transformed variable")
 rank(colSums(.Call(C_rkt,x),na.rm=TRUE))
}
