#' @title \code{qqtest} A self-calibrated quantile quantile plot for testing distributional shape.
#'
#' @description Draws a quantile quantile plot for visually assessing whether the data come from a test distribution that has been defined in one of many ways.  
#'    The vertical axis plots the data quantiles, the horizontal those of a test distribution.
#'    Interval estimates and exemplars provide different comparative information to assess the evidence provided by the qqplot against the hypothesis that the data come from the test distribution (default is normal or gaussian).  Interval estimates provide test information related to individual quantiles, exemplars provide test information related to the shape of the quantile quantile curve.  
#'    Optionally, a visual test of significance (a lineup plot) can be displayed to provide a coarse level of significance for testing the null hypothesis that the data come from the test distribution. 
#'    The default behaviour generates 1000 samples from the test distribution and overlays the plot with pointwise interval estimates for the ordered quantiles from the test distribution.  A small number of independently generated exemplar test distribution sample quantile traces are also overlaid.  Various option choices are available to effect different visualizations of the uncertainty surrounding the quantile quantile plot (see argument descriptions and examples).
#'
#' @export qqtest 
#' 
#' @param data A univariate dataset to be tested. If data has more than one column, the first is used.
#' @param dist The name of the distribution against which the comparison is made, the test distribution for a few built-in distributions.  One of \code{"gaussian"} (or \code{"normal"}), \code{"log normal"},\code{"half normal"}, \code{"uniform"},\code{"student"},  \code{"chi-squared"}, or \code{"kay"}.  Only the first three characters of any of these is needed to specify the dist.  If dist is \code{"student"},  \code{"chi-squared"}, or \code{"kay"}, then a value for the degrees of freedom argument (\code{df} below) is also required.
#' @param df Degrees of freedom of \code{dist} to be used when \code{dist} is either \code{"student"} or \code{"chi-squared"}.
#' @param qfunction If non-\code{NULL}, this must be a function of a single argument (a proportion, p say) which will be used to calculate the quantiles for the test distribution.  If non-\code{NULL}, the \code{rfunction} should also be non-\code{NULL}.  The value of the \code{dist} argument will be ignored when this is the case.
#' @param rfunction If non-\code{NULL}, this must be a function of a single argument (a count, n say) which will be used to randomly select a sample of size n from the test distribution.   If non-\code{NULL}, the \code{qfunction} must also be non-NULL.  If  \code{qfunction} is non-\code{NULL} and  \code{rfunction} is \code{NULL}, then \code{qfunction} will be applied to the output of a call to \code{runif} in place of the \code{NULL} \code{rfunction} (i.e. a probability integral transform is used to generate a random sample).  The value of the \code{dist} argument will be ignored whenever \code{qfunction} is a function.
#' @param dataTest If non-\code{NULL}, this must be a second data set.  The empirical distribution given by this data will be used as the test distribution against which the value of data will be tested.   If non-\code{NULL}, the values of the arguments \code{dist}, \code{qfunction}, and \code{rfunction} will all be ignored in favour of using this empirical distribution as the test distribution.
#' @param xAxisAsProbs If \code{TRUE} (the default) the horizontal axis will be labelled as probabilities.  These are the cumulative probabilities according to the test distribution.  They are located at the corresponding quantile values.  They are handy in comparing percentiles of the test and data distributions as well as giving some measure of the symmetry and tail weights of the test distribution by their location.  If \code{FALSE} the axis is labelled according to the quantile values.
#' @param yAxisAsProbs If \code{TRUE} (the default) the vertical axis will be labelled as probabilities.  These are the cumulative probabilities according to the empirical distribution of the \code{data}.  They are located at the corresponding quantile values.  They are handy in comparing percentiles of the test and data distributions as well as giving some measure of the symmetry and tail weights of the \code{data} distribution by their location. If \code{FALSE} the axis is labelled according to the quantile values.
#' @param xAxisProbs A vector of probabilities to be used to label the x axis ticks when \code{xAxisAsProbs} is \code{TRUE}.  Default is \code{c(0.05, 0.25, 0.50, 0.75, 0.95)}.  Ignored if \code{xAxisAsProbs} is \code{FALSE}.
#' @param yAxisProbs A vector of probabilities to be used to label the y axis ticks when \code{yAxisAsProbs} is \code{TRUE}.  Default is \code{c(0.05, 0.25, 0.50, 0.75, 0.95)}.  Ignored if \code{yAxisAsProbs} is \code{FALSE}.
#' @param nreps The number of replicate samples to be taken from the test distribution to construct the pointwise intervals for each quantile.  Default is 1000.  From these samples, an empirical distribution is generated from the test distribution for the ordered quantiles corresponding to the values of\code{ppoints(length(data))}.  These are used to construct central intervals of whatever proportions are given by \code{centralPercents}.  
#' @param centralPercents The vector of proportions determining the central intervals of the empirical distribution of each ordered quantile from the test distribution.  Default is \code{c(0.90, 0.95, 0.99)} corresponding to central 90, 95, and 99\% simulated pointwise confidence intervals for each quantile coming from the test distribution for a sample the same size as \code{data}.The quality of these interval locations typically increases with \code{nreps} and decreases with the probability used for each interval.
#' @param envelope If \code{TRUE} (the default), a grey envelope is plotted showing the central intervals for each quantile as a shade of grey.  The higher is the corresponding probability associated with the interval, the lighter is the shade.  The outermost edges of the envelope are the range of the simulated data from the test distribution.  The envelope thus provides a (pointwise) density estimate of the quantiles drawn from the test distribution for this sample size.   If \code{FALSE} no envelope is drawn.
#' @param drawPercentiles If \code{TRUE} (the default), a pair of curves is plotted to show each of the central intervals as a different line type.  These are plotted over the envelope if \code{envelope} is \code{TRUE}. If \code{FALSE} no simulated percentile curves are drawn.
#' @param drawQuartiles If \code{TRUE} (the default), a pair of curves is plotted to show the quartiles (central 50\% region) of the ordered quantiles simulated from the test distribution.  The median of these is also plotted as a solid line type. These are plotted over the envelope if \code{envelope} is \code{TRUE}.  If \code{FALSE} none of these curves are drawn.
#' @param legend If \code{TRUE} (the default), a legend for the appearance of the simulated ranges of the central intervals is added to the plot.  If \code{FALSE}, no legend appears.
#' @param nexemplars The number of replicate samples to be taken from the test distribution and plotted as a coloured trail on the qqplot.  Each such trail is a sample of the same size as \code{data} but truly coming from the test distribution.  Each trail gives some idea of what the shape of a qqplot would be for a sample of that size from the test distribution. Together, they give some sense of the variability in the plot's shape. 
#' @param plainTrails If \code{TRUE}, then a single grey colour is used for all exemplar trails.  If \code{FALSE} (the default), each exemplar trail is shown in a different colour. 
#' @param alphaTrails The alpha transparency to be used in plotting all exemplar trails. The default is 0.5.  Because the trails will over plot, a lower \code{alphaTrails} value is recommended as \code{nexemplars} increases.
#' @param lwdTrails The graphical line width (\code{lwd}) to be used in plotting all exemplar trails.   The default is 1.  Because the trails will over plot, combining a larger \code{lwdTrails} with \code{envelope = FALSE}, a lower \code{alphaTrails} value larger \code{nexemplars} can give a truer sense of the density of qqplot configurations than with \code{envelope = TRUE}. 
#' @param lineup If \code{TRUE}, the qqplot of \code{data} is randomly located in a grid of \code{nsuspects} plots.  Identical arguments are given to construct all \code{qqtest} plots in the grid.  Assuming the viewer has not seen the qqplot of this \code{data} before, a successful selection of the true \code{data} plot out of the grid of plots corresponds to evidence against the hypothesis that the \code{data} come from the test distribution.  Significance level is 1/\code{nsuspects}. Default corresponds to a (1/20) =  5\% significance level.  Each plot is given a suspect number from 1 to \code{nsuspects} (left to right, top to bottom).  The suspect number of the plot corresponding to the actual \code{data} is returned, slightly obfuscated to help keep the test honest.
#' @param nsuspects The total number of plots to be viewed in the lineup display when \code{lineup} is \code{lineup}.
#' @param col If non-\code{NULL}, \code{col} must be colour to be used for the points in the plot.  If \code{NULL} (the default), an \code{hcl} colour will be used from the values of the arguments \code{h}, \code{c}, \code{l}, and \code{alpha}.
#' @param h The hue of the colour of the points.  Specified as an angle in degrees from 0 to 360 around a colour wheel.  E.g. 0 is red, 120 green, 240 blue,  Default is 260 (a bluish).
#' @param c The chroma of the colour of the points.  Takes values from 0 to an upper bound that is a function of hue, \code{h},  and luminance, \code{l}.  Roughly, for fixed \code{h} and \code{l} the higher the value of \code{c} the greater the intensity of colour.
#' @param l The luminance of the colour of the points.  Takes values from 0 to 100. For any given combination of hue, \code{h},  and chroma, \code{c}, only a subset of this range will be possible.  Roughly, for fixed \code{h} and \code{c} the higher the value of \code{l} the lighter is the colour.
#' @param alpha The alpha transparency of the colour of the points.  Takes values from 0 to 1. Values near 0 are more transparent, values near 1 are more opaque. Alpha values sum when points over plot, giving some indication of density.
#' @param cex The graphical parameter \code{cex} for the size of the points.
#' @param pch The graphical parameter \code{pch} for the point character to be used for the points.  Default is 19, a filled circle.
#' @param xlab The graphical parameter \code{xlab} labelling the x axis of the plot.  If \code{NULL} (the default), an \code{xlab} is created based on the information available from the other arguments to \code{qqtest} about the test distribution.  An empty string will suppress the labelling.
#' @param ylab The graphical parameter \code{ylab} labelling the y axis of the plot.  If \code{NULL} (the default), a \code{ylab} is created based on the information available from the other arguments to \code{qqtest}. An empty string will suppress the labelling.
#' @param xlim The graphical parameter \code{xlim} determining the display limits of the x axis. 
#' @param ylim The graphical parameter \code{ylim} determining the display limits of the y axis. 
#' @param ... Any further graphical parameters to be passed to the \code{plot} function. 
#'
#' @return Displays the qqplot.  If \code{lineup} is \code{TRUE}, it returns a list with the location (\code{TrueLoc}) of the plot that corresponds to \code{data} encoded as a string whose contents need to be evaluated.  This provides some simple obfuscation of the true location so that the visual assessment can be honest.
#' @examples
#' # default qqtest plot
#' qqtest(precip)
#' #
#' # default qqtest plot
#' qqtest(precip, main = "Precipitation (inches/year) in 70 US cities")
#' #
#' # compare qqtest default to qqnorm
#' op <- par(mfrow=c(1,2))
#' qqnorm(precip) ; qqtest(precip)
#' par(op)
#' #
#' #  gaussian - no quartiles, no exemplars
#' qqtest(precip, nexemplars=0, drawQuartiles=FALSE, 
#'        main = "Precipitation (inches/year) in 70 US cities")
#' #
#' #  gaussian - no quartiles, no percentiles, 
#' #             no envelope  just coloured exemplars
#' qqtest(precip, nexemplars=20, envelope=FALSE, 
#'        drawPercentiles= FALSE, drawQuartiles=FALSE,  
#'        main = "Precipitation (inches/year) in 70 US cities")
#' #
#' #  gaussian - no quartiles, no percentiles, no envelope, 
#' #             plain grey trails, wide trails show density
#' qqtest(precip, nexemplars=20, 
#'        lwdTrails=10, plainTrails=TRUE, alphaTrail=0.3,  
#'        envelope=FALSE, drawPercentiles= FALSE, drawQuartiles=FALSE,  
#'        main = "Precipitation (inches/year) in 70 US cities")
#' #
#' #  gaussian - no quartiles, no percentiles, no envelope,  
#' #             colour trails, wide trails show density
#' qqtest(precip, nexemplars=20, lwdTrails=10, col="black",  
#'        plainTrails=FALSE, alphaTrail=0.3,  
#'        envelope=FALSE, drawPercentiles= FALSE, drawQuartiles=FALSE,  
#'        main = "Precipitation (inches/year) in 70 US cities")
#' #
#' #  gaussian - common qqplot like qqnorm
#' qqtest(precip, xAxisAsProbs=FALSE,  yAxisAsProbs=FALSE,  
#'        nexemplars=0,  envelope=FALSE,  
#'        drawPercentiles= FALSE, drawQuartiles=FALSE,  
#'        col="black", main= "Normal Q-Q Plot",  
#'        xlab="Theoretical Quantiles",  
#'        ylab="Precipitation (inches/year) in 70 US cities", 
#'        pch=21)
#' #
#' #  gaussian - traditional qqplot, but now showing in the line up
#' result <- qqtest(precip, nexemplars=0, nreps=0,  
#'                  envelope=FALSE,  
#'                  drawPercentiles= FALSE, drawQuartiles=FALSE, 
#'                  lineup=TRUE, 
#'                  cex=0.75, col="grey20",
#'                  xlab="", ylab="", 
#'                  pch=21)
#' # the location of the real data in the line up can be found by evaluating
#' # the contents of the string
#'  result$TrueLoc
#' #
#' # lognormal 
#' qqtest(precip, dist = "lognormal", 
#'        main = "Precipitation (inches/year) in 70 US cities")
#' #
#' #
#' # Half normal ... using the penicillin data from Daniel(1959)
#' data(penicillin)
#' 
#' qqtest(penicillin, dist = "half-normal")
#'
#' # Or the same again but with significant contrast labelled
#'
#'  
#' with (penicillin, 
#' 	{qqtest(value, yAxisProbs=c(0.1, 0.75, 0.90, 0.95),  
#'          dist="half-normal",
#' 			ylab="Sample cumulative probability",  
#'          xlab="Half-normal cumulative probability")
#' 	 ppAdj <- (1+ppoints(31))/2  # to get half-normals from normal
#' 	 x <- qnorm(ppAdj)
#' 	 valOrder <- order(value)    # need data and rownames in increasing order
#' 	 y <- value[valOrder]
#' 	 tags <- rownames(penicillin)[valOrder]
#' 	 selPoints <- 28:31          # going to label only the largest effects
#' 	 text(x[selPoints], y[selPoints],  
#'        tags[selPoints],  
#'        pos=2, cex=0.75)
#' 	} 
#' )
#' #
#' # student on 3 df
#' qqtest(precip, dist = "student", df = 3,  
#'        main = "Precipitation (inches/year) in 70 US cities")
#' #
#' # chi-squared on 3 df
#' qqtest(precip, dist = "chi-squared", df = 3,  
#'        main = "Precipitation (inches/year) in 70 US cities")
#' #
#' # user supplied qfunction and rfunction -- compare to beta distribution
#' qqtest(precip, 
#'        qfunction=function(p){qbeta(p, 2, 2)},
#'        rfunction=function(n){rbeta(n, 2, 2)}, 
#'        main = "Precipitation (inches/year) in 70 US cities")
#' #
#' #
#' # user supplied qfunction only -- compare to beta distribution
#' qqtest(precip, 
#'        qfunction=function(p){qbeta(p, 2, 2)}, 
#'        main = "Precipitation (inches/year) in 70 US cities")
#' #
#' # comparing data samples
#' # 
#' # Does the sample of beaver2's temperatures look like they
#' # could have come from a distribution shaped like beaver1's? 
#' # 
#'  	qqtest(beaver2[,"temp"], 
#' 		       dataTest=beaver1[,"temp"], 
#' 		       ylab="Beaver 2", xlab="Beaver 1",
#' 		       main="Beaver body temperatures")
#' # 
#' # 
#' # For the famous iris data, does the sample of iris versicolor
#' # appear to have the same (marginal) distributional shape
#' # as does that of iris virginica (to which it is more closely related)?
#' # 
#' op <- par(mfrow=c(2,2))
#' with(iris, {
#' 	qqtest(Sepal.Length[Species=="versicolor"], 
#' 		   dataTest= Sepal.Length[Species=="virginica"], 
#' 		   ylab="versicolor", xlab="virginica",
#' 		   main="Sepal length")
#' 	qqtest(Sepal.Width[Species=="versicolor"], 
#' 		   dataTest= Sepal.Width[Species=="virginica"],
#' 		   ylab="versicolor", xlab="virginica",
#' 		   main="Sepal width")
#' 	qqtest(Petal.Length[Species=="versicolor"], 
#' 		   dataTest=Petal.Length[Species=="virginica"],
#' 		   ylab="versicolor", xlab="virginica",
#' 		   main="Petal length")
#' 	qqtest(Petal.Width[Species=="versicolor"], 
#' 		   dataTest= Petal.Width[Species=="virginica"],
#' 		   ylab="versicolor", xlab="virginica",
#' 		   main="Petal width")
#' 	}
#' 	)
#' par(op)


qqtest <- function (data,
			        dist="gaussian",
			        df=1,
			        qfunction=NULL,
			        rfunction=NULL,
			        dataTest=NULL,
			        xAxisAsProbs = TRUE,
			        yAxisAsProbs = TRUE,
			        xAxisProbs =  c(0.05, 0.25, 0.50, 0.75, 0.95),
			        yAxisProbs = c(0.05, 0.25, 0.50, 0.75, 0.95),
			        nreps=1000,
			        centralPercents=c(0.90, 0.95, 0.99),
			        envelope=TRUE,
			        drawPercentiles =TRUE,
			        drawQuartiles = TRUE,
			        legend=TRUE,
			        nexemplars=10,
			        plainTrails=FALSE,
			        alphaTrails=0.5,
			        lwdTrails=1,
			        lineup=FALSE,
			        nsuspects=20,
			        col=NULL,
			        h=260,
			        c=90,
			        l=60,
			        alpha=1.0,
			        cex=1,
			        pch=19,
			        xlab=NULL,
			        ylab=NULL,
			        xlim=NULL,
			        ylim=NULL,
			        ...
                      ) {
    
    if (is.matrix(data) || is.data.frame(data)) {
            data <- data.matrix(data)[,1]
            }   
	n <- length(data)
	p <- ppoints(n)
	
	# The following stay null if there are none to create   
	reps <- NULL
    exemplars <- NULL
    
	# Has the user supplied some axis labels?
	makeXlabel <- is.null(xlab) 
	makeYlabel <- is.null(ylab)
	#
 				
	# If lineup, don't do all reps yet
	if (lineup) {
		if (!(nsuspects > 0)) {
			warning("nsuspects must be > 0, set to 20 instead")
			nsuspects <- 20
			}
		nrepsInput <- nreps
		nreps <- nsuspects
		nexemplarsInput <- nexemplars
		nexemplars <- 1
	}
	
 
 	#
	# Getting some reps from the appropriate distribution
	
	if (!is.null(dataTest)) {
		## plan is to test the data against the 
		## empirical distribution from dataTest
		qfunction <- function(p){quantile(dataTest,p)}
		rfunction <- function(n){sample(dataTest,n,replace=TRUE)}
		q <- sapply(p,qfunction)
	    xAxisTicksAt <- sapply(xAxisProbs,qfunction)  
		if (nreps > 0) {reps <- sapply(n*nreps,rfunction)}	
		if (nexemplars > 0) {exemplars <- sapply(n*nexemplars,rfunction)}
	    if (makeXlabel) {xlab <- "Empirical distribution"}
	} else {
	if (is.null(qfunction)){
		dist <- substring(dist,1,3)
		if (dist=="nor") {dist <- "gau"}
		if (dist=="gau") {
			q <- qnorm(p)
			xAxisTicksAt <- qnorm(xAxisProbs)
			if (nreps > 0) {reps <- rnorm(n*nreps)}
			if (nexemplars > 0) {exemplars <- rnorm(n*nexemplars)}
			if (makeXlabel) {xlab <- "Gaussian"}
			}

		if (dist=="log") {
			q <- qlnorm(p)
			xAxisTicksAt <- qlnorm(xAxisProbs)
			if (nreps > 0) {reps <- rlnorm(n*nreps)}
			if (nexemplars > 0) {exemplars <- rlnorm(n*nexemplars)}
			if (makeXlabel) {xlab <- "Lognormal"}
			}

		if (dist=="hal") {
			q <- qnorm((1 + p)/2)
			xAxisTicksAt <- qnorm((1 + xAxisProbs)/2 )
			if (nreps > 0) {reps <- abs(rnorm(n*nreps))}
			if (nexemplars > 0) {exemplars <- abs(rnorm(n*nexemplars))}
			if (makeXlabel) {xlab <- "Half-normal"}
			}

		if (dist=="uni") {
			q <- qlnorm(p)
			xAxisTicksAt <- qunif(xAxisProbs)
			if (nreps > 0) {reps <- runif(n*nreps)}
			if (nexemplars > 0) {exemplars <- runif(n*nexemplars)}
			if (makeXlabel) {xlab <- "Uniform"}
			}

		if (dist=="kay") {
			q <- qkay(p,df)
			xAxisTicksAt <- qkay(xAxisProbs,df)
			if (nreps > 0) {reps <- rkay(n*nreps,df)}
			if (nexemplars > 0) {exemplars <- rkay(n* nexemplars,df)}
			if (makeXlabel) {xlab <- paste("K(",df,")", sep="")}
			}

		if (dist=="chi") {
			q <- qchisq(p,df)
			xAxisTicksAt <- qchisq(xAxisProbs,df)
			if (nreps > 0) {reps <- rchisq(n*nreps,df)}
			if (nexemplars > 0) {exemplars <- rchisq(n* nexemplars,df)}
			if (makeXlabel) {xlab <- paste("Chi-squared(",df,")", sep="")}
			}

		if (dist=="stu") {
			q <- qt(p,df)
			xAxisTicksAt <- qt(xAxisProbs,df)
			if (nreps > 0) {reps <- rt(n*nreps,df)}
			if (nexemplars > 0) {exemplars <- rt(n* nexemplars,df)}
			if (makeXlabel) {xlab <- paste("Student t(",df,")", sep="")}
			}
	
	} else if(is.function(qfunction)) {
		## use these functions to do the work
		q <- sapply(p,qfunction)
	    xAxisTicksAt <- sapply(xAxisProbs,qfunction)  
	    if (!is.function(rfunction)) {
	    	rfunction <- function(n) {
	    		sapply(runif(n),qfunction)
	    	}
	    }
		if (nreps > 0) {reps <- sapply(n*nreps,rfunction)}	
		if (nexemplars > 0) {exemplars <- sapply(n*nexemplars,rfunction)}
	    if (makeXlabel) {xlab <- "Hypothetical distribution"}
	    } else {warning("qfunction must be a function that returns quantiles")}
		
	}
	
 # Get the location and scale changes to use for the simulations.
 # Use a robust line fit to get the location scale correction for the
 # data sampled from the test distribution	(tukeyline not good enough)
 if ((nreps > 0) || (nexemplars > 0)) {
 	loc <- 0
 	allscales <- sort(data)/q
 	scale <- allscales[1]
 	# Check for perfect fit to avoid modelling problems
 	ones <- rep(1,length(allscales))
 	ones[allscales==scale] <- 0
 	if (sum(ones) > 0) {
 		# not a perfect fit so use a robust line
 	 	line <- MASS::rlm(sort(data) ~ 1 + q, psi=MASS::psi.bisquare, method="MM")
 	 	loc <- line$coefficients[1]
 	 	scale <- line$coefficients[2]
 	}
 	}
 
 if (nreps > 0) {
 	reps <- loc + scale * reps
 	reps <- array(reps,dim=c(nreps,n))
 }
 
 if (nexemplars > 0) {
 	exemplars <- loc + scale * exemplars
 	exemplars <- array(exemplars,dim=c(nexemplars,n))
 	}
 
 
 # Do lineup plot if asked. ... recursive
 if (lineup){
 	suspects <- reps[1:nsuspects,]
 	trueLoc <- sample(1:nsuspects,1)
 	suspects[trueLoc,] <- data
 	nrow <- floor(sqrt(nsuspects))
 	ncol <- ceiling(sqrt(nsuspects))
 	parOptions <- par(mfrow=c(nrow,ncol))
 	for (i in 1:nsuspects) {
 		qqtest(data = suspects[i,],
                      dist=dist,
                      df=df,
                      qfunction=qfunction,
                      rfunction= rfunction,
                      dataTest=dataTest,
                      xAxisProbs = xAxisProbs,
                      yAxisProbs = yAxisProbs,
                      xAxisAsProbs = xAxisAsProbs,
                      yAxisAsProbs = yAxisAsProbs,
                      nreps= nrepsInput,
                      centralPercents = centralPercents,
                      envelope = envelope,
                      drawPercentiles = drawPercentiles,
                      drawQuartiles = drawQuartiles,
                      legend = legend,
                      nexemplars= nexemplarsInput,
                      plainTrails = plainTrails,
                      alphaTrails = alphaTrails,
                      lwdTrails = lwdTrails,
                      lineup=FALSE,
                      nsuspects=1,
                      col=col,
                      h=h,
                      c=c,
                      l=l,
                      alpha=alpha,
                      cex=cex,
                      pch=pch,
                      xlab = xlab,
                      ylab = ylab,
                      xlim = xlim,
                      ylim = ylim,
                      sub=paste("Suspect", i),
                      ...)
 	}
 	par(parOptions)
 	base <- sample(2:2*nsuspects,1)
 	list(TrueLoc = paste("log(",base^(trueLoc + 2),", base=",base,") - 2",sep=""))
 } else {
 	# Otherwise construct the plot.
 	#
 if (!is.null(reps)) {
 	reps <- apply(reps, 1, sort)
 	
	 if (envelope||drawPercentiles){
	 	Nums <- apply(reps, 1, fivenum)
 		nLevels <- length(centralPercents)
 		centralPercents <- sort(centralPercents)
 		SymmetricAdjust <-(1-centralPercents)/2
		bottomPcts <- (1-centralPercents)-SymmetricAdjust
 		topPcts <-  centralPercents + SymmetricAdjust
 		Pcts <- c(rev(bottomPcts),topPcts)
 		NumsPct <- apply(reps, 1, function(x){quantile(x,Pcts)})
 		
		# defining grey
 		greyhue <- 260
 		greyluminance <- 65
 		greychroma <- 0
 		greyalpha <- 0.6/(nLevels +1)  # plus 1 for the range
 		grey <- grDevices::hcl(h= greyhue, c= greychroma, l= greyluminance, 
 								alpha= greyalpha)
 	
	 	}                 	                    
 }
 
 ## Get the base plot   
 xlim <- if(!is.null(xlim)) {xlim} else {range(q)} 		
 ylim <- if(!is.null(ylim)) {
 				ylim
 				} else {
 				    yrange <- range(data)
 					if (!is.null(reps)) {
 						yrange <- range(c(yrange, range(reps)))
 						} 				
 					if (!is.null(exemplars)) {
 						yrange <- range(c(yrange, range(exemplars)))
 						}
 					yrange 
 				  }
 				  
 				                               
 plot(0,0, 
       xlab = if(makeXlabel) {
       			paste(xlab, if(xAxisAsProbs) {
       						"cumulative probability (on quantile scale)"
       						} else {"quantiles"})
       			} else xlab,
       ylab = if(makeYlabel) {
       			paste(if(yAxisAsProbs) {
       						"Sample cumulative probability (on quantile scale)"
       						} else {"Sample quantiles"})
       			} else ylab,
       col = "white", 
       xlim = xlim,
       ylim = ylim,
       axes = FALSE,
       ...             #other plot parameters
       )

if (xAxisAsProbs) {
	 Axis(side=1, 
	 	  labels = paste(xAxisProbs), 
	      at = xAxisTicksAt) 
	      }   else  {
     Axis(side=1)
		}
 

if (yAxisAsProbs) {
	 Axis(side=2, 
	 	  labels = paste(yAxisProbs), 
	      at = quantile(data,yAxisProbs)) 
	      } else {
     Axis(side=2)
		}
   # put the box around to look standard
   box()

	           
 ## Add the envelope for the replicates 
 
if(envelope && !is.null(reps)) {     
 # first the range  
 polygon(c(q,rev(q)),
         c(Nums[1,],rev(Nums[5,])),
         border=grey,
         col=grey)
 # now the central pointwise "confidence" intervals
 nLevels <- length(centralPercents)
 for (i in 1:nLevels){
 	iLower <- i
 	iUpper <- nLevels*2 - iLower +1
 	polygon(c(q,rev(q)),
         	c(NumsPct[iLower,],rev(NumsPct[iUpper,])),
         	border=grey,
         	col=grey)
	}
  }
 ## Add exemplar trails                         
 if (!is.null(exemplars)&&nexemplars>=1) {
 	if(!plainTrails){
 			for (i in 1:nexemplars){
    	 		lines(q,sort(exemplars[i,]), 
              		  col=grDevices::hcl(h=i*360/nexemplars, c=90, l=60, alpha=alphaTrails),
               		  lwd=lwdTrails)
     			}
 		} else { # grey trails
 			plainCol <- grDevices::hcl(h=260, c=0, l= 90, alpha=alphaTrails)
    			for (i in 1: nexemplars){
    	 			lines(q,sort(exemplars[i,]), 
               		 col=plainCol,
               		 lwd=lwdTrails)
                }
   		}
   }
 
if(drawPercentiles && !is.null(reps)){
 # and the various percentiles 
 if (drawQuartiles) {
 	 lineTypes <- c(3:(nLevels +2))
 	} else{
 		lineTypes <- c(1:nLevels)
 		}
 		
  lineCols <- rep("darkgrey",nLevels)
  for(i in 1:nLevels) {
 	# central intervals
 	lineCols[i] <- "darkgrey" #grDevices::hcl(h= greyhue, c= greychroma, l= greyluminance, alpha= i * greyalpha)
 	# draw the lower line
 	lines(q,NumsPct[i,],col= lineCols[i],lty= lineTypes[i], lwd=2)
    lines(q,NumsPct[2*nLevels-i+1,],col= lineCols[i],lty= lineTypes[i], lwd=2)
    }
 }
 # quartiles
 if(drawQuartiles && !is.null(reps)){
   	# draw quartiles
 	lines(q,Nums[2,],col="black",lty=2)
 	lines(q,Nums[4,],col="black",lty=2)
 	# draw median
 	lines(q,Nums[3,],col="black",lty=1)
 	}
 	
 
 # and optional legend
 if (legend && (drawQuartiles||drawPercentiles) && !is.null(reps)){
 	if (drawPercentiles) {
 		if (drawQuartiles) {
 			legendString <- c(paste(signif(100*rev(centralPercents),3), "% central range", sep=""),
 							  "quartiles", "median")
 			legendLineTypes <- c(lineTypes,2,1)
 			legendCols <- c(lineCols,"black","black")
 		} else {
 			# no quartiles
 			legendString <- paste(signif(100*rev(centralPercents),3), "% central range", sep="")
 			legendLineTypes <- lineTypes
 			legendCols <- lineCols
 		   }
 	} else {
 			# only quartiles 
 			legendString=c("quartiles", "median")
 			legendLineTypes <- c(2,1)
 			legendCols <- c("black","black")
 		}
    # Draw it
    if (envelope && !drawQuartiles) {
    fillCols <- rep(grey, nLevels+1)
    for (i in 2:(nLevels+1)) {
    	fillCols[i] <- grDevices::hcl(h= greyhue, c= greychroma, l= greyluminance, alpha=i * greyalpha)
    }

    	legend("topleft", 
      	  	legend = legendString,
             lwd=1, 
             cex=0.8,
             bty="n",
             lty= legendLineTypes,
             col= legendCols,
             fill=fillCols, 
             border=fillCols,
             text.col="darkgrey",
             title=paste("Simulated ranges", "n =", nreps))
    	
    } else {
    	legend("topleft", 
      	  	legend = legendString,
             lwd=1, 
             cex=0.8,
             bty="n",
             lty= legendLineTypes,
             col= legendCols, 
             text.col="darkgrey",
             title=paste("Simulated ranges", "n =", nreps)) 
             }
   }

 if (legend && !drawQuartiles && !drawPercentiles && envelope && !is.null(reps)){
    # Draw it
    fillCols <- rep(grey, nLevels+1)
    for (i in 2:(nLevels+1)) {
    	fillCols[i] <- grDevices::hcl(h= greyhue, c= greychroma, l= greyluminance, alpha=i * greyalpha)
    }
 	legend("topleft", 
      	  	legend = c("Range",paste(signif(100*rev(centralPercents),3), "% central range", sep="")),
             col=fillCols,
             fill=fillCols, 
             border=fillCols, 
             cex=0.8,
             bty="n",
             text.col="darkgrey",
             title=paste("Simulated ranges", "n =", nreps)) 
             }
   ## And finally the points  
if (is.null(col)){
	plot_colour <- grDevices::hcl(h=h, c=c, l=l, alpha=alpha)
	} else {plot_colour <- col}
 
 points(q,sort(data), 
        col=plot_colour, 
        pch=pch,
        cex=cex)

    } #end of else from #lineup    

 }