% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/detailed_itineraries.R
\name{detailed_itineraries}
\alias{detailed_itineraries}
\title{Calculate detailed itineraries between origin destination pairs}
\usage{
detailed_itineraries(
  r5r_core,
  origins,
  destinations,
  mode = "WALK",
  mode_egress = "WALK",
  departure_datetime = Sys.time(),
  max_walk_dist = Inf,
  max_bike_dist = Inf,
  max_trip_duration = 120L,
  walk_speed = 3.6,
  bike_speed = 12,
  max_rides = 3,
  max_lts = 2,
  shortest_path = TRUE,
  n_threads = Inf,
  verbose = TRUE,
  progress = TRUE,
  drop_geometry = FALSE
)
}
\arguments{
\item{r5r_core}{rJava object to connect with R5 routing engine}

\item{origins, destinations}{a spatial sf POINT object with WGS84 CRS, or a
data.frame containing the columns 'id', 'lon',
'lat'.}

\item{mode}{string. Transport modes allowed for the trips. Defaults to
"WALK". See details for other options.}

\item{mode_egress}{string. Transport mode used after egress from public
transport. It can be either 'WALK', 'BICYCLE', or 'CAR'.
Defaults to "WALK". Ignored when public transport is not
used.}

\item{departure_datetime}{POSIXct object. If working with public transport
networks, please check \code{calendar.txt} within
the GTFS file for valid dates. See details for
further information on how datetimes are parsed.}

\item{max_walk_dist}{numeric. Maximum walking distance (in meters) to access
and egress the transit network, or to make transfers
within the network. Defaults to no restrictions as long
as \code{max_trip_duration} is respected. The max distance is
considered separately for each leg (e.g. if you set
\code{max_walk_dist} to 1000, you could potentially walk up
to 1 km to reach transit, and up to \emph{another} 1 km to
reach the destination after leaving transit). Obs: if you
want to set the maximum walking distance considering
walking-only trips you have to set the \code{max_trip_duration}
accordingly (e.g. to set a distance of 1 km assuming a
walking speed of 3.6 km/h you have to set \code{max_trip_duration = 1 / 3.6 * 60}).}

\item{max_bike_dist}{numeric. Maximum cycling distance (in meters) to access
and egress the transit network. Defaults to no
restrictions as long as \code{max_trip_duration} is respected.
The max distance is considered separately for each leg
(e.g. if you set \code{max_bike_dist} to 1000, you could
potentially cycle up to 1 km to reach transit, and up
to \emph{another}  1 km to reach the destination after leaving
transit). Obs: if you want to set the maximum cycling
distance considering cycling-only trips you have to set
the \code{max_trip_duration} accordingly (e.g. to set a
distance of 5 km assuming a cycling speed of 12 km/h you
have to set \code{max_trip_duration = 5 / 12 * 60}).}

\item{max_trip_duration}{numeric. Maximum trip duration in minutes. Defaults
to 120 minutes (2 hours).}

\item{walk_speed}{numeric. Average walk speed in km/h. Defaults to 3.6 km/h.}

\item{bike_speed}{numeric. Average cycling speed in km/h. Defaults to 12 km/h.}

\item{max_rides}{numeric. The max number of public transport rides allowed in
the same trip. Defaults to 3.}

\item{max_lts}{numeric (between 1 and 4). The maximum level of traffic stress
that cyclists will tolerate. A value of 1 means cyclists will
only travel through the quietest streets, while a value of 4
indicates cyclists can travel through any road. Defaults to 2.
See details for more information.}

\item{shortest_path}{logical. Whether the function should only return the
fastest route alternative (the default) or multiple
alternatives.}

\item{n_threads}{numeric. The number of threads to use in parallel computing.
Defaults to use all available threads (Inf).}

\item{verbose}{logical. \code{TRUE} to show detailed output messages (the default).}

\item{progress}{logical. \code{TRUE} to show a progress counter. Only works when
\code{verbose} is set to \code{FALSE}, so the progress counter does not
interfere with R5's output messages. Setting \code{progress} to \code{TRUE}
may impose a small penalty for computation efficiency, because
the progress counter must be synchronized among all active
threads.}

\item{drop_geometry}{logical. Indicates whether R5 should drop segment's
geometry column. It can be helpful for saving memory.}
}
\value{
A LINESTRING sf with detailed information about the itineraries
between specified origins and destinations. Distances are in meters
and travel times are in minutes.
}
\description{
Fast computation of (multiple) detailed itineraries between one
or many origin destination pairs.
}
\section{Transport modes:}{
R5 allows for multiple combinations of transport modes. The options include:
\subsection{Transit modes}{

TRAM, SUBWAY, RAIL, BUS, FERRY, CABLE_CAR, GONDOLA, FUNICULAR. The option
'TRANSIT' automatically considers all public transport modes available.
}

\subsection{Non transit modes}{

WALK, BICYCLE, CAR, BICYCLE_RENT, CAR_PARK
}
}

\section{max_lts, Maximum Level of Traffic Stress:}{
When cycling is enabled in R5, setting \code{max_lts} will allow cycling only on
streets with a given level of danger/stress. Setting \code{max_lts} to 1, for example,
will allow cycling only on separated bicycle infrastructure or low-traffic
streets; routing will revert to walking when traversing any links with LTS
exceeding 1. Setting \code{max_lts} to 3 will allow cycling on links with LTS 1, 2,
or 3.

The default methodology for assigning LTS values to network edges is based on
commonly tagged attributes of OSM ways. See more info about LTS in the original
documentation of R5 from Conveyal at \url{https://docs.conveyal.com/learn-more/traffic-stress}.
In summary:
\itemize{
\item \strong{LTS 1}: Tolerable for children. This includes low-speed, low-volume streets,
as well as those with separated bicycle facilities (such as parking-protected
lanes or cycle tracks).
\item \strong{LTS 2}: Tolerable for the mainstream adult population. This includes streets
where cyclists have dedicated lanes and only have to interact with traffic at
formal crossing.
\item \strong{LTS 3}: Tolerable for “enthused and confident” cyclists. This includes streets
which may involve close proximity to moderate- or high-speed vehicular traffic.
\item \strong{LTS 4}: Tolerable for only “strong and fearless” cyclists. This includes streets
where cyclists are required to mix with moderate- to high-speed vehicular traffic.
}

For advanced users, you can provide custom LTS values by adding a tag
<key = "lts> to the \code{osm.pbf} file
}

\section{Routing algorithm:}{
The detailed_itineraries function uses an R5-specific extension to the
McRAPTOR routing algorithm to find paths that are optimal or less than
optimal, with some heuristics around multiple access modes, riding the same
patterns, etc. The specific extension to McRAPTOR to do suboptimal
path routing are not documented yet, but a detailed description of base
McRAPTOR can be found in Delling et al (2015).
\itemize{
\item Delling, D., Pajor, T., & Werneck, R. F. (2015). Round-based public transit
routing. Transportation Science, 49(3), 591-604.
}
}

\section{Datetime parsing}{
\code{r5r} ignores the timezone attribute of datetime objects when parsing dates
and times, using the study area's timezone instead. For example, let's say
you are running some calculations using Rio de Janeiro, Brazil, as your study
area. The datetime \code{as.POSIXct("13-05-2019 14:00:00", format = "\%d-\%m-\%Y \%H:\%M:\%S")} will be parsed as May 13th, 2019, 14:00h in
Rio's local time, as expected. But \code{as.POSIXct("13-05-2019 14:00:00", format = "\%d-\%m-\%Y \%H:\%M:\%S", tz = "Europe/Paris")} will also be parsed as
the exact same date and time in Rio's local time, perhaps surprisingly,
ignoring the timezone attribute.
}

\examples{
if (interactive()) {
library(r5r)

# build transport network
data_path <- system.file("extdata/poa", package = "r5r")
r5r_core <- setup_r5(data_path = data_path, temp_dir = TRUE)

# load origin/destination points
points <- read.csv(file.path(data_path, "poa_points_of_interest.csv"))

# inputs
departure_datetime <- as.POSIXct("13-05-2019 14:00:00", format = "\%d-\%m-\%Y \%H:\%M:\%S")

dit <- detailed_itineraries(r5r_core,
                            origins = points[10,],
                            destinations = points[12,],
                            mode = c("WALK", "TRANSIT"),
                            departure_datetime = departure_datetime,
                            max_walk_dist = 1000,
                            max_trip_duration = 120L)

stop_r5(r5r_core)
}
}
\seealso{
Other routing: 
\code{\link{accessibility}()},
\code{\link{travel_time_matrix}()}
}
\concept{routing}
