

#' R Markdown site generator for Radix websites
#'
#' @inheritParams rmarkdown::default_site_generator
#'
#' @export
radix_website <- function(input, encoding = getOption("encoding"), ...) {

  # first great the default site generator
  default <- rmarkdown::default_site_generator(input, encoding, ...)

  # then wrap/delegate to the render and clean functions
  list(
    name = default$name,

    output_dir = default$output_dir,

    render = function(input_file, output_format, envir, quiet, encoding, ...) {

      # get the site config and collections
      config <- site_config(input, encoding)
      site_collections <- site_collections(input, config)

      # check if this is an incremental render
      incremental <- !is.null(input_file)

      # if it's incremental then only render collection specified in 'listing'
      if (incremental) {
        metadata <- yaml_front_matter(input_file, encoding)
        if (!is.null(metadata$listing)) {
          if (is.list(metadata$listing))
            site_collections <- site_collections[names(metadata$listing)]
          else
            site_collections <- site_collections[metadata$listing]
        } else {
          site_collections <- list()
        }
      }

      # enumerate collections
      collections <- enumerate_collections(input, config, site_collections)

      # write metadata (do this now so that pages have access to collection metadata)
      write_collections_metadata(input, collections)
      on.exit(remove_collections_metadata(input, collections), add = TRUE)

      # track site outputs (for moving to the output_dir)
      track_site_outputs(config$output_dir)
      on.exit(remove_site_outputs(), add = TRUE)

      # delegate to default site generator
      result <- default$render(input_file, output_format, envir, quiet, encoding, ...)

      # render collections to the output directory
      render_collections(input, config, collections, quiet)

      # write sitemap
      write_sitemap_xml(input, config)

      # return result
      result
    },

    clean = function() {

      # files generated by default site generator
      generated <- default$clean()

      # if we are generating in-place then add collection metadata and dirs
      config <- site_config(input, encoding)
      if (config$output_dir == ".") {
        # collection output directories
        collections <- site_collections(input, config)
        for (collection in names(collections)) {
          generated <- c(generated,
            file.path(paste0("_", collection), paste0(collection, ".yml")),
            paste0(collection, "/")
          )
        }
        # sitemap
        generated <- c(generated, "sitemap.xml")
      }

      # filter out by existence
      generated[file.exists(file.path(input, generated))]
    }
  )
}


.site_outputs <- new.env(parent = emptyenv())
.site_outputs$files <- c()
.site_outputs$output_dir <- NULL

track_site_outputs <- function(output_dir) {
  .site_outputs$files <- c()
  .site_outputs$output_dir <- output_dir
}

add_site_output <- function(file) {
  .site_outputs$files <- c(.site_outputs$files, file)
  file
}

remove_site_outputs <- function() {

  on.exit({
    .site_outputs$files <- c()
    .site_outputs$output_dir <- NULL
  }, add = TRUE)

  if (.site_outputs$output_dir != ".")
    lapply(.site_outputs$files, file.remove)
}




