/*
 * Decompiled with CFR 0.152.
 */
package org.openscience.cdk.geometry;

import java.awt.Dimension;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import javax.vecmath.Point2d;
import javax.vecmath.Point3d;
import javax.vecmath.Tuple2d;
import javax.vecmath.Tuple3d;
import javax.vecmath.Vector2d;
import javax.vecmath.Vector3d;
import org.openscience.cdk.annotations.TestClass;
import org.openscience.cdk.annotations.TestMethod;
import org.openscience.cdk.exception.CDKException;
import org.openscience.cdk.interfaces.IAtom;
import org.openscience.cdk.interfaces.IAtomContainer;
import org.openscience.cdk.interfaces.IBond;
import org.openscience.cdk.interfaces.IChemModel;
import org.openscience.cdk.interfaces.IReaction;
import org.openscience.cdk.interfaces.IRing;
import org.openscience.cdk.interfaces.IRingSet;
import org.openscience.cdk.tools.ILoggingTool;
import org.openscience.cdk.tools.LoggingToolFactory;
import org.openscience.cdk.tools.manipulator.ChemModelManipulator;
import org.openscience.cdk.tools.manipulator.ReactionManipulator;

@TestClass(value="org.openscience.cdk.geometry.GeometryToolsTest")
public class GeometryTools {
    private static ILoggingTool logger = LoggingToolFactory.createLoggingTool(GeometryTools.class);

    public static void translateAllPositive(IAtomContainer atomCon) {
        double minX = Double.MAX_VALUE;
        double minY = Double.MAX_VALUE;
        for (IAtom atom : atomCon.atoms()) {
            if (atom.getPoint2d() == null) continue;
            if (atom.getPoint2d().x < minX) {
                minX = atom.getPoint2d().x;
            }
            if (!(atom.getPoint2d().y < minY)) continue;
            minY = atom.getPoint2d().y;
        }
        logger.debug((Object)("Translating: minx=" + minX + ", minY=" + minY));
        GeometryTools.translate2D(atomCon, minX * -1.0, minY * -1.0);
    }

    public static void translate2D(IAtomContainer atomCon, double transX, double transY) {
        GeometryTools.translate2D(atomCon, new Vector2d(transX, transY));
    }

    public static void scaleMolecule(IAtomContainer atomCon, Dimension areaDim, double fillFactor) {
        Dimension molDim = GeometryTools.get2DDimension(atomCon);
        double widthFactor = (double)areaDim.width / (double)molDim.width;
        double heightFactor = (double)areaDim.height / (double)molDim.height;
        double scaleFactor = Math.min(widthFactor, heightFactor) * fillFactor;
        GeometryTools.scaleMolecule(atomCon, scaleFactor);
    }

    public static void scaleMolecule(IAtomContainer atomCon, double scaleFactor) {
        for (int i = 0; i < atomCon.getAtomCount(); ++i) {
            if (atomCon.getAtom(i).getPoint2d() == null) continue;
            atomCon.getAtom((int)i).getPoint2d().x *= scaleFactor;
            atomCon.getAtom((int)i).getPoint2d().y *= scaleFactor;
        }
    }

    public static void center(IAtomContainer atomCon, Dimension areaDim) {
        Dimension molDim = GeometryTools.get2DDimension(atomCon);
        int transX = (areaDim.width - molDim.width) / 2;
        int transY = (areaDim.height - molDim.height) / 2;
        GeometryTools.translateAllPositive(atomCon);
        GeometryTools.translate2D(atomCon, new Vector2d((double)transX, (double)transY));
    }

    public static void translate2D(IAtomContainer atomCon, Vector2d vector) {
        for (IAtom atom : atomCon.atoms()) {
            if (atom.getPoint2d() != null) {
                atom.getPoint2d().add((Tuple2d)vector);
                continue;
            }
            logger.warn((Object)"Could not translate atom in 2D space");
        }
    }

    public static void rotate(IAtomContainer atomCon, Point2d center, double angle) {
        double costheta = Math.cos(angle);
        double sintheta = Math.sin(angle);
        for (int i = 0; i < atomCon.getAtomCount(); ++i) {
            IAtom atom = atomCon.getAtom(i);
            Point2d point = atom.getPoint2d();
            double relativex = point.x - center.x;
            double relativey = point.y - center.y;
            point.x = relativex * costheta - relativey * sintheta + center.x;
            point.y = relativex * sintheta + relativey * costheta + center.y;
        }
    }

    public static void rotate(IAtom atom, Point3d p1, Point3d p2, double angle) {
        Point3d r = new Point3d();
        r.x = p2.x - p1.x;
        r.y = p2.y - p1.y;
        r.z = p2.z - p1.z;
        GeometryTools.normalize(r);
        angle = angle * Math.PI / 180.0;
        double costheta = Math.cos(angle);
        double sintheta = Math.sin(angle);
        Point3d p = atom.getPoint3d();
        p.x -= p1.x;
        p.y -= p1.y;
        p.z -= p1.z;
        Point3d q = new Point3d(0.0, 0.0, 0.0);
        q.x += (costheta + (1.0 - costheta) * r.x * r.x) * p.x;
        q.x += ((1.0 - costheta) * r.x * r.y - r.z * sintheta) * p.y;
        q.x += ((1.0 - costheta) * r.x * r.z + r.y * sintheta) * p.z;
        q.y += ((1.0 - costheta) * r.x * r.y + r.z * sintheta) * p.x;
        q.y += (costheta + (1.0 - costheta) * r.y * r.y) * p.y;
        q.y += ((1.0 - costheta) * r.y * r.z - r.x * sintheta) * p.z;
        q.z += ((1.0 - costheta) * r.x * r.z - r.y * sintheta) * p.x;
        q.z += ((1.0 - costheta) * r.y * r.z + r.x * sintheta) * p.y;
        q.z += (costheta + (1.0 - costheta) * r.z * r.z) * p.z;
        q.x += p1.x;
        q.y += p1.y;
        q.z += p1.z;
        atom.setPoint3d(q);
    }

    public static void normalize(Point3d point) {
        double sum = Math.sqrt(point.x * point.x + point.y * point.y + point.z * point.z);
        point.x /= sum;
        point.y /= sum;
        point.z /= sum;
    }

    public static Dimension get2DDimension(IAtomContainer atomCon) {
        double[] minmax = GeometryTools.getMinMax(atomCon);
        double maxX = minmax[2];
        double maxY = minmax[3];
        double minX = minmax[0];
        double minY = minmax[1];
        return new Dimension((int)(maxX - minX + 1.0), (int)(maxY - minY + 1.0));
    }

    public static Rectangle2D getRectangle2D(IAtomContainer container) {
        double[] minmax = GeometryTools.getMinMax(container);
        return new Rectangle2D.Double(minmax[0], minmax[1], minmax[2] - minmax[0], minmax[3] - minmax[1]);
    }

    public static double[] getMinMax(IAtomContainer container) {
        double maxX = -1.7976931348623157E308;
        double maxY = -1.7976931348623157E308;
        double minX = Double.MAX_VALUE;
        double minY = Double.MAX_VALUE;
        for (int i = 0; i < container.getAtomCount(); ++i) {
            IAtom atom = container.getAtom(i);
            if (atom.getPoint2d() == null) continue;
            if (atom.getPoint2d().x > maxX) {
                maxX = atom.getPoint2d().x;
            }
            if (atom.getPoint2d().x < minX) {
                minX = atom.getPoint2d().x;
            }
            if (atom.getPoint2d().y > maxY) {
                maxY = atom.getPoint2d().y;
            }
            if (!(atom.getPoint2d().y < minY)) continue;
            minY = atom.getPoint2d().y;
        }
        double[] minmax = new double[]{minX, minY, maxX, maxY};
        return minmax;
    }

    public static void translate2DCentreOfMassTo(IAtomContainer atomCon, Point2d p) {
        Point2d com = GeometryTools.get2DCentreOfMass(atomCon);
        Vector2d translation = new Vector2d(p.x - com.x, p.y - com.y);
        for (IAtom atom : atomCon.atoms()) {
            if (atom.getPoint2d() == null) continue;
            atom.getPoint2d().add((Tuple2d)translation);
        }
    }

    public static Point2d get2DCenter(Iterable<IAtom> atoms) {
        double xsum = 0.0;
        double ysum = 0.0;
        int length = 0;
        for (IAtom atom : atoms) {
            if (atom.getPoint2d() == null) continue;
            xsum += atom.getPoint2d().x;
            ysum += atom.getPoint2d().y;
            ++length;
        }
        return new Point2d(xsum / (double)length, ysum / (double)length);
    }

    public static Point2d get2DCenter(Iterator<IAtom> atoms) {
        double xsum = 0.0;
        double ysum = 0.0;
        int length = 0;
        while (atoms.hasNext()) {
            IAtom atom = atoms.next();
            if (atom.getPoint2d() != null) {
                xsum += atom.getPoint2d().x;
                ysum += atom.getPoint2d().y;
            }
            ++length;
        }
        return new Point2d(xsum / (double)length, ysum / (double)length);
    }

    public static Point2d get2DCenter(IRingSet ringSet) {
        double centerX = 0.0;
        double centerY = 0.0;
        for (int i = 0; i < ringSet.getAtomContainerCount(); ++i) {
            Point2d centerPoint = GeometryTools.get2DCenter((IAtomContainer)((IRing)ringSet.getAtomContainer(i)));
            centerX += centerPoint.x;
            centerY += centerPoint.y;
        }
        return new Point2d(centerX / (double)ringSet.getAtomContainerCount(), centerY / (double)ringSet.getAtomContainerCount());
    }

    public static Point2d get2DCentreOfMass(IAtomContainer ac) {
        double xsum = 0.0;
        double ysum = 0.0;
        double totalmass = 0.0;
        for (IAtom a : ac.atoms()) {
            Double mass = a.getExactMass();
            if (mass == null) {
                return null;
            }
            totalmass += mass.doubleValue();
            xsum += mass * a.getPoint2d().x;
            ysum += mass * a.getPoint2d().y;
        }
        return new Point2d(xsum / totalmass, ysum / totalmass);
    }

    public static Point2d get2DCenter(IAtomContainer container) {
        double centerX = 0.0;
        double centerY = 0.0;
        double counter = 0.0;
        for (IAtom atom : container.atoms()) {
            if (atom.getPoint2d() == null) continue;
            centerX += atom.getPoint2d().x;
            centerY += atom.getPoint2d().y;
            counter += 1.0;
        }
        return new Point2d(centerX / counter, centerY / counter);
    }

    public static void translate2DCenterTo(IAtomContainer container, Point2d p) {
        Point2d com = GeometryTools.get2DCenter(container);
        Vector2d translation = new Vector2d(p.x - com.x, p.y - com.y);
        for (IAtom atom : container.atoms()) {
            if (atom.getPoint2d() == null) continue;
            atom.getPoint2d().add((Tuple2d)translation);
        }
    }

    public static Point3d get3DCentreOfMass(IAtomContainer ac) {
        double xsum = 0.0;
        double ysum = 0.0;
        double zsum = 0.0;
        double totalmass = 0.0;
        for (IAtom a : ac.atoms()) {
            Double mass = a.getExactMass();
            if (a.getPoint3d() == null) {
                return null;
            }
            if (mass == null) {
                return null;
            }
            totalmass += mass.doubleValue();
            xsum += mass * a.getPoint3d().x;
            ysum += mass * a.getPoint3d().y;
            zsum += mass * a.getPoint3d().z;
        }
        return new Point3d(xsum / totalmass, ysum / totalmass, zsum / totalmass);
    }

    public static Point3d get3DCenter(IAtomContainer ac) {
        double centerX = 0.0;
        double centerY = 0.0;
        double centerZ = 0.0;
        double counter = 0.0;
        for (IAtom atom : ac.atoms()) {
            if (atom.getPoint3d() == null) continue;
            centerX += atom.getPoint3d().x;
            centerY += atom.getPoint3d().y;
            centerZ += atom.getPoint3d().z;
            counter += 1.0;
        }
        return new Point3d(centerX / counter, centerY / counter, centerZ / counter);
    }

    public static double getAngle(double xDiff, double yDiff) {
        double angle = 0.0;
        if (xDiff >= 0.0 && yDiff >= 0.0) {
            angle = Math.atan(yDiff / xDiff);
        } else if (xDiff < 0.0 && yDiff >= 0.0) {
            angle = Math.PI + Math.atan(yDiff / xDiff);
        } else if (xDiff < 0.0 && yDiff < 0.0) {
            angle = Math.PI + Math.atan(yDiff / xDiff);
        } else if (xDiff >= 0.0 && yDiff < 0.0) {
            angle = Math.PI * 2 + Math.atan(yDiff / xDiff);
        }
        return angle;
    }

    public static int[] distanceCalculator(int[] coords, double dist) {
        double angle = coords[2] - coords[0] == 0 ? 1.5707963267948966 : Math.atan(((double)coords[3] - (double)coords[1]) / ((double)coords[2] - (double)coords[0]));
        int begin1X = (int)(Math.cos(angle + 1.5707963267948966) * dist + (double)coords[0]);
        int begin1Y = (int)(Math.sin(angle + 1.5707963267948966) * dist + (double)coords[1]);
        int begin2X = (int)(Math.cos(angle - 1.5707963267948966) * dist + (double)coords[0]);
        int begin2Y = (int)(Math.sin(angle - 1.5707963267948966) * dist + (double)coords[1]);
        int end1X = (int)(Math.cos(angle - 1.5707963267948966) * dist + (double)coords[2]);
        int end1Y = (int)(Math.sin(angle - 1.5707963267948966) * dist + (double)coords[3]);
        int end2X = (int)(Math.cos(angle + 1.5707963267948966) * dist + (double)coords[2]);
        int end2Y = (int)(Math.sin(angle + 1.5707963267948966) * dist + (double)coords[3]);
        return new int[]{begin1X, begin1Y, begin2X, begin2Y, end1X, end1Y, end2X, end2Y};
    }

    public static double[] distanceCalculator(double[] coords, double dist) {
        double angle = coords[2] - coords[0] == 0.0 ? 1.5707963267948966 : Math.atan((coords[3] - coords[1]) / (coords[2] - coords[0]));
        double begin1X = Math.cos(angle + 1.5707963267948966) * dist + coords[0];
        double begin1Y = Math.sin(angle + 1.5707963267948966) * dist + coords[1];
        double begin2X = Math.cos(angle - 1.5707963267948966) * dist + coords[0];
        double begin2Y = Math.sin(angle - 1.5707963267948966) * dist + coords[1];
        double end1X = Math.cos(angle - 1.5707963267948966) * dist + coords[2];
        double end1Y = Math.sin(angle - 1.5707963267948966) * dist + coords[3];
        double end2X = Math.cos(angle + 1.5707963267948966) * dist + coords[2];
        double end2Y = Math.sin(angle + 1.5707963267948966) * dist + coords[3];
        return new double[]{begin1X, begin1Y, begin2X, begin2Y, end1X, end1Y, end2X, end2Y};
    }

    public static int[] getBondCoordinates(IBond bond) {
        if (bond.getAtom(0).getPoint2d() == null || bond.getAtom(1).getPoint2d() == null) {
            logger.error((Object)"getBondCoordinates() called on Bond without 2D coordinates!");
            return new int[0];
        }
        int beginX = (int)bond.getAtom((int)0).getPoint2d().x;
        int endX = (int)bond.getAtom((int)1).getPoint2d().x;
        int beginY = (int)bond.getAtom((int)0).getPoint2d().y;
        int endY = (int)bond.getAtom((int)1).getPoint2d().y;
        return new int[]{beginX, beginY, endX, endY};
    }

    public static IAtom getClosestAtom(int xPosition, int yPosition, IAtomContainer atomCon) {
        IAtom closestAtom = null;
        double smallestMouseDistance = -1.0;
        for (int i = 0; i < atomCon.getAtomCount(); ++i) {
            IAtom currentAtom = atomCon.getAtom(i);
            double atomX = currentAtom.getPoint2d().x;
            double atomY = currentAtom.getPoint2d().y;
            double mouseDistance = Math.sqrt(Math.pow(atomX - (double)xPosition, 2.0) + Math.pow(atomY - (double)yPosition, 2.0));
            if (!(mouseDistance < smallestMouseDistance) && smallestMouseDistance != -1.0) continue;
            smallestMouseDistance = mouseDistance;
            closestAtom = currentAtom;
        }
        return closestAtom;
    }

    public static IAtom getClosestAtom(IAtomContainer atomCon, IAtom atom) {
        IAtom closestAtom = null;
        double min = Double.MAX_VALUE;
        Point2d atomPosition = atom.getPoint2d();
        for (int i = 0; i < atomCon.getAtomCount(); ++i) {
            double d;
            IAtom currentAtom = atomCon.getAtom(i);
            if (currentAtom == atom || !((d = atomPosition.distance(currentAtom.getPoint2d())) < min)) continue;
            min = d;
            closestAtom = currentAtom;
        }
        return closestAtom;
    }

    public static IAtom getClosestAtom(double xPosition, double yPosition, IAtomContainer atomCon, IAtom toignore) {
        IAtom closestAtom = null;
        double smallestSquaredMouseDistance = -1.0;
        for (int i = 0; i < atomCon.getAtomCount(); ++i) {
            IAtom currentAtom = atomCon.getAtom(i);
            if (currentAtom == toignore) continue;
            double atomX = currentAtom.getPoint2d().x;
            double atomY = currentAtom.getPoint2d().y;
            double mouseSquaredDistance = Math.pow(atomX - xPosition, 2.0) + Math.pow(atomY - yPosition, 2.0);
            if (!(mouseSquaredDistance < smallestSquaredMouseDistance) && smallestSquaredMouseDistance != -1.0) continue;
            smallestSquaredMouseDistance = mouseSquaredDistance;
            closestAtom = currentAtom;
        }
        return closestAtom;
    }

    public static IAtom getClosestAtom(double xPosition, double yPosition, IAtomContainer atomCon) {
        IAtom closestAtom = null;
        double smallestMouseDistance = -1.0;
        for (int i = 0; i < atomCon.getAtomCount(); ++i) {
            IAtom currentAtom = atomCon.getAtom(i);
            double atomX = currentAtom.getPoint2d().x;
            double atomY = currentAtom.getPoint2d().y;
            double mouseDistance = Math.sqrt(Math.pow(atomX - xPosition, 2.0) + Math.pow(atomY - yPosition, 2.0));
            if (!(mouseDistance < smallestMouseDistance) && smallestMouseDistance != -1.0) continue;
            smallestMouseDistance = mouseDistance;
            closestAtom = currentAtom;
        }
        return closestAtom;
    }

    public static IBond getClosestBond(int xPosition, int yPosition, IAtomContainer atomCon) {
        IBond closestBond = null;
        double smallestMouseDistance = -1.0;
        for (IBond currentBond : atomCon.bonds()) {
            Point2d bondCenter = GeometryTools.get2DCenter(currentBond.atoms());
            double mouseDistance = Math.sqrt(Math.pow(bondCenter.x - (double)xPosition, 2.0) + Math.pow(bondCenter.y - (double)yPosition, 2.0));
            if (!(mouseDistance < smallestMouseDistance) && smallestMouseDistance != -1.0) continue;
            smallestMouseDistance = mouseDistance;
            closestBond = currentBond;
        }
        return closestBond;
    }

    public static IBond getClosestBond(double xPosition, double yPosition, IAtomContainer atomCon) {
        IBond closestBond = null;
        double smallestMouseDistance = -1.0;
        for (IBond currentBond : atomCon.bonds()) {
            Point2d bondCenter = GeometryTools.get2DCenter(currentBond.atoms());
            double mouseDistance = Math.sqrt(Math.pow(bondCenter.x - xPosition, 2.0) + Math.pow(bondCenter.y - yPosition, 2.0));
            if (!(mouseDistance < smallestMouseDistance) && smallestMouseDistance != -1.0) continue;
            smallestMouseDistance = mouseDistance;
            closestBond = currentBond;
        }
        return closestBond;
    }

    public static void sortBy2DDistance(IAtom[] atoms, Point2d point) {
        boolean doneSomething;
        do {
            doneSomething = false;
            for (int f = 0; f < atoms.length - 1; ++f) {
                IAtom atom1 = atoms[f];
                IAtom atom2 = atoms[f + 1];
                double distance1 = point.distance(atom1.getPoint2d());
                double distance2 = point.distance(atom2.getPoint2d());
                if (!(distance2 < distance1)) continue;
                atoms[f] = atom2;
                atoms[f + 1] = atom1;
                doneSomething = true;
            }
        } while (doneSomething);
    }

    public static double getScaleFactor(IAtomContainer container, double bondLength) {
        double currentAverageBondLength = GeometryTools.getBondLengthAverage(container);
        if (currentAverageBondLength == 0.0 || Double.isNaN(currentAverageBondLength)) {
            return 1.0;
        }
        return bondLength / currentAverageBondLength;
    }

    public static double getBondLengthAverage(IAtomContainer container) {
        double bondLengthSum = 0.0;
        Iterator bonds = container.bonds().iterator();
        int bondCounter = 0;
        while (bonds.hasNext()) {
            IBond bond = (IBond)bonds.next();
            IAtom atom1 = bond.getAtom(0);
            IAtom atom2 = bond.getAtom(1);
            if (atom1.getPoint2d() == null || atom2.getPoint2d() == null) continue;
            ++bondCounter;
            bondLengthSum += GeometryTools.getLength2D(bond);
        }
        return bondLengthSum / (double)bondCounter;
    }

    public static double getLength2D(IBond bond) {
        if (bond.getAtom(0) == null || bond.getAtom(1) == null) {
            return 0.0;
        }
        Point2d point1 = bond.getAtom(0).getPoint2d();
        Point2d point2 = bond.getAtom(1).getPoint2d();
        if (point1 == null || point2 == null) {
            return 0.0;
        }
        return point1.distance(point2);
    }

    public static boolean has2DCoordinates(IAtomContainer container) {
        if (container == null || container.getAtomCount() == 0) {
            return Boolean.FALSE;
        }
        for (IAtom atom : container.atoms()) {
            if (atom != null && atom.getPoint2d() != null) continue;
            return Boolean.FALSE;
        }
        return Boolean.TRUE;
    }

    @TestMethod(value="get2DCoordinateCoverage_EmptyAtomContainer,get2DCoordinateCoverage_Partial,get2DCoordinateCoverage_Full")
    public static CoordinateCoverage get2DCoordinateCoverage(IAtomContainer container) {
        if (container == null || container.getAtomCount() == 0) {
            return CoordinateCoverage.NONE;
        }
        int count = 0;
        for (IAtom atom : container.atoms()) {
            count += atom != null && atom.getPoint2d() != null ? 1 : 0;
        }
        return count == 0 ? CoordinateCoverage.NONE : (count == container.getAtomCount() ? CoordinateCoverage.FULL : CoordinateCoverage.PARTIAL);
    }

    @Deprecated
    public static int has2DCoordinatesNew(IAtomContainer container) {
        if (container == null) {
            return 0;
        }
        boolean no2d = false;
        boolean with2d = false;
        for (IAtom atom : container.atoms()) {
            if (atom.getPoint2d() == null) {
                no2d = true;
                continue;
            }
            with2d = true;
        }
        if (!no2d && with2d) {
            return 2;
        }
        if (no2d && with2d) {
            return 1;
        }
        return 0;
    }

    public static boolean has2DCoordinates(IAtom atom) {
        return atom.getPoint2d() != null;
    }

    public static boolean has2DCoordinates(IBond bond) {
        for (IAtom iAtom : bond.atoms()) {
            if (iAtom.getPoint2d() != null) continue;
            return false;
        }
        return true;
    }

    @TestMethod(value="testHas3DCoordinates_IAtomContainer,testHas3DCoordinates_EmptyAtomContainer")
    public static boolean has3DCoordinates(IAtomContainer container) {
        if (container == null || container.getAtomCount() == 0) {
            return Boolean.FALSE;
        }
        for (IAtom atom : container.atoms()) {
            if (atom != null && atom.getPoint3d() != null) continue;
            return Boolean.FALSE;
        }
        return Boolean.TRUE;
    }

    @TestMethod(value="get3DCoordinateCoverage_EmptyAtomContainer,get3DCoordinateCoverage_Partial,get3DCoordinateCoverage_Full")
    public static CoordinateCoverage get3DCoordinateCoverage(IAtomContainer container) {
        if (container == null || container.getAtomCount() == 0) {
            return CoordinateCoverage.NONE;
        }
        int count = 0;
        for (IAtom atom : container.atoms()) {
            count += atom != null && atom.getPoint3d() != null ? 1 : 0;
        }
        return count == 0 ? CoordinateCoverage.NONE : (count == container.getAtomCount() ? CoordinateCoverage.FULL : CoordinateCoverage.PARTIAL);
    }

    public static Vector2d calculatePerpendicularUnitVector(Point2d point1, Point2d point2) {
        Vector2d vector = new Vector2d();
        vector.sub((Tuple2d)point2, (Tuple2d)point1);
        vector.normalize();
        return new Vector2d(-1.0 * vector.y, vector.x);
    }

    public static double getNormalizationFactor(IAtomContainer container) {
        double bondlength = 0.0;
        double desiredBondLength = 1.5;
        int counter = 0;
        for (IBond bond : container.bonds()) {
            if (bond.getAtomCount() != 2) continue;
            ++counter;
            IAtom atom1 = bond.getAtom(0);
            IAtom atom2 = bond.getAtom(1);
            bondlength += Math.sqrt(Math.pow(atom1.getPoint2d().x - atom2.getPoint2d().x, 2.0) + Math.pow(atom1.getPoint2d().y - atom2.getPoint2d().y, 2.0));
        }
        double ratio = desiredBondLength / (bondlength /= (double)counter);
        return ratio;
    }

    public static int getBestAlignmentForLabel(IAtomContainer container, IAtom atom) {
        double overallDiffX = 0.0;
        for (IAtom connectedAtom : container.getConnectedAtomsList(atom)) {
            overallDiffX += connectedAtom.getPoint2d().x - atom.getPoint2d().x;
        }
        if (overallDiffX <= 0.0) {
            return 1;
        }
        return -1;
    }

    public static int getBestAlignmentForLabelXY(IAtomContainer container, IAtom atom) {
        double overallDiffX = 0.0;
        double overallDiffY = 0.0;
        for (IAtom connectedAtom : container.getConnectedAtomsList(atom)) {
            overallDiffX += connectedAtom.getPoint2d().x - atom.getPoint2d().x;
            overallDiffY += connectedAtom.getPoint2d().y - atom.getPoint2d().y;
        }
        if (Math.abs(overallDiffY) > Math.abs(overallDiffX)) {
            if (overallDiffY < 0.0) {
                return 2;
            }
            return -2;
        }
        if (overallDiffX <= 0.0) {
            return 1;
        }
        return -1;
    }

    public static List<IAtom> findClosestInSpace(IAtomContainer container, IAtom startAtom, int max) throws CDKException {
        Point3d originalPoint = startAtom.getPoint3d();
        if (originalPoint == null) {
            throw new CDKException("No point3d, but findClosestInSpace is working on point3ds");
        }
        TreeMap<Double, IAtom> atomsByDistance = new TreeMap<Double, IAtom>();
        for (IAtom atom : container.atoms()) {
            if (atom == startAtom) continue;
            if (atom.getPoint3d() == null) {
                throw new CDKException("No point3d, but findClosestInSpace is working on point3ds");
            }
            double distance = atom.getPoint3d().distance(originalPoint);
            atomsByDistance.put(distance, atom);
        }
        Set keySet = atomsByDistance.keySet();
        Iterator keyIter = keySet.iterator();
        ArrayList<IAtom> returnValue = new ArrayList<IAtom>();
        for (int i = 0; keyIter.hasNext() && i < max; ++i) {
            returnValue.add((IAtom)atomsByDistance.get(keyIter.next()));
        }
        return returnValue;
    }

    public static Map<Integer, Integer> mapAtomsOfAlignedStructures(IAtomContainer firstAtomContainer, IAtomContainer secondAtomContainer, double searchRadius, Map<Integer, Integer> mappedAtoms) throws CDKException {
        GeometryTools.getLargestAtomContainer(firstAtomContainer, secondAtomContainer);
        double[][] distanceMatrix = new double[firstAtomContainer.getAtomCount()][secondAtomContainer.getAtomCount()];
        for (int i = 0; i < firstAtomContainer.getAtomCount(); ++i) {
            Point3d firstAtomPoint = firstAtomContainer.getAtom(i).getPoint3d();
            for (int j = 0; j < secondAtomContainer.getAtomCount(); ++j) {
                distanceMatrix[i][j] = firstAtomPoint.distance(secondAtomContainer.getAtom(j).getPoint3d());
            }
        }
        for (int i = 0; i < firstAtomContainer.getAtomCount(); ++i) {
            double minimumDistance = searchRadius;
            for (int j = 0; j < secondAtomContainer.getAtomCount(); ++j) {
                if (!(distanceMatrix[i][j] < searchRadius) || !(distanceMatrix[i][j] < minimumDistance) || !GeometryTools.checkAtomMapping(firstAtomContainer, secondAtomContainer, i, j)) continue;
                minimumDistance = distanceMatrix[i][j];
                mappedAtoms.put(firstAtomContainer.getAtomNumber(firstAtomContainer.getAtom(i)), secondAtomContainer.getAtomNumber(secondAtomContainer.getAtom(j)));
            }
        }
        return mappedAtoms;
    }

    private static void getLargestAtomContainer(IAtomContainer firstAC, IAtomContainer secondAC) {
        if (firstAC.getAtomCount() < secondAC.getAtomCount()) {
            try {
                IAtomContainer tmp = firstAC.clone();
                firstAC = secondAC.clone();
                secondAC = tmp.clone();
            }
            catch (CloneNotSupportedException e) {
                e.printStackTrace();
            }
        }
    }

    private static boolean checkAtomMapping(IAtomContainer firstAC, IAtomContainer secondAC, int posFirstAtom, int posSecondAtom) {
        IAtom firstAtom = firstAC.getAtom(posFirstAtom);
        IAtom secondAtom = secondAC.getAtom(posSecondAtom);
        return firstAtom.getSymbol().equals(secondAtom.getSymbol()) && firstAC.getConnectedAtomsList(firstAtom).size() == secondAC.getConnectedAtomsList(secondAtom).size() && firstAtom.getBondOrderSum() == secondAtom.getBondOrderSum() && firstAtom.getMaxBondOrder() == secondAtom.getMaxBondOrder();
    }

    private static IAtomContainer setVisitedFlagsToFalse(IAtomContainer atomContainer) {
        for (int i = 0; i < atomContainer.getAtomCount(); ++i) {
            atomContainer.getAtom(i).setFlag(16, false);
        }
        return atomContainer;
    }

    public static double getBondLengthRMSD(IAtomContainer firstAtomContainer, IAtomContainer secondAtomContainer, Map<Integer, Integer> mappedAtoms, boolean Coords3d) {
        Iterator<Integer> firstAtoms = mappedAtoms.keySet().iterator();
        double sum = 0.0;
        double n = 0.0;
        double distance1 = 0.0;
        double distance2 = 0.0;
        GeometryTools.setVisitedFlagsToFalse(firstAtomContainer);
        GeometryTools.setVisitedFlagsToFalse(secondAtomContainer);
        while (firstAtoms.hasNext()) {
            IAtom centerAtomFirstMolecule = firstAtomContainer.getAtom(firstAtoms.next().intValue());
            centerAtomFirstMolecule.setFlag(16, true);
            IAtom centerAtomSecondMolecule = secondAtomContainer.getAtom(mappedAtoms.get(firstAtomContainer.getAtomNumber(centerAtomFirstMolecule)).intValue());
            List connectedAtoms = firstAtomContainer.getConnectedAtomsList(centerAtomFirstMolecule);
            for (int i = 0; i < connectedAtoms.size(); ++i) {
                IAtom conAtom = (IAtom)connectedAtoms.get(i);
                if (conAtom.getFlag(16)) continue;
                if (Coords3d) {
                    distance1 = centerAtomFirstMolecule.getPoint3d().distance(conAtom.getPoint3d());
                    distance2 = centerAtomSecondMolecule.getPoint3d().distance(secondAtomContainer.getAtom(mappedAtoms.get(firstAtomContainer.getAtomNumber(conAtom)).intValue()).getPoint3d());
                    sum += Math.pow(distance1 - distance2, 2.0);
                    n += 1.0;
                    continue;
                }
                distance1 = centerAtomFirstMolecule.getPoint2d().distance(conAtom.getPoint2d());
                distance2 = centerAtomSecondMolecule.getPoint2d().distance(secondAtomContainer.getAtom(mappedAtoms.get(firstAtomContainer.getAtomNumber(conAtom)).intValue()).getPoint2d());
                sum += Math.pow(distance1 - distance2, 2.0);
                n += 1.0;
            }
        }
        GeometryTools.setVisitedFlagsToFalse(firstAtomContainer);
        GeometryTools.setVisitedFlagsToFalse(secondAtomContainer);
        return Math.sqrt(sum / n);
    }

    public static double getAngleRMSD(IAtomContainer firstAtomContainer, IAtomContainer secondAtomContainer, Map<Integer, Integer> mappedAtoms) {
        Iterator<Integer> firstAtoms = mappedAtoms.keySet().iterator();
        double sum = 0.0;
        double n = 0.0;
        while (firstAtoms.hasNext()) {
            int firstAtomNumber = firstAtoms.next();
            IAtom centerAtomfirstAC = firstAtomContainer.getAtom(firstAtomNumber);
            List connectedAtoms = firstAtomContainer.getConnectedAtomsList(centerAtomfirstAC);
            if (connectedAtoms.size() <= 1) continue;
            for (int i = 0; i < connectedAtoms.size() - 1; ++i) {
                IAtom firstAtomfirstAC = (IAtom)connectedAtoms.get(i);
                for (int j = i + 1; j < connectedAtoms.size(); ++j) {
                    double angleFirstMolecule = GeometryTools.getAngle(centerAtomfirstAC, firstAtomfirstAC, (IAtom)connectedAtoms.get(j));
                    IAtom centerAtomsecondAC = secondAtomContainer.getAtom(mappedAtoms.get(firstAtomContainer.getAtomNumber(centerAtomfirstAC)).intValue());
                    IAtom firstAtomsecondAC = secondAtomContainer.getAtom(mappedAtoms.get(firstAtomContainer.getAtomNumber(firstAtomfirstAC)).intValue());
                    IAtom secondAtomsecondAC = secondAtomContainer.getAtom(mappedAtoms.get(firstAtomContainer.getAtomNumber((IAtom)connectedAtoms.get(j))).intValue());
                    double angleSecondMolecule = GeometryTools.getAngle(centerAtomsecondAC, firstAtomsecondAC, secondAtomsecondAC);
                    sum += Math.pow(angleFirstMolecule - angleSecondMolecule, 2.0);
                    n += 1.0;
                }
            }
        }
        return Math.sqrt(sum / n);
    }

    private static double getAngle(IAtom atom1, IAtom atom2, IAtom atom3) {
        Vector3d centerAtom = new Vector3d();
        centerAtom.x = atom1.getPoint3d().x;
        centerAtom.y = atom1.getPoint3d().y;
        centerAtom.z = atom1.getPoint3d().z;
        Vector3d firstAtom = new Vector3d();
        Vector3d secondAtom = new Vector3d();
        firstAtom.x = atom2.getPoint3d().x;
        firstAtom.y = atom2.getPoint3d().y;
        firstAtom.z = atom2.getPoint3d().z;
        secondAtom.x = atom3.getPoint3d().x;
        secondAtom.y = atom3.getPoint3d().y;
        secondAtom.z = atom3.getPoint3d().z;
        firstAtom.sub((Tuple3d)centerAtom);
        secondAtom.sub((Tuple3d)centerAtom);
        return firstAtom.angle(secondAtom);
    }

    public static double getAllAtomRMSD(IAtomContainer firstAtomContainer, IAtomContainer secondAtomContainer, Map<Integer, Integer> mappedAtoms, boolean Coords3d) throws CDKException {
        double sum = 0.0;
        Iterator<Integer> firstAtoms = mappedAtoms.keySet().iterator();
        int n = 0;
        while (firstAtoms.hasNext()) {
            int firstAtomNumber = firstAtoms.next();
            try {
                int secondAtomNumber = mappedAtoms.get(firstAtomNumber);
                IAtom firstAtom = firstAtomContainer.getAtom(firstAtomNumber);
                if (Coords3d) {
                    sum += Math.pow(firstAtom.getPoint3d().distance(secondAtomContainer.getAtom(secondAtomNumber).getPoint3d()), 2.0);
                    ++n;
                    continue;
                }
                sum += Math.pow(firstAtom.getPoint2d().distance(secondAtomContainer.getAtom(secondAtomNumber).getPoint2d()), 2.0);
                ++n;
            }
            catch (Exception ex) {
                throw new CDKException(ex.getMessage(), (Throwable)ex);
            }
        }
        double RMSD = Math.sqrt(sum / (double)n);
        return RMSD;
    }

    public static double getHeavyAtomRMSD(IAtomContainer firstAtomContainer, IAtomContainer secondAtomContainer, Map<Integer, Integer> mappedAtoms, boolean hetAtomOnly, boolean Coords3d) {
        double sum = 0.0;
        double RMSD = 0.0;
        Iterator<Integer> firstAtoms = mappedAtoms.keySet().iterator();
        int firstAtomNumber = 0;
        int secondAtomNumber = 0;
        int n = 0;
        while (firstAtoms.hasNext()) {
            firstAtomNumber = firstAtoms.next();
            secondAtomNumber = mappedAtoms.get(firstAtomNumber);
            IAtom firstAtom = firstAtomContainer.getAtom(firstAtomNumber);
            if (hetAtomOnly) {
                if (firstAtom.getSymbol().equals("H") || firstAtom.getSymbol().equals("C")) continue;
                if (Coords3d) {
                    sum += Math.pow(firstAtom.getPoint3d().distance(secondAtomContainer.getAtom(secondAtomNumber).getPoint3d()), 2.0);
                    ++n;
                    continue;
                }
                sum += Math.pow(firstAtom.getPoint2d().distance(secondAtomContainer.getAtom(secondAtomNumber).getPoint2d()), 2.0);
                ++n;
                continue;
            }
            if (firstAtom.getSymbol().equals("H")) continue;
            if (Coords3d) {
                sum += Math.pow(firstAtom.getPoint3d().distance(secondAtomContainer.getAtom(secondAtomNumber).getPoint3d()), 2.0);
                ++n;
                continue;
            }
            sum += Math.pow(firstAtom.getPoint2d().distance(secondAtomContainer.getAtom(secondAtomNumber).getPoint2d()), 2.0);
            ++n;
        }
        RMSD = Math.sqrt(sum / (double)n);
        return RMSD;
    }

    public static double getBondLengthAverage3D(IAtomContainer container) {
        double bondLengthSum = 0.0;
        int bondCounter = 0;
        for (IBond bond : container.bonds()) {
            IAtom atom1 = bond.getAtom(0);
            IAtom atom2 = bond.getAtom(1);
            if (atom1.getPoint3d() == null || atom2.getPoint3d() == null) continue;
            ++bondCounter;
            bondLengthSum += atom1.getPoint3d().distance(atom2.getPoint3d());
        }
        return bondLengthSum / (double)bondCounter;
    }

    public static Rectangle2D shiftContainer(IAtomContainer container, Rectangle2D bounds, Rectangle2D last, double gap) {
        if (last.getMaxX() + gap >= bounds.getMinX()) {
            double xShift = last.getMaxX() + gap - bounds.getMinX();
            Vector2d shift = new Vector2d(xShift, 0.0);
            GeometryTools.translate2D(container, shift);
            return new Rectangle2D.Double(bounds.getX() + xShift, bounds.getY(), bounds.getWidth(), bounds.getHeight());
        }
        return bounds;
    }

    public static double getBondLengthAverage(IReaction reaction) {
        double bondlenghtsum = 0.0;
        int containercount = 0;
        List<IAtomContainer> containers = ReactionManipulator.getAllAtomContainers(reaction);
        for (IAtomContainer container : containers) {
            ++containercount;
            bondlenghtsum += GeometryTools.getBondLengthAverage(container);
        }
        return bondlenghtsum / (double)containercount;
    }

    public static boolean has3DCoordinates(IChemModel chemModel) {
        List<IAtomContainer> acs = ChemModelManipulator.getAllAtomContainers(chemModel);
        Iterator<IAtomContainer> it = acs.iterator();
        while (it.hasNext()) {
            if (GeometryTools.has3DCoordinates(it.next())) continue;
            return false;
        }
        return true;
    }

    public static Rectangle2D shiftReactionVertical(IReaction reaction, Rectangle2D bounds, Rectangle2D last, double gap) {
        if (last.getMaxY() + gap >= bounds.getMinY()) {
            double yShift = bounds.getHeight() + last.getHeight() + gap;
            Vector2d shift = new Vector2d(0.0, yShift);
            List<IAtomContainer> containers = ReactionManipulator.getAllAtomContainers(reaction);
            for (IAtomContainer container : containers) {
                GeometryTools.translate2D(container, shift);
            }
            return new Rectangle2D.Double(bounds.getX(), bounds.getY() + yShift, bounds.getWidth(), bounds.getHeight());
        }
        return bounds;
    }

    public static enum CoordinateCoverage {
        FULL,
        PARTIAL,
        NONE;

    }
}

