#' @name exportUsers
#' @title Export the Users for a Project
#' 
#' @description Retrieve a data frame giving the users, expiration dates,
#' and data access privileges for each user.
#'
#' @param rcon A REDCap connection object as generated by \code{redcapConnection}.
#' @param dates Logical. Indicates if the expiration date is converted to a
#'   \code{POSIXct} object.
#' @param labels Logical. Indicates if the data export and form access rights are
#'   converted to factor objects.
#' @param ... Arguments to be passed to other methods.
#' @param bundle A \code{redcap_bundle} object.  
#' @param error_handling An option for how to handle errors returned by the API.
#'   see \code{\link{redcap_error}}
#'
#' @details
#' For some reason I have yet to identify, some User Tables do not
#' export correctly. In some situations, the fields are all shifted one
#' column to the left and the form names are not always exported.
#' This seems to be more common in projects still in Development mode.
#' I have seen one instance of a project in Production where one user had
#' one more column given than any other user.  If you notice this behavior,
#' please report it to me as it may help me narrow down the source of the problem
#' 
#' @section REDCap API Documentation (6.5.0):
#' This function allows you to export the users for a project
#' 
#' @section REDCap Version:
#' 5.8.2 (Perhaps earlier) 
#' 
#' @section Known REDCap Limitations:
#' None
#' 
#' @return 
#' Returns a data frame. The number of columns in the data frame will depend on your 
#' version of REDCap.
#' \itemize{
#'   \item{\code{username }}{User name}
#'   \item{\code{email }}{The user's e-mail address}
#'   \item{\code{firstname }}{The user's first name}
#'   \item{\code{lastname }}{The user's last name}
#'   \item{\code{expiration }}{The expiration date of the user's access to the project}
#'   \item{\code{data_access_group }}{The data access group the user is assigned to}
#'   \item{\code{data_export }}{The user's data export rights. 0=no access, 
#'     2=De-Identified, 1=Full Data Set}
#'   \item{\code{mobile_app }}{(6.5.0+) Flag for if the user has permissions for the 
#'     mobile application}
#'   \item{\code{mobile_app_download_data }}{(6.5.0+) Flag for if the user may download
#'     data from the mobile app}
#' }
#' 
#' The data frame will have one additional column for each form giving the user's 
#' form-level permissions in the project.  0=no access, 2=read only, 
#' 1=view records/responses and
#' edit records (survey responses are read-only), 3 = edit survey responses
#'
#' @author Benjamin Nutter
#'
#' @references
#' Please refer to your institution's API documentation.
#'
#' Additional details on API parameters are found on the package wiki at
#' \url{https://github.com/nutterb/redcapAPI/wiki/REDCap-API-Parameters}
#'
#' @export

exportUsers <- function(rcon, ...) UseMethod("exportUsers")

#' @rdname exportUsers
#' @export

exportUsers.redcapDbConnection <- function(rcon, dates=TRUE, labels=TRUE, ...){
  message("Please accept my apologies.  The exportUsers method for redcapDbConnection objects\n",
          "has not yet been written.  Please consider using the API.")
}

#' @rdname exportUsers
#' @export

exportUsers.redcapApiConnection <- function(rcon, dates=TRUE, labels=TRUE, ...,
                                            bundle = getOption("redcap_bundle"),
                                            error_handling = getOption("redcap_error_handling")){
  if (!is.na(match("proj", names(list(...)))))
  {
    message("The 'proj' argument is deprecated.  Please use 'bundle' instead")
    bundle <- list(...)[["proj"]]
  }
  
  coll <- checkmate::makeAssertCollection()
  
  massert(~ rcon + bundle,
          fun = checkmate::assert_class,
          classes = list(rcon = "redcapApiConnection",
                         bundle = "redcapBundle"),
          null.ok = list(rcon = FALSE, 
                         bundle = TRUE),
          fixed = list(add = coll))
  
  massert(~ dates + labels,
          fun = checkmate::assert_logical,
          fixed = list(len = 1,
                       add = coll))
  
  error_handling <- checkmate::matchArg(x = error_handling,
                                        choices = c("null", "error"))
  
  checkmate::reportAssertions(coll)
  
  #* parameters for the Users File Export
  body <- list(token = rcon$token, 
               content = 'user', 
               format = 'csv', 
               returnFormat = 'csv')
  
  #* Export Users file and convert to data frame
  x <- httr::POST(url = rcon$url, 
                  body = body, 
                  config = rcon$config)
  
  if (x$status_code != 200) redcap_error(x, error_handling)
  
  x <- utils::read.csv(textConnection(as.character(x)),
                       stringsAsFactors = FALSE,
                       na.strings = "")
                             
  #* convert expiration date to POSIXct class
  if (dates) x$expiration <- as.POSIXct(x$expiration, format="%Y-%m-%d")
  
  #* convert data export and form editing privileges to factors
  if (labels){
    x$data_export <- 
      factor(x$data_export, 
             levels = c(0, 2, 1), 
             labels = c("No access", "De-identified", "Full data set"))
    
    meta_data <- 
      if (is.null(bundle$meta_data)) 
        exportMetaData(rcon)
      else 
        bundle$meta_data
    
    form_names <- unique(meta_data$form)

    FormAccess <- x$forms
    FormAccess <- lapply(FormAccess, 
                         FUN = strsplit, 
                         split = ",")
    FormAccess <- lapply(FormAccess,
                         FUN = function(x){
                           data.frame(form_access = x[[1]],
                                      stringsAsFactors = FALSE)
                         })
    FormAccess <- lapply(FormAccess,
                         FUN = tidyr::separate,
                         col = form_access,
                         into = c("form", "access"),
                         sep = "[:]")
    FormAccess <- lapply(FormAccess,
                         FUN = function(x){
                           x$form <- paste0("form_", x$form)
                           x$access <- factor(x$access,
                                              levels = c(0, 2, 1, 3),
                                              labels = c("No access", "Read only", 
                                                         "view records/responses and edit records",
                                                         "Edit survey responses"))
                           x
                         }) 
    FormAccess <- lapply(FormAccess,
                         FUN = tidyr::spread,
                         key = form,
                         value = access)
    FormAccess <- do.call("rbind", FormAccess)

    x <- cbind(x, FormAccess)
  }
  
  x
}

utils::globalVariables(c("form_access", "form", "access"))