% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Random_Numbers.R
\name{rtelliptical}
\alias{rtelliptical}
\title{Sampling Random Numbers from Truncated Multivariate Elliptical Distributions}
\usage{
rtelliptical(
  n = 10000,
  mean,
  Sigma = diag(length(mean)),
  lower = rep(-Inf, length(mean)),
  upper = rep(Inf, length(mean)),
  dist = "Normal",
  nu = NULL,
  expr = "exp(1)^(-t)/(1+exp(1)^(-t))^2",
  gFun = NULL,
  ginvFun = NULL,
  burn.in = 0,
  thinning = 1
)
}
\arguments{
\item{n}{number of observations to generate. Must be an integer >= 1.}

\item{mean}{numeric vector of length \eqn{p} representing the location parameter.}

\item{Sigma}{numeric positive definite matrix with dimension \eqn{pxp} representing the
scale parameter.}

\item{lower}{vector of lower truncation points of length \eqn{p}.}

\item{upper}{vector of upper truncation points of length \eqn{p}.}

\item{dist}{represents the truncated distribution to be used. The values are \code{Normal},
\code{t}, \code{PE}, \code{PVII}, \code{Slash} and \code{CN} for the truncated Normal, Student-t,
Power Exponential, Pearson VII, Slash and Contaminated Normal distributions, respectively.}

\item{nu}{additional parameter or vector of parameters depending on the
density generating function. See Details.}

\item{expr}{a character with the density generating function. See Details.}

\item{gFun}{an R function with the density generating function. See Details.}

\item{ginvFun}{an R function with the inverse of the density generating function defined in
\code{gFun}. See Details.}

\item{burn.in}{number of samples to be discarded as a burn-in phase.}

\item{thinning}{factor for reducing the autocorrelation of random points.}
}
\value{
It returns a matrix of dimensions \eqn{nxp} with the random points sampled.
}
\description{
This function generates random numbers from a truncated multivariate elliptical distribution
with location parameter equal to \code{mean}, scale matrix \code{Sigma}, lower and upper
truncation points \code{lower} and \code{upper} via Slice Sampling algorithm with Gibbs sampler steps.
}
\details{
The argument \code{nu} is a parameter or vector of parameters depending on the density generating
function (DGF). For the truncated Student-t, Power Exponential, and Slash distribution, \code{nu} is a
positive number. For the truncated Pearson VII,
\code{nu} is a vector with the first element greater than \eqn{p/2} and the second element a positive number.
For the truncated Contaminated Normal distribution, \code{nu} is a vector of length 2 assuming values between 0 and 1.

This function also allows us to generate random points from other truncated elliptical distributions by specifying
the DGF through arguments \code{expr} or \code{gFun} and making \code{dist} equal to \code{NULL}.
The DGF must be non-negative and strictly decreasing on the interval \code{(0, Inf)}.
The DGF is given as a character to argument \code{expr}. The notation used in \code{expr} needs to be
understood by package \code{Ryacas0} and the environment of R. For example if the DGF is \eqn{g(t)=e^{-t}}, then
\code{expr="exp(1)^(-t)"}. In this case, the algorithm tries to compute a closed expression for the inverse function
of \eqn{g(t)}, a warning message is returned when it is not possible. Additionally, the function in \code{expr} should
depend only on variable \eqn{t}, and any additional parameter must be a fixed value.
Argument \code{gFun} can be accessed when \code{expr='NULL'}. It accepts the DGF as an R function.
The inverse of the function defined in \code{gFun} can be provided as an R function through \code{ginvFun}.
If \code{ginvFun='NULL'}, then the inverse of \code{gFun} is approximated numerically.
In the examples, we show how to use \code{expr} and \code{gFun} to draw samples from the bivariate truncated Logistic
and Kotz-type distributions. Note that the DGF of Kotz-type distribution is strictly decreasing for values of \eqn{N} between
(2-p)/2 and 1, see \insertCite{fang2018symmetric;textual}{relliptical}.
}
\note{
The Normal distribution is a particular case of the Power Exponential distribution when \code{nu = 1}.
The Student-t distribution with \eqn{\nu} degrees of freedom results from the Pearson VII
distribution when \code{nu = } ((\eqn{\nu}+p)/2, \eqn{\nu}).
}
\examples{
library(ggplot2)
library(ggExtra)
library(gridExtra)

# Example 1: Sampling from the Truncated Normal distribution
set.seed(1234)
mean  = c(0, 1)
Sigma = matrix(c(1,0.70,0.70,3), 2, 2)
lower = c(-2, -3)
upper = c(3, 3)
sample1 = rtelliptical(5e4, mean, Sigma, lower, upper, dist="Normal")

# Histogram and density for variable 1
ggplot(data.frame(sample1), aes(x=X1)) +
   geom_histogram(aes(y=..density..), colour="black", fill="grey", bins=15) +
   geom_density(color="red") + labs(x=bquote(X[1]), y="Density")

# Histogram and density for variable 2
ggplot(data.frame(sample1), aes(x=X2)) +
   geom_histogram(aes(y=..density..), colour="black", fill="grey", bins=15) +
   geom_density(color="red") + labs(x=bquote(X[2]), y="Density")


# Example 2: Sampling from the Truncated Logistic distribution

# Function for plotting the sample autocorrelation using ggplot2
acf.plot = function(samples){
 p = ncol(samples); n = nrow(samples); q1 = qnorm(0.975)/sqrt(n); acf1 = list(p)
 for (i in 1:p){
   bacfdf = with(acf(samples[,i], plot=FALSE), data.frame(lag, acf))
   acf1[[i]] = ggplot(data=bacfdf, aes(x=lag,y=acf)) + geom_hline(aes(yintercept=0)) +
     geom_segment(aes(xend=lag, yend=0)) + labs(x="Lag", y="ACF", subtitle=bquote(X[.(i)])) +
     geom_hline(yintercept=c(q1,-q1), color="red", linetype="twodash")
 }
 return (acf1)
}

set.seed(5678)
mean  = c(0, 0)
Sigma = matrix(c(1,0.70,0.70,1), 2, 2)
lower = c(-2, -2)
upper = c(3, 2)
# Sample autocorrelation with no thinning
sample2 = rtelliptical(1000, mean, Sigma, lower, upper, dist=NULL,
                       expr="exp(1)^(-t)/(1+exp(1)^(-t))^2")
grid.arrange(grobs=acf.plot(sample2), top="Logistic distribution with no thinning", nrow=1)

# Sample autocorrelation with thinning = 3
sample3 = rtelliptical(1000, mean, Sigma, lower, upper, dist=NULL,
                       expr="exp(1)^(-t)/(1+exp(1)^(-t))^2", thinning=3)
grid.arrange(grobs=acf.plot(sample3), top="Logistic distribution with thinning = 3", nrow=1)


# Example 3: Sampling from the Truncated Kotz-type distribution
sample4 = rtelliptical(1500, mean, Sigma, lower, upper, dist=NULL, expr=NULL,
                       gFun=function(t){ t^(-1/2)*exp(-2*t^(1/4)) })
f1 = ggplot(data.frame(sample4), aes(x=X1,y=X2)) + geom_point(size=0.50) +
     labs(x=expression(X[1]), y=expression(X[2]), subtitle="Kotz(2,1/4,1/2)")
ggMarginal(f1, type="histogram", fill="grey")
}
\references{
{
  \insertRef{fang2018symmetric}{relliptical}

  \insertRef{ho2012some}{relliptical}

  \insertRef{neal2003slice}{relliptical}
}
}
\seealso{
\code{\link{mvtelliptical}}
}
\author{
Katherine L. Valeriano, Christian E. Galarza and Larissa A. Matos
}
