#' Repo: The Data-centered Data Flow Manager
#'
#' @details
#' The Repo package is meant to help with the management of R data
#' files. It builds one (or more) centralized repository where R
#' objects are stored together with corresponding annotations, tags,
#' dependency notes, provenance traces. It also provides navigation
#' tools to easily locate and load previously stored resources.
#' 
#' Create a new repository with \code{rp <- repo_open()}.
#'
#' Given the object rp of class \code{repo}, the \code{repo} function
#' \code{foo} can be called in two ways: 1) \code{repo_foo(rp)} or 2)
#' \code{rp$foo()}.
#'
#' For a complete list of functions, use \code{library(help = "repo")}.
#' 
#' @docType package
#' @name repo-package
#' @author Francesco Napolitano \email{franapoli@@gmail.com}
#' @aliases repo
NULL

#' Open an existing repository or create a new one.
#'
#' If a repository does not exist at the specified location, creates a
#' directory and stores the repository index in it. If a repository
#' exists, the index is loaded and a \code{repo} object is built.
#' 
#' @param root Path to store data in. Defaults to "~/.R_repo".
#' @param force Don't ask for confirmation.
#' @return An object of class \code{repo}.
#' @name repo_open
#' @examples
#' ## Creates a new repository in a temporary directory without asking for
#' ## confirmation.
#' rp_path <- file.path(tempdir(), "example_repo")
#' rp <- repo_open(rp_path, TRUE)
#' rp$put(0, "zero", "a random item", "a_tag")
#' rp$info()
#' ## wiping temporary repo
#' unlink(rp_path, TRUE)
NULL


#' Build and/or plots a dependency graph
#'
#' Creates a weighted adjacency matrix, in which \code{(i,j) = x} means that
#' item \code{i} is in relation \code{x} with item \code{j}. The resulting graph is
#' plotted.
#'
#' @details The relation between any two items \code{i} and \code{j} can have
#'     values 1, 2 or 3, respectively meaning:
#' \itemize{
#'   \item{depends on: }{to build item \code{i}, item \code{j} was necessary.}
#'   \item{attached to: }{item \code{i} is an attachment item and is attached to
#'         item \code{j}.}
#'   \item{generated by: }{item \code{i} has been generated by item \code{j}. Item
#'         \code{j} is usually an attachment containing the source code that
#'         generated item \code{i}.}
#' }
#'
#' @param rp An object of class repo.
#' @param depends If TRUE, show "depends on" edges.
#' @param attached If TRUE, show "attached to" edges.
#' @param generated If TRUE, show "generated by" edges.
#' @param plot If TRUE (default), plot the dependency graph.
#' @return Adjacency matrix representing the graph, with edges labeled
#' 1, 2, 3 corresponding to "depends", "attached" and "generated" respectively.
#' @examples
#' ## Repository creation
#' rp_path <- file.path(tempdir(), "example_repo")
#' rp <- repo_open(rp_path, TRUE)
#' 
#' ## Producing some irrelevant data
#' data1 <- 1:10
#' data2 <- data1 * 2
#' data3 <- data1 + data2
#'
#' ## Putting the data in the database, specifying dependencies
#' rp$put(data1, "item1", "First item",
#'     "repo_dependencies")
#' rp$put(data2, "item2", "Item dependent on item1", 
#'     "repo_dependencies", depends="item1")
#' rp$put(data3, "item3", "Item dependent on item1 and item2",
#'     "repo_dependencies", depends=c("item1", "item2"))
#'
#' ## Creating a temporary plot and attaching it
#' fpath <- file.path(rp$root(), "temp.pdf")
#' pdf(fpath)
#' plot(data1)
#' dev.off()
#' rp$attach(fpath, "visualization of item1", "plot",
#'    to="item1")
#'
#' ## Obtaining the dependency matrix
#' depmat <- rp$dependencies(plot=FALSE)
#' print(depmat)
#' ## The matrix can be plotted as a graph (requires igraph package)
#' rp$dependencies()
#' ## The following hides "generated" edges
#' rp$dependencies(generated=FALSE)
#' 
#' ## wiping temporary repo
#' unlink(rp_path, TRUE)

repo_dependencies <- function(rp, depends=T, attached=T, generated=T, plot=T)
    rp$dependencies(depends, attached, generated, plot)

#' Check repository integrity.
#'
#' Checks that all indexed data are present in the repository root,
#' that files are not corrupt and that no unindexed files are present.
#'
#' @details Every time the object associated to an item is stored, an
#'     MD5 checksum is saved to the repository index. \code{check}
#'     will use those to verify that the object was not changed by
#'     anything other than Repo itself.
#' @param rp An object of class repo.
#' @return Used for side effects.
#' @examples
#' ## Repository creation
#' rp_path <- file.path(tempdir(), "example_repo")
#' rp <- repo_open(rp_path, TRUE)
#'
#' rp_path <- file.path(tempdir(), "example_repo")
#' rp <- repo_open(rp_path, TRUE)
#' rp$put(0, "item1", "A sample item", "repo_check")
#' rp$check()
#' 
#' ## wiping temporary repo
#' unlink(rp_path, TRUE)
repo_check <- function(rp) rp$check()

#' Copy items to another repository
#'
#' Copies an object file from one repository to another and creates a new entry
#' in the index of the destination repository. Supports tags and multiple names.
#' 
#' @param rp An object of class repo (will copy from it)
#' @param destrepo An object of class repo (will copy to it)
#' @param name The name (or list of names) of the item/s to copy
#' @param tags If not NULL, copy all items matching tags.
#' @return Used for side effects.
#' @examples
#' ## Repository creation
#' rp_path1 <- file.path(tempdir(), "example_repo1")
#'
#' rp1 <- repo_open(rp_path1, TRUE)
#' rp1$put(0, "item1", "A sample item", "tag1")
#' rp_path2 <- file.path(tempdir(), "example_repo2")
#' rp2 <- repo_open(rp_path2, TRUE)
#' rp1$copy(rp2, "item1")
#'
#' ## wiping temporary repo
#' unlink(rp_path1, TRUE)
#' unlink(rp_path2, TRUE)

repo_copy <- function(rp, destrepo, name, tags=NULL)
    rp$copy(destrepo, name, tags)


#' Provides simplified access to repository items.
#'
#' Creates a list of functions, each one associated with a repository
#' item, that can be used to access items directly.
#'
#' @details Repository handlers are functions associated with
#'     items. As opposed to item names, they can take advantage of IDE
#'     auto-completion features and do not require quotation marks. A
#'     handler to the \code{repo} object itself is provided in the
#'     list.
#' @param rp An object of class repo.
#' @return A list of functions.
#' @examples
#'
#' ## Repository creation
#' rp_path <- file.path(tempdir(), "example_repo")
#' rp <- repo_open(rp_path, TRUE)
#' 
#' ## Putting some irrelevant data
#' rp$put(1, "item1", "Sample item 1", "repo_handlers")
#' rp$put(2, "item2", "Sample item 2", "repo_handlers")
#'
#' ## Getting item handlers
#' h <- rp$handlers()
#' ## handlers have the same names as the items in the repo (and they include
#' ## an handler to the repo itself).
#' names(h)
#'
#' ## Without arguments, function "item1" loads item named "item1".
#' i1 <- h$item1()
#'
#' ## Arguments can be used to call other repo functions on the item.
#' h$item1("info")
#'
#' ## After putting new data, the handlers must be refreshed.
#' rp$put(3, "item3", "Sample item 3", "repo_handlers")
#' h <- rp$handlers()
#' names(h)
#'
#' ## wiping temporary repo
#' unlink(rp_path, TRUE)
repo_handlers <- function(rp)rp$handlers()

#' List all tags
#'
#' Shows list of all unique tags associated with any item in the
#' repository.
#' 
#' @param rp An object of class repo.
#' @return Character vector of unique tags defined in the repo.
#' @seealso repo_put
#' @examples
#' rp_path <- file.path(tempdir(), "example_repo")
#' rp <- repo_open(rp_path, TRUE)
#'
#' ## Putting two items with a few tags
#' rp$put(1, "item1", "Sample item 1",
#'     c("repo_tags", "tag1"))
#' rp$put(2, "item2", "Sample item 2",
#'     c("repo_tags", "tag2"))
#'
#' ## Looking up tags
#' rp$tags()
#'
#' ## wiping temporary repo
#' unlink(rp_path, TRUE)
repo_tags <- function(rp) rp$tags()

#' Run system call on an item
#'
#' Runs a system command passing as parameter the file name containing
#' the object associated with an item.
#' 
#' @param rp An object of class repo.
#' @param name Name of a repo item. The path to the file that contains
#' the item will be passed to the system program.
#' @param command System command
#' @return Used for side effects.
#' @examples
#' ## Repository creation
#' rp_path <- file.path(tempdir(), "example_repo")
#' rp <- repo_open(rp_path, TRUE)
#'
#' ## Creating a PDF file with a figure.
#' pdffile <- file.path(rp_path, "afigure.pdf")
#' pdf(pdffile)
#' plot(runif(30), runif(30))
#' dev.off()
#'
#' ## Attaching the PDF file to the repo
#' rp$attach(pdffile, "A plot of random numbers", "repo_sys")
#' ## don't need the original PDF file anymore
#' file.remove(pdffile)
#'
#' ## Opening the stored PDF with Evince document viewer
#' \dontrun{
#' rp$sys("afigure.pdf", "evince")
#'}
#'
#' ## wiping temporary repo
#' unlink(rp_path, TRUE)
repo_sys <- function(rp, name, command)
    rp$sys(name, command)

#' Match items by matching any field
#'
#' @param rp An object of class repo.
#' @param find Character to match any filed (see Details).
#' @param all Show also items tagged with "hide".
#' @param show Select columns to show.
#' @return Used for side effects.
#' @details
#'
#' This function actually calls print specifying the find
#' parameters. The find parameter can be any character string to be
#' matched against any item field, including string-converted size
#' (like "10x3").
#' 
#' @examples
#' rp_path <- file.path(tempdir(), "example_repo")
#' 
#' rp <- repo_open(rp_path, TRUE)
#' rp$put(1, "item1", "Sample item 1", c("tag1", "tag2"))
#' rp$put(2, "item2", "Sample item 2", c("tag1", "hide"))
#' rp$put(3, "item3", "Sample item 3", c("tag2", "tag3"))
#' rp$print()
#' rp$find("tEm2")
#' rp$find("ag2", show="t")
#' 
#' ## wiping the temp repo
#' unlink(rp_path, TRUE)
#'
repo_find <- function(rp, find=NULL, all=F, show="ds")
    rp$find(find=find, all=all, show=show)

#' Show a summary of the repository contents.
#'
#' @param x An object of class repo.
#' @param tags A list of character tags. Only items matching all the
#' tags will be shown.
#' @param tagfun How to combine tags (see Details).
#' @param find Character to match any filed (see Details).
#' @param all Show also items tagged with "hide".
#' @param show Select columns to show.
#' @param ... Further arguments passed to or from other methods
#' @return Used for side effects.
#' @details The \code{tagfun} param specifies how to combine multiple
#' tags when matching items. It can be either a character or a
#' function. As a character, it can be one of \code{OR}, \code{AND} or
#' \code{NOT} to specify that one, all or none of the tags must be
#' matched, respectively. If it is a function, it must take two tag
#' vectors, the first of which corresponds to \code{tags}, and return
#' TRUE for a match, FALSE otherwise.
#'
#' The find param can be any character string to be matched against
#' any item field, including string-converted size (like "10x3").
#' 
#' @examples
#' rp_path <- file.path(tempdir(), "example_repo")
#' rp <- repo_open(rp_path, TRUE)
#' rp$put(1, "item1", "Sample item 1", c("tag1", "tag2"))
#' rp$put(2, "item2", "Sample item 2", c("tag1", "hide"))
#' rp$put(3, "item3", "Sample item 3", c("tag2", "tag3"))
#' rp$print()
#' rp$print(all=TRUE)
#' rp$print(show="tds", all=TRUE)
#' rp$print(show="tds", all=TRUE, tags="tag1")
#' ## wiping the temp repo
#' unlink(rp_path, TRUE)
#'
#' ## wiping temporary repo
#' unlink(rp_path, TRUE)
print.repo <- function(x, tags=NULL, tagfun="OR", find=NULL, all=F, show="ds", ...)
    x$print(tags=tags, tagfun=tagfun, find=find, all=all, show=show)

#' Export \code{repo} items to RDS file.
#' 
#' @param rp An object of class repo.
#' @param name Name (or list of names) of the item/s to export.
#' @param where Destination directory
#' @param tags List of tags: all items tagged with all the tags in the
#' list will be exported.
#' @return TRUE on success, FALSE otherwise.
#' @examples
#' rp_path <- file.path(tempdir(), "example_repo")
#' rp <- repo_open(rp_path, TRUE)
#' rp$put(1, "item1", "Sample item 1", "export")
#' rp$export("item1", tempdir()) # creates item1.RDS in a tempdir
#'
#' ## wiping temporary repo
#' unlink(rp_path, TRUE)
repo_export <- function(rp, name, where=".", tags=NULL)
    rp$export(name, where, tags)

#' Provides detailed information about an item.
#' 
#' @param rp An object of class repo.
#' @param name Item name (or list of names). If both name and tags are NULL, information
#' about the whole repo will be provided.
#' @param tags List of tags: info will run on all items matching the tag list.
#' @return Used for side effects.
#' @examples
#' rp_path <- file.path(tempdir(), "example_repo")
#' rp <- repo_open(rp_path, TRUE)
#' rp$put(1, "item1", "Sample item 1", "info")
#' rp$info("item1")
#'
#' ## wiping temporary repo
#' unlink(rp_path, TRUE)
repo_info <- function(rp, name = NULL, tags = NULL)
    rp$info(name, tags)

#' Remove item from the repo (and the disk).
#' 
#' @param rp An object of class repo.
#' @param name An item's name.
#' @param tags A list of tags: all items matching the list will be
#' removed.
#' @param force Don't ask for confirmation.
#' @return Used for side effects.
#' @examples
#' rp_path <- file.path(tempdir(), "example_repo")
#' rp <- repo_open(rp_path, TRUE)
#' rp$put(1, "item1", "Sample item 1", "info")
#' rp$put(2, "item2", "Sample item 2", "info")
#' print(rp)
#' rp$rm("item1")
#' print(rp)
#'
#' ## wiping temporary repo
#' unlink(rp_path, TRUE)
repo_rm <- function(rp, name = NULL, tags = NULL, force = F)
    rp$rm(name, tags, force)

#' Retrieve an item from the repo.
#' 
#' @param rp An object of class repo.
#' @param name An item's name.
#' @return The previously stored object.
#' @examples
#' rp_path <- file.path(tempdir(), "example_repo")
#' rp <- repo_open(rp_path, TRUE)
#' rp$put(1, "item1", "Sample item 1", "get")
#' print(rp$get("item1"))
#'
#' ## wiping temporary repo
#' unlink(rp_path, TRUE)
repo_get <- function(rp, name)rp$get(name)

#' Low-level list of item entries.
#' 
#' @param rp An object of class repo.
#' @return A detailed list of item entries.
#' @examples
#' rp_path <- file.path(tempdir(), "example_repo")
#' rp <- repo_open(rp_path, TRUE)
#' rp$put(1, "item1", "Sample item 1", "entries")
#' rp$put(2, "item2", "Sample item 2", "entries")
#' rp$put(3, "item3", "Sample item 3", "entries")
#' print(rp$entries())
#'
#' ## wiping temporary repo
#' unlink(rp_path, TRUE)
repo_entries <- function(rp)rp$entries()

#' Add tags to an item.
#' 
#' @param rp An object of class repo.
#' @param name An item name.
#' @param newtags A list of tags that will be added to the item's tag
#' list.
#' @param tags A list of tags: newtags will be added to all items
#' matching the list.
#' @return Used for side effects.
#' @seealso repo_untag, repo_set
#' @examples
#' rp_path <- file.path(tempdir(), "example_repo")
#' rp <- repo_open(rp_path, TRUE)
#' rp$put(1, "item1", "Sample item 1", "tag1")
#' rp$print(show="t")
#' rp$tag("item1", "tag2")
#' rp$print(show="t")
#'
#' ## wiping temporary repo
#' unlink(rp_path, TRUE)
repo_tag <- function(rp, name = NULL, newtags, tags = NULL)
    rp$tag(name, newtags, tags)


#' Run expression with cache.
#'
#' lazydo searches the repo for previous execution of an
#' expression. If a previous execution is found, the result is loaded
#' and returned. Otherwise, the expression is executed and the result
#' stashed.
#'
#' @param rp An object of class repo.
#' @param expr An object of class exrepssion (the code to run).
#' @param force If TRUE, execute expr anyway
#' @param env Environment for expr, defaults to parent.
#' @return Results of the expression (either loaded or computed on the
#'     fly).
#' @details The expression results are stashed as usual. The name of
#' the resource is obtained by digesting the expression, so it will
#' look like an MD5 string in the repo. Note that the expression, and
#' not its result, will uniquely identify the item in the repo.
#'
#' The new item is automatically tagged with "stash", "hide" and
#' "lazydo".
#' @seealso repo_stash, repo_put
#' 
#' @examples
#' rp_path <- file.path(tempdir(), "example_repo")
#' rp <- repo_open(rp_path, TRUE)
#' expr <- expression(
#'     {
#'         v <- vector("numeric", 10)
#'         for(i in 1:10) {
#'             v[i] <- i
#'             Sys.sleep(1/10)
#'         }
#'         print("Done.")
#'         v
#'     }
#' )
#' 
#' system.time(v <- rp$lazydo(expr)) # first run
#' ## Repo needs to build resource.
#' ## [1] "Done."
#' ##    user  system elapsed 
#' ##   0.006   0.000   1.007
#'
#' system.time(v <- rp$lazydo(expr)) # second run
#' ## Repo found precomputed resource.
#' ##    user  system elapsed 
#' ##   0.000   0.004   0.001
#'
#' ## The item's name in the repo can be obtained using digest:
#' library(digest)
#' resname <- digest(expr)
#' ## Or as the name of the last item added:
#' resname <- tail(rp$entries(),1)[[1]]$name
#' 
#' rp$rm(resname) ## single cached item cleared
#'
#' ## wiping temporary repo
#' unlink(rp_path, TRUE)
repo_lazydo = function(rp, expr, force=F, env=parent.frame())
    rp$lazydo(expr, force, env)

#' Remove tags from an item.
#' 
#' @param rp An object of class repo.
#' @param name An item name.
#' @param rmtags A list of tags that will be removed from the item's
#' tag list.
#' @param tags A list of tags: rmtags will be removed from all items
#' matching the list.
#' @return Used for side effects.
#' @seealso repo_tag, repo_set
#' @examples
#' rp_path <- file.path(tempdir(), "example_repo")
#' rp <- repo_open(rp_path, TRUE)
#' rp$put(1, "item1", "Sample item 1", c("tag1", "tag2"))
#' rp$print(show="t")
#' rp$untag("item1", "tag2")
#' rp$print(show="t")
#' 
#' ## wiping temporary repo
#' unlink(rp_path, TRUE)
repo_untag <- function(rp, name = NULL, rmtags, tags = NULL)
    rp$untag(name, rmtags, tags)

#' Edit all items info using a text file.
#' 
#' @details Exactly one of \code{outfile} or \code{infile} must be
#'     supplied. All repository entry fields are copied to a
#'     tab-separated file when using the \code{outfile} parameter. All
#'     repo entries are updated reading from \code{infile} when the
#'     \code{infile} parameter is used. Within the TAGS field, tags
#'     must be comma-separated. The system writes a checksum to the
#'     \code{outfile} that prevents from using it as \code{infile} if
#'     repo has changed in the meantime.
#' @param rp An object of class repo.
#' @param outfile Name of a file to put entries data to.
#' @param infile Name of a file to read entries data from.
#' @return Used for side effects.
#' @seealso repo_set
#' @examples
#' rp_path <- file.path(tempdir(), "example_repo")
#' rp <- repo_open(rp_path, TRUE)
#' rp$put(1, "item1", "Sample item 1", c("tag1", "tag2"))
#' 
#' items_data_file <- tempfile()
#' rp$bulkedit(items_data_file)
#' ## Manually edit items_data_file, then update items:
#' rp$bulkedit(infile=items_data_file)
#' 
#' ## wiping temporary repo
#' unlink(rp_path, TRUE)
repo_bulkedit <- function(rp, outfile=NULL, infile=NULL)
    rp$bulkedit(outfile, infile)


#' Edit an existing item.
#' 
#' @param rp An object of class repo.
#' @param name An item name.
#' @param obj An R object to replace the one currently associated with the item.
#' @param newname Newname of the item.
#' @param description Item's description.
#' @param tags New item's tags as a ist of character.
#' @param src New item's provenance as a list of character.
#' @param depends List of item names indicating dependancies.
#' @param addtags Tags to be added to current item's tags. Can not be
#' used together with the parameter "tags".
#' @return Used for side effects.
#' @seealso repo_put
#' @examples
#' rp_path <- file.path(tempdir(), "example_repo")
#' rp <- repo_open(rp_path, TRUE)
#' rp$put(1, "item1", "Sample item 1", c("tag1", "tag2"))
#' rp$set("item1", obj=2)
#' print(rp$get("item1"))
#' rp$set("item1", description="Modified description", tags="new_tag_set")
#' rp$info("item1")
#'
#' ## wiping temporary repo
#' unlink(rp_path, TRUE)
repo_set <- function(rp, name, obj=NULL, newname=NULL, description=NULL,
                     tags=NULL, src=NULL, depends=NULL, addtags=NULL)
    rp$set(name, obj, newname, description, tags, src, depends, addtags)

#' Create a new item from an existing file.
#' 
#' @param rp An object of class repo.
#' @param filepath The path to the file to be stored in the repo.
#' @param description A character description of the item.
#' @param tags A list of tags to sort the item. Tags are useful for
#' selecting sets of items and run bulk actions.
#' @param src The item's provenance as a list of character. Usually
#' the name of the script producing the stored object, a website where
#' the object was downloaded, and so on. If one of the provenance
#' strings matches the name of a repo's item, this will create a
#' dependency link.
#' @param replace If the item exists, overwrite the specified fields.
#' @param to An existing item name to attach the file to.
#' @return Used for side effects.
#' @examples
#' rp_path <- file.path(tempdir(), "example_repo")
#' rp <- repo_open(rp_path, TRUE)
#' 
#' \dontrun{
#' ## Creating a PDF file with a figure.
#' pdf("afigure.pdf")
#' ## Drawing a random plot in the figure
#' plot(runif(100), runif(100))
#' dev.off()
#' ## Attaching the PDF file to the repo
#' rp$attach("afigure.pdf", "A plot of random numbers", "repo_sys")
#' ## don't need the PDF file anymore
#' file.remove("afigure.pdf")
#' ## Opening the stored PDF with Evince document viewer
#' rp$sys("afigure.pdf", "evince")
#' }
#'
#' ## wiping temporary repo
#' unlink(rp_path, TRUE)
repo_attach <- function(rp, filepath, description, tags, src=NULL, replace=F, to=NULL)
    rp$attach(filepath, description, tags, src, replace, to)

#' Quickly store temporary data
#'
#' A very simplified call to put that only requires to specify
#' a variable name.
#'
#' @details
#' The name parameter is used to search the parent
#' (or a different specified) environment for the actual object to
#' store. Then it is also used as the item name. The reserved tags
#' "stash" and "hide" are set. In case a stashed item by the same name
#' already exists, it is automatically overwritten. In case a
#' non-stashed item by the same name already exists, an error is
#' raised. A different name can be specified through the rename
#' parameter in such cases.
#' @param rp An object of class repo.
#' @param object The object to store in the repo.
#' @param name An optional character containing the new name for the
#' item. Otherwise the name of object is used as item's name.
#' @return Used for side effects.
#' @seealso repo_put, repo_lazydo
#' @examples
#' rp_path <- file.path(tempdir(), "example_repo")
#' rp <- repo_open(rp_path, TRUE)
#' tempdata <- runif(10)
#' rp$stash(tempdata)
#' rp$info("tempdata")
#'
#' ## wiping temporary repo
#' unlink(rp_path, TRUE)
repo_stash <- function(rp, object, name=deparse(substitute(object)))
    rp$stash(object, name)

#' Remove all stashed data
#'
#' @param rp An object of class repo.
#' @param force If TRUE, no confirmation is asked.
#' @return Used for side effects.
#' @seealso repo_rm, repo_stash
#' @examples
#' rp_path <- file.path(tempdir(), "example_repo")
#' rp <- repo_open(rp_path, TRUE)
#' tempdata <- runif(10)
#' rp$stash("tempdata")
#' rp$print(all=TRUE)
#' rp$stashclear(TRUE)
#'
#' ## wiping temporary repo
#' unlink(rp_path, TRUE)
repo_stashclear <- function(rp, force=F)
    rp$stashclear(force)


#' Create a new item in the repo.
#'
#' Given an R object, stores it to an RDS file in the \code{repo} root
#' and add an associated item to the \code{repo} index, including
#' object name, description, tags and more.
#' 
#' @details The item \code{name} can be any string, however it should
#'     be a concise identifier, possibly without special character
#'     (could become mandatory soon). Some tags have a special
#'     meaning, like "hide" (do not show the item by default),
#'     "attachment" (the item is an attachment - this should never be
#'     set manually), "stash" (the item is a stashed item, makes the
#'     item over-writable by other "stash" items by default).
#' @param rp An object of class repo.
#' @param obj An R object to store in the repo.
#' @param name A character identifier for the new item.
#' @param description A character description of the item.
#' @param tags A list of tags to sort the item. Tags are useful for
#'     selecting sets of items and run bulk actions.
#' @param src Name of an existing item to be recorded as the
#' "generator" of the new item. Usually it is an attachment item
#' containing the source that generated the new item.
#' @param depends List of character: items that depend on this item.
#' @param replace One of: V, F, "addversion". Default is F. If V,
#'     overwrite an existing item by the same name. If F stops with an
#'     error. If "addversion" the new item is stored as a new version
#'     and the old item is renamed by appending a "#N" suffix.
#' @param asattach Specifies that the item is to be trated as an
#'     attachment (see attach).
#' @param to Optionally specifies which item this item is attached to.
#' @return Used for side effects.
#' @seealso get, set, attach, info
#' @examples
#' ## Repository creation
#' rp_path <- file.path(tempdir(), "example_repo")
#' rp <- repo_open(rp_path, TRUE)
#'
#' ## Producing some irrelevant data
#' data1 <- 1:10
#' data2 <- data1 * 2
#' data3 <- data1 / 2
#'
#' ## Putting the data in the database, specifying dependencies
#' rp$put(
#'     obj = data1,
#'     name = "item1",
#'     description = "First item",
#'     tags = c("repo_put", "a_random_tag"),
#'     )
#' rp$put(data2, "item2", "Item dependent on item1",
#'     "repo_dependencies", depends="item1")
#' rp$put(data3, "item3", "Item dependent on item1 and item2",
#'     "repo_dependencies", depends=c("item1", "item2"))
#'
#' print(rp)
#'
#' ## Creating another version of item1
#' data1.2 <- data1 + runif(10)
#' rp$put(data1.2, name = "item1", "First item with additional noise",
#'     tags = c("repo_put", "a_random_tag"), replace="addversion")
#' print(rp, all=TRUE)
#' rp$info("item1#1")
#'
#' ## wiping temporary repo
#' unlink(rp_path, TRUE)
repo_put <- function(rp, obj, name, description, tags, src=NULL,
                     depends = NULL, replace=F, asattach=F, to=NULL)
    rp$put(obj, name, description, tags, src,
             depends, replace, asattach, to)

#' Download item remote content
#'
#' @details Repo index files can be used as pointers to remote
#'     data. The pull function will download the actual data from the
#'     Internet, including regular items or attachment. Another use of
#'     the URL item's parameter is to attach a remote resource without
#'     downloading it.
#' @param rp An object of class repo.
#' @param name Name of the existing item that will be updated.
#' @param replace If TRUE, existing item's object is overwritten.
#' @return Used for side effects.
#' @examples
#' ## Repository creation
#' rp_path <- file.path(tempdir(), "example_repo")
#' rp <- repo_open(rp_path, TRUE)
#'
#' ## The following item will have remote source
#' rp$put("Local content", "item1", "Sample item 1", "tag",
#'          URL="http://www.francesconapolitano.it/repo/remote")
#' print(rp$get("item1"))
#' rp$pull("item1")
#' print(rp$get("item1"))
#'
#' ## wiping temporary repo
#' unlink(rp_path, TRUE)
repo_pull <- function(rp, name, replace=F)
    rp$pull(name, replace)


#' Append text to an existing item content.
#'
#' This feature is experimental.
#' 
#' @param rp An object of class repo.
#' @param id The name of an item whose object is of class character.
#' @param txtorfunc Text to be appended to the item's object. It can
#' also be a on object of class function: in this case, its source is
#' appended.
#' @return Used for side effects.
repo_append <- function(rp, id, txtorfunc)
    rp$append(id, txtorfunc)


#' Show path to repo root
#' 
#' @param rp An object of class repo.
#' @return character cotaining the path to the root of the repo.
#' @examples
#' rp_path <- file.path(tempdir(), "example_repo")
#' rp <- repo_open(rp_path, TRUE)
#' print(rp$root())
#'
#' ## wiping temporary repo
#' unlink(rp_path, TRUE)
repo_root <- function(rp)
    rp$root()

