% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/robustDigest.R
\docType{methods}
\name{.robustDigest}
\alias{.robustDigest}
\alias{.robustDigest,ANY-method}
\alias{.robustDigest,cluster-method}
\alias{.robustDigest,function-method}
\alias{.robustDigest,expression-method}
\alias{.robustDigest,character-method}
\alias{.robustDigest,Path-method}
\alias{.robustDigest,environment-method}
\alias{.robustDigest,list-method}
\alias{.robustDigest,Raster-method}
\alias{.robustDigest,Spatial-method}
\title{Create reproducible digests of objects in R}
\usage{
.robustDigest(object, objects, compareRasterFileLength = 1e+06,
  algo = "xxhash64", digestPathContent = FALSE, classOptions = list())

\S4method{.robustDigest}{ANY}(object, compareRasterFileLength, algo,
  digestPathContent, classOptions)

\S4method{.robustDigest}{cluster}(object, compareRasterFileLength, algo,
  digestPathContent, classOptions)

\S4method{.robustDigest}{`function`}(object, compareRasterFileLength, algo,
  digestPathContent, classOptions)

\S4method{.robustDigest}{expression}(object, compareRasterFileLength, algo,
  digestPathContent, classOptions)

\S4method{.robustDigest}{character}(object, compareRasterFileLength, algo,
  digestPathContent, classOptions)

\S4method{.robustDigest}{Path}(object, compareRasterFileLength, algo,
  digestPathContent, classOptions)

\S4method{.robustDigest}{environment}(object, compareRasterFileLength, algo,
  digestPathContent, classOptions)

\S4method{.robustDigest}{list}(object, compareRasterFileLength, algo,
  digestPathContent, classOptions)

\S4method{.robustDigest}{Raster}(object, compareRasterFileLength, algo,
  digestPathContent, classOptions)

\S4method{.robustDigest}{Spatial}(object, compareRasterFileLength, algo,
  digestPathContent, classOptions)
}
\arguments{
\item{object}{an object to digest.}

\item{objects}{Optional character vector indicating which objects are to
be considered while making digestible. This is only relevant
if the object being passed is an environment or list or the like.}

\item{compareRasterFileLength}{Numeric. Optional. When there are Rasters, that
have file-backed storage, this is passed to the length arg in \code{digest}
when determining if the Raster file is already in the database.
Note: uses \code{\link[digest]{digest}} for file-backed Raster.
Default 1e6. Passed to \code{.prepareFileBackedRaster}.}

\item{algo}{The algorithms to be used; currently available choices are
    \code{md5}, which is also the default, \code{sha1}, \code{crc32},
    \code{sha256}, \code{sha512}, \code{xxhash32}, \code{xxhash64} and
    \code{murmur32}.}

\item{digestPathContent}{Logical. Should arguments that are of class \code{Path}
(see examples below) have their name digested
(\code{FALSE}), or their file contents (\code{TRUE}; default).}

\item{classOptions}{Optional list. This will pass into \code{.robustDigest} for
specific classes. Should be options that the \code{.robustDigest} knows what
to do with.}
}
\value{
A hash i.e., digest of the object passed in.
}
\description{
Not all aspects of R objects are captured by current hashing tools in R (e.g.
\code{digest::digest}, \code{fastdigest::fastdigest}, \code{knitr} caching,
\code{archivist::cache}). This is mostly because many objects have "transient"
(e.g., functions have environments), or "disk-backed" features. This function
allows for these accommodations to be made and uses \code{\link[fastdigest]{fastdigest}}
internally.  Since
the goal of using reproducibility is to have tools that are not session specific,
this function
attempts to strip all session specific information so that the fastdigest
works between sessions and operating systems. It is tested under many
conditions and object types, there are bound to be others that don't
work correctly.
}
\section{Classes}{


\code{Raster*} objects have the potential for disk-backed storage.
If the object in the R session is cached using \code{archivist::cache}, only
the header component will be assessed for caching. Thus, objects like this
require more work. Also, because \code{Raster*} can have a built-in representation
for having their data content located on disk, this format will be maintained if the
raster already is file-backed, i.e., to create \code{.tif} or \code{.grd} backed rasters,
use \code{writeRaster} first, then Cache. The .tif or .grd will be copied to the "raster"
subdirectory of the \code{cacheRepo}.
Their RAM representation (as an R object) will still be in the usual \file{gallery/} directory.
For \code{inMemory} raster objects, they will remain as binary \code{.RData} files.

Functions (which are contained within environments) are
converted to a text representation via a call to \code{format(FUN)}.

Objects contained within a list or environment are recursively hashed
using \code{\link[fastdigest]{fastdigest}}, while removing all references to
environments.

Character strings are first assessed with \code{dir.exists} and \code{file.exists}
to check for paths. If they are found to be paths, then the path is hashed with
only its filename via \code{basename(filename)}. If it is actually a path, we suggest
using \code{asPath(thePath)}
}

\examples{

a <- 2
tmpfile1 <- tempfile()
tmpfile2 <- tempfile()
save(a, file = tmpfile1)
save(a, file = tmpfile2)

# treats as character string, so 2 filenames are different
fastdigest::fastdigest(tmpfile1)
fastdigest::fastdigest(tmpfile2)

# tests to see whether character string is representing a file
.robustDigest(tmpfile1)
.robustDigest(tmpfile2) # same

# if you tell it that it is a path, then you can decide if you want it to be
#  treated as a character string or as a file path
.robustDigest(asPath(tmpfile1))
.robustDigest(asPath(tmpfile2)) # different because you have a choice

.robustDigest(asPath(tmpfile1), digestPathContent = TRUE)
.robustDigest(asPath(tmpfile2), digestPathContent = TRUE) # same

# Rasters are interesting because it is not know a priori if it
#   it has a file name associated with it.
library(raster)
r <- raster(extent(0,10,0,10), vals = 1:100)

# write to disk
r1 <- writeRaster(r, file = tmpfile1)
r2 <- writeRaster(r, file = tmpfile2)

digest::digest(r1)
digest::digest(r2) # different
fastdigest::fastdigest(r1)
fastdigest::fastdigest(r2) # different
.robustDigest(r1)
.robustDigest(r2) # same... data are the same in the file

# note, this is not true for comparing memory and file-backed rasters
.robustDigest(r)
.robustDigest(r1) # different

}
\seealso{
\code{\link[archivist]{cache}}.

\code{\link[fastdigest]{fastdigest}}.
}
\author{
Eliot McIntire
}
\keyword{internal}
