% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/add_min_iic_constraint.R
\name{add_min_iic_constraint}
\alias{add_min_iic_constraint}
\title{Add constraint to enforce a minimum integral index of connectivity (IIC) value}
\usage{
add_min_iic_constraint(
  problem,
  min_iic,
  distance_threshold = -1,
  unit = "m",
  precision = 4
)
}
\arguments{
\item{problem}{\code{\link[=restopt_problem]{restopt_problem()}} Restoration problem object.}

\item{min_iic}{\code{numeric} Minimum IIC value (between 0 and 1).}

\item{distance_threshold}{\code{numeric} greater than 0. Minimum distance (in
\code{unit}) between two patches to consider them connected in the computation of
the IIC. The default value -1 causes the function to use 1 aggregated cell as
the distance threshold.}

\item{unit}{\code{unit} object or a \code{character} that can be coerced to a distance
unit (see \code{unit} package), or "cells" for cell width of aggregated
habitat raster. Units of the \code{distance_threshold} parameter. If the input
habitat raster does not use a projected coordinate system, only "cells" is
available. Meters by default, expected if \code{distance_threshold} is set to its
default value (-1), which causes the function to use 1 cell by default.}

\item{precision}{\code{integer} Precision for calculations.
Defaults to 4.}
}
\value{
An updated restoration problem (\code{\link[=restopt_problem]{restopt_problem()}}) object.
}
\description{
Add constraint to a restoration problem (\code{\link[=restopt_problem]{restopt_problem()}}) object
to specify the minimum integral index of connectivity of a solution.
}
\details{
The integral index of connectivity (IIC) is a graph-based inter-patch
connectivity index based on a binary connection model (Pascual-Hortal &
Saura, 2006). Its maximization in the context of restoration favours
restoring the structural connectivity between large patches. IIC is unitless
and comprised between 0 (no connectivity) and 1 (all the landscape is
habitat, thus fully connected). The \code{distance_threshold} parameter indicates
to the solver how to construct the habitat graph, i.e. what is the minimum
distance between two patches to consider them as connected. Note that, as
the computation occurs on aggregated cells, if \code{distance_threshold} is used
with a different unit than "cells", it will be rounded to the closest
corresponding number of cells.

The effective mesh size (MESH) is a measure of landscape fragmentation
based on the probability that two randomly chosen points are located in the
same patch (Jaeger, 2000). Maximizing it in the context of restoration
favours fewer and larger patches.
}
\examples{
\donttest{
# load data
habitat_data <- rast(
  system.file("extdata", "habitat_hi_res.tif", package = "restoptr")
)

locked_out_data <- rast(
 system.file("extdata", "locked_out.tif", package = "restoptr")
)

# create problem with locked out constraints
p <- restopt_problem(
    existing_habitat = habitat_data,
    aggregation_factor = 16,
    habitat_threshold = 0.7
  ) \%>\%
  add_min_iic_constraint(0.2) \%>\%
  add_restorable_constraint(
    min_restore = 200,
    max_restore = 300,
  ) \%>\%
  add_locked_out_constraint(data = locked_out_data) \%>\%
  add_compactness_constraint(2500, unit = "m") \%>\%
  add_settings(time_limit = 1)

# print problem
print(p)

# solve problem
s <- solve(p)

# plot solution
plot(s)
}

}
\references{
Pascual-Hortal, L., & Saura, S. (2006).
Comparison and development of new graph-based landscape connectivity indices:
Towards the priorization of habitat patches and corridors for conservation.
Landscape Ecology, 21(7), 959‑967. https://doi.org/10.1007/s10980-006-0013-z
}
\seealso{
\link{set_max_iic_objective}

Other constraints: 
\code{\link{add_available_areas_constraint}()},
\code{\link{add_compactness_constraint}()},
\code{\link{add_components_constraint}()},
\code{\link{add_connected_constraint}()},
\code{\link{add_locked_out_constraint}()},
\code{\link{add_min_mesh_constraint}()},
\code{\link{add_restorable_constraint}()}
}
\concept{constraints}
