% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/identityCoefs.R
\name{identityCoefs}
\alias{identityCoefs}
\alias{detailed2condensed}
\title{Omnibus function for identity coefficients}
\usage{
identityCoefs(
  x,
  ids = labels(x),
  detailed = FALSE,
  Xchrom = FALSE,
  self = FALSE,
  simplify = TRUE,
  method = c("auto", "K", "WL", "LS", "GC", "idcoefs", "identity", "merlin"),
  verbose = FALSE,
  ...
)

detailed2condensed(d)
}
\arguments{
\item{x}{A pedigree in the form of a \code{\link[pedtools:ped]{pedtools::ped}} object.}

\item{ids}{A vector of two ID labels.}

\item{detailed}{A logical. If FALSE (default), the 9 condensed coefficients
are computed; otherwise the 15 detailed identity coefficients.}

\item{Xchrom}{A logical, by default FALSE.}

\item{self}{A logical indicating if self-relationships (e.g., between a
pedigree member and itself) should be included. FALSE by default.}

\item{simplify}{Simplify the output (to a numeric of length 9) if \code{ids} has
length 2. Default: TRUE.}

\item{method}{Either "auto", "K", "WL", "LS", "GC", "idcoefs", "identity" or
"merlin". By default ("auto") a suitable algorithm is chosen automatically.}

\item{verbose}{A logical.}

\item{...}{Further arguments.}

\item{d}{Either a numeric vector of length 15, or a data frame with 17
columns.}
}
\value{
A data frame with L + 2 columns, where L is either 9 or 15 (if
\code{detailed = TRUE}).

If \code{simplify = TRUE} and \code{length(ids) = 2}: A numeric vector of length \code{L}.
}
\description{
This function calculates the pairwise identity coefficients described by
Jacquard (1974). Unlike the previous \code{\link[=condensedIdentity]{condensedIdentity()}} (which will
continue to exist), this function also computes the 15 \emph{detailed} identity
coefficients. The implementation supports pedigrees with inbred founders, and
X-chromosomal coefficients.
}
\details{
Both the condensed and detailed coefficients are given in the orders used by
Jacquard (1974). The function \code{detailed2condensed()} converts from detailed
coefficients (d1, ... d15) to condensed ones (D1, ..., D9) using the
following relations:
\itemize{
\item D1 = d1
\item D2 = d6
\item D3 = d2 + d3
\item D4 = d7
\item D5 = d4 + d5
\item D6 = d8
\item D7 = d9 + d12
\item D8 = d10 + d11 + d13 + d14
\item D9 = d15
}
\subsection{Algorithms for computing identity coefficients}{

The following is a brief overview of various algorithms for computing
(single-locus) condensed and/or detailed identity coefficients. This topic is
closely linked to that of \emph{generalised kinship coefficients}, which is
further described in the documentation of \code{\link[=gKinship]{gKinship()}}.

For each algorithm below, it is indicated in brackets how to enforce it in
\code{identityCoefs()}.
\itemize{
\item Karigl (1981) gave the first recursive algorithm for the 9 condensed
identity coefficients. [\code{method = "K"}]
\item Weeks & Lange (1988) suggested a broader and more natural generalisation of
kinship coefficients, leading to a slightly different algorithm for condensed
coefficients. [\code{method = "WL"}]
\item Lange & Sinsheimer (1992) described an even further generalisation of
kinship coefficients, allowing a mix of deterministic and random sampling of
alleles. They used this to give (i) an alternative algorithm for the 9
condensed identity coefficients, and (ii) an algorithm for the 15 detailed
coefficients. [\code{method = "LS"}]
\item The C program \code{IdCoefs} (version 2.1.1) by Mark Abney (2009) uses a graph
model to obtain very fast computation of condensed identity coefficients.
This requires \code{IdCoefs} to be installed on the computer (see link under
References) and available on the system search path. The function then writes
the necessary files to disk and calls \code{IdCoefs} via \code{\link[=system]{system()}}. [\code{method = "idcoefs"}]
\item The R package \code{identity} provides an R interface for \code{IdCoefs}, avoiding
calls to \code{system()}. [\code{method = "identity"}]
\item The MERLIN software (Abecasis et al, 2002) offers an option "--extended"
for computing detailed identity coefficients. This option requires MERLIN to
be installed on the system. The function then writes the necessary files to
disk and calls MERLIN via \code{\link[=system]{system()}}. If \code{detailed = FALSE}, the coefficients
are transformed with \code{detailed2condensed()} before returning. Note: MERLIN
rounds all numbers to 3 decimal places. Since this rounding is done on the
detailed coefficients, rounding errors may happen when converting to the
condensed ones. [\code{method = "merlin"}]
}
}
}
\examples{
x = fullSibMating(1)

### Condensed coefficients
j1 = identityCoefs(x, method = "K")
j2 = identityCoefs(x, method = "WL")
j3 = identityCoefs(x, method = "LS")
j4 = identityCoefs(x, method = "GC")
j5 = condensedIdentity(x, ids = 1:6) # legacy version

stopifnot(all.equal(j1,j2), all.equal(j1,j3), all.equal(j1,j4), all.equal(j1,j5))

### Detailed coefficients
jdet1 = identityCoefs(x, detailed = TRUE, method = "LS")
jdet2 = identityCoefs(x, detailed = TRUE, method = "GC")

stopifnot(all.equal(jdet1,jdet2))

### X-chromosomal coefficients
jx1 = identityCoefs(x, Xchrom = TRUE, method = "K")
jx2 = identityCoefs(x, Xchrom = TRUE, method = "GC")
jx3 = condensedIdentityX(x, ids = 1:6)  # legacy version

stopifnot(all.equal(jx1,jx2), all.equal(jx1,jx3))

### Detailed X-chromosomal coefficients
jdx = identityCoefs(x, detailed = TRUE, Xchrom = TRUE, method = "GC")

stopifnot(all.equal(detailed2condensed(jdx), jx1))

}
\references{
\itemize{
\item Jacquard, A. (1974). The Genetic Structure of Populations. Springer.
\item Karigl, G. (1981). A recursive algorithm for the calculation of identity
coefficients. Ann. Hum. Genet.
\item Weeks, D.E. & Lange, K. (1988). The affected-pedigree-member method of
linkage analysis. Am. J. Hum. Genet
\item Lange, K. & Sinsheimer, J.s. (1992). Calculation of genetic identity
coefficients. Ann. Hum. Genet.
\item Abney, Mark (2009). A graphical algorithm for fast computation of identity
coefficients and generalized kinship coefficients. Bioinformatics, 25,
1561-1563. \url{https://home.uchicago.edu/~abney/abney_web/Software.html}
}
}
\seealso{
\code{\link[=condensedIdentity]{condensedIdentity()}}, \code{\link[=gKinship]{gKinship()}}
}
