% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gqPen.R
\name{rq.gq.pen}
\alias{rq.gq.pen}
\title{Title Quantile regression estimation and consistent variable selection across multiple quantiles}
\usage{
rq.gq.pen(
  x,
  y,
  tau,
  lambda = NULL,
  nlambda = 100,
  eps = ifelse(nrow(x) < ncol(x), 0.01, 0.001),
  weights = NULL,
  penalty.factor = NULL,
  scalex = TRUE,
  tau.penalty.factor = NULL,
  gmma = 0.2,
  max.iter = 200,
  lambda.discard = TRUE,
  converge.eps = 1e-04,
  beta0 = NULL
)
}
\arguments{
\item{x}{covariate matrix}

\item{y}{a univariate response variable}

\item{tau}{a sequence of quantiles to be modeled, must be of at least length 3.}

\item{lambda}{shrinkage parameter. Default is NULL, and the algorithm provides a solution path.}

\item{nlambda}{Number of lambda values to be considered.}

\item{eps}{If not pre-specified the lambda vector will be from lambda_max to lambda_max times eps}

\item{weights}{observation weights. Default is NULL, which means equal weights.}

\item{penalty.factor}{weights for the shrinkage parameter for each covariate. Default is equal weight.}

\item{scalex}{Whether x should be scaled before fitting the model. Coefficients are returned on the original scale.}

\item{tau.penalty.factor}{weights for different quantiles. Default is equal weight.}

\item{gmma}{tuning parameter for the Huber loss}

\item{max.iter}{maximum number of iteration. Default is 200.}

\item{lambda.discard}{Default is TRUE, meaning that the solution path stops if the relative deviance changes sufficiently small. It usually happens near the end of solution path. However, the program returns at least 70 models along the solution path.}

\item{converge.eps}{The epsilon level convergence. Default is 1e-4.}

\item{beta0}{Initial estimates. Default is NULL, and the algorithm starts with the intercepts being the quantiles of response variable and other coefficients being zeros.}
}
\value{
An rq.pen.seq object. 
\describe{
\item{models: }{ A list of each model fit for each tau and a combination.}
\item{n:}{ Sample size.}
\item{p:}{ Number of predictors.}
\item{alg:}{ Algorithm used. Options are "huber" or any method implemented in rq(), such as "br". }
\item{tau:}{ Quantiles modeled.}
\item{a:}{ Tuning parameters a used.}
\item{modelsInfo:}{ Information about the quantile and a value for each model.}
\item{lambda:}{ Lambda values used for all models. If a model has fewer coefficients than lambda, say k. Then it used the first k values of lambda. Setting lambda.discard to TRUE will gurantee all values use the same lambdas, but may increase computational time noticeably and for little gain.}
\item{penalty:}{ Penalty used.}
\item{call:}{ Original call.}
}
Each model in the models list has the following values. 
\describe{
\item{coefficients:}{ Coefficients for each value of lambda.}
\item{rho:}{ The unpenalized objective function for each value of lambda.}
\item{PenRho:}{ The penalized objective function for each value of lambda.}
\item{nzero:}{ The number of nonzero coefficients for each value of lambda.}
\item{tau:}{ Quantile of the model.}
\item{a:}{ Value of a for the penalized loss function.}
}
}
\description{
Uses the group lasso penalty across the quantiles to provide consistent selection across all, K, modeled quantiles. Let \eqn{\beta^q}
be the coefficients for the kth quantiles, \eqn{\beta_j} be the Q-dimensional vector of the jth coefficient for each quantile, and
\eqn{\rho_\tau(u)} is the quantile loss function. The method minimizes
\deqn{\sum_{q=1}^Q \frac{1}{n} \sum_{i=1}^n \rho_\tau(y_i-x_i^\top\beta^q) + \lambda \sum_{j=1}^p ||\beta_j||_{2,w}  .}
Uses a Huber approximation in the fitting of model, as presented in Sherwood and Li (2022). Where,
\deqn{||\beta_j||_{2,w} = \sqrt{\sum_{k=1}^K w_km_j\beta_{kj}^2}}, where \eqn{w_k} is a quantile weight 
that can be specified by \code{tau.penalty.factor} and \eqn{m_j} is a predictor weight that can be assigned by \code{penalty.factor}.
}
\examples{
\dontrun{ 
n<- 200
p<- 10
X<- matrix(rnorm(n*p),n,p)
y<- -2+X[,1]+0.5*X[,2]-X[,3]-0.5*X[,7]+X[,8]-0.2*X[,9]+rt(n,2)
taus <- seq(0.1, 0.9, 0.2)
fit<- rq.gq.pen(X, y, taus)
#use IC to select best model, see rq.gq.pen.cv() for a cross-validation approach
qfit <- qic.select(fit)
}
}
\references{
\insertRef{heteroIdQR}{rqPen}

\insertRef{huberGroup}{rqPen}
}
\author{
Shaobo Li \email{shaobo.li@ku.edu} and Ben Sherwood, \email{ben.sherwood@ku.edu}
}
